/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    user_algorithm_interface.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-07-01
 * @section DESCRIPTION
 *			Interface to be passed as a parameter for the user algorithm
 */

#ifndef USER_ALGORITHM_INTERFACE_H
#define USER_ALGORITHM_INTERFACE_H

#include "utils.h"

#ifdef RECOSIM_PART
	#include <systemc.h>
#endif

#ifdef MIDDLEWARE_GEN
	#include "rs_event.h"
	#include "rs_time.h"
	#include <sched.h>
	#include <sys/types.h>
	#include <stdint.h>
#endif

using namespace std;

/**
 * Action enumeration for user algorithms
 */
enum User_algorithm_action {CURRENT_MODULE_READY, NEXT_MODULE_READY, WAIT_FOR_NEXT_MODULE, WAIT_FOR_CURRENT_MODULE};

template<int Ni, int No>
class User_algorithm_interface : public sc_interface {

public:

	//
	// ======== DATA ========
	//

	/**
	 * @brief Get pointer to incoming data from socket 'socket'
	 * @param socket Socket ID
	 * @return Pointer to an array of data
	 */
	virtual int* get_data_in_ptr(int socket) = 0;

	/**
	 * @brief Get pointer to incoming data from socket 'name'
	 * @param name Name of the socket connection
	 * @return Pointer to an array of data
	 */
	virtual int* get_data_in_ptr(string name) = 0;

	/**
	 * @brief Get pointer to outgoing data from socket 'socket'
	 * @param socket Socket ID
	 * @return Pointer to an array of data
	 */
	virtual int* get_data_out_ptr(int socket) = 0;

	/**
	 * @brief Get pointer to outgoing data from socket 'name'
	 * @param name Name of the socket connection
	 * @return Pointer to an array of data
	 */
	virtual int* get_data_out_ptr(string name) = 0;

	/**
	 * @brief Retrieve address from the incoming transaction on socket 'socket'
	 * @param socket Socket ID
	 * @return The address
	 */
	virtual sc_dt::uint64 get_address_in(int socket) const = 0;

	/**
	 * @brief Retrieve address from the incoming transaction on socket 'name'
	 * @param name Name of the socket connection
	 * @return The address
	 */
	virtual sc_dt::uint64 get_address_in(string name) const = 0;

	/**
	 * @brief Set address for outgoing transaction on socket 'socket'
	 * @param socket Socket ID
	 * @param value New value
	 * @return The address
	 */
	virtual void set_address_out(int socket, sc_dt::uint64 value) = 0;

	/**
	 * @brief Set address for outgoing transaction on socket 'socket'
	 * @param name Name of the socket connection
	 * @param value New value
	 * @return The address
	 */
	virtual void set_address_out(string name, sc_dt::uint64 value) = 0;

	/**
	 * @brief Get input socket data length
	 * @param socket Socket ID
	 * @return data length
	 */
	virtual int get_data_in_length(int socket) const = 0;

	/**
	 * @brief Get input socket data length
	 * @param name Name of the socket connection
	 * @return data length
	 */
	virtual int get_data_in_length(string name) const = 0;

	/**
	 * @brief Get output socket data length
	 * @param socket Socket ID
	 * @return data length
	 */
	virtual int get_data_out_length(int socket) const = 0;

	/**
	 * @brief Get output socket data length
	 * @param name Name of the socket connection
	 * @return data length
	 */
	virtual int get_data_out_length(string name) const = 0;

	/**
	 * @brief Get task worst case execution time (WCET), depending on current implementation
	 * @return WCET
	 */
	virtual sc_time get_WCET(void) const = 0;

	/**
	 * @brief Get task best case execution time (BCET), depending on current implementation
	 * @return BCET
	 */
	virtual sc_time get_BCET(void) const = 0;

	/**
	 * @brief Get input socket id by the name of the connection
	 * @param name Connection name
	 * @return Socket ID
	 */
	virtual int get_input_socket_id(string name) const = 0;

	/**
	 * @brief Get input socket id by the name of the connection
	 * @param name Connection name
	 * @return Socket ID
	 */
	virtual int get_output_socket_id(string name) const = 0;


	//
	// ======== Control ========
	//

	/**
	 * @brief Function to call at the end of algorithm
	 */
	virtual void end_of_algorithm(void) = 0;

	/**
	 * @brief Function to call to emulate preemption
	 */
	virtual void preemption_point(void) = 0;

	/**
	 * @brief Wait until an algorithm execution has been requested/granted by the manager
	 */
	virtual void b_execution_requested(void) = 0;

	/**
	 * @brief Emulate an execution time of the algorithm.
	 * @param duration Execution time
	 */
	virtual void compute(sc_time duration) = 0;

	//
	// ======== Trace ========
	//

	/**
	 * @brief Function to call when the algorithm is running in order to
	 * have an accurate trace
	 */
	virtual void set_algorithm_running(void) = 0;

	/**
	 * @brief Function to call when the algorithm is idle in order to
	 * have an accurate trace
	 */
	virtual void set_algorithm_idle(void) = 0;

	/**
	 * @brief Function to call when the algorithm is waiting for data in order to
	 * have an accurate trace
	 */
	virtual void set_algorithm_waiting(void) = 0;


	//
	// ======== User algorithm management ========
	//

	/**
	 * @brief Get the event launched when the user algorithm has to be
	 * modified (e.g. changing working implementation)
	 * @return constant reference to the SystemC event
	 */
	virtual const sc_event& update_user_algorithm(void) const = 0;

	virtual void wait_for_update_user_algorithm(void) = 0;

	/**
	 * @brief Get a reference to a boolean indicating whether the user
	 * algorithm should be killed or not (for instance, to be erplaced
	 * with a new thread).
	 * @return reference to the boolean
	 */
	virtual bool& kill_user_algorithm(void) = 0;


	//
	// ======== Data received ========
	//

	/**
	 * @brief Check if the data from channel 'name' have been
	 * received yet. Blocking function: if the data is not available
	 * at the time of the call, wait until it is received.
	 * @param name Name of the channel to check
	 */
	virtual void b_data_received(string name) = 0;

	/**
	 * @brief Check if the data from channel 'id' have been
	 * received yet. Blocking function: if the data is not available
	 * at the time of the call, wait until it is received.
	 * @param id ID of the channel to check
	 */
	virtual void b_data_received(int id) = 0;

	/**
	 * @brief Check if the data from all channels have been
	 * received yet. Blocking function: if the data is not available
	 * at the time of the call, wait until it is received.
	 */
	virtual void b_all_data_received(void) = 0;

	/**
	 * @brief Check if the data from channel 'name' have been
	 * received yet. Non-blocking function.
	 * @param name Name of the channel to check
	 * @return true if data is available
	 */
	virtual bool nb_data_received(string name) = 0;

	/**
	 * @brief Check if the data from channel 'id' have been
	 * received yet. Non-blocking function.
	 * @param id ID of the channel to check
	 * @return true if data is available
	 */
	virtual bool nb_data_received(int id) = 0;

	/**
	 * @brief Check if the data from all channels have been
	 * received yet. Non-blocking function.
	 * @return true if data is available
	 */
	virtual bool nb_all_data_received(void) = 0;


	//
	// ======== Send data ========
	//

	/**
	 * @brief Send data towards channel 'name'. Non-blocking function.
	 * @param name Name of the channel where data will be sent
	 */
	virtual void nb_send_data(string name) = 0;

	/**
	 * @brief Send data towards channel 'id'. Non-blocking function.
	 * @param id ID of the channel where data will be sent
	 */
	virtual void nb_send_data(int id) = 0;

	/**
	 * @brief Send data towards all channels. Non-blocking function.
	 */
	virtual void nb_send_all_data(void) = 0;

	/**
	 * @brief Send data towards channel 'name'. Blocking function: it will
	 * exit only when the following module has accepted the transaction
	 * @param name Name of the channel where data will be sent
	 */
	virtual void b_send_data(string name) = 0;

	/**
	 * @brief Send data towards channel 'id'. Blocking function: it will
	 * exit only when the following module has accepted the transaction
	 * @param id ID of the channel where data will be sent
	 */
	virtual void b_send_data(int id) = 0;

	/**
	 * @brief Send data towards every channels. Blocking function: it will
	 * exit only when the following modules have accepted all transactions
	 */
	virtual void b_send_all_data(void) = 0;


	//
	// ======== Data sent ========
	//

	/**
	 * @brief Check if data has been sent on a particular socket.
	 * Blocking function: if data send is not finished yet, wait until
	 * the end of the transaction. If no send request has ever been made,
	 * considers the socket ready.
	 * @param id Socket ID
	 */
	virtual void b_data_sent(int id) = 0;

	/**
	 * @brief Check if data has been sent on a particular socket.
	 * Blocking function: if data send is not finished yet, wait until
	 * the end of the transaction. If no send request has ever been made,
	 * considers the socket ready.
	 * @param name Name of the connection bound to the socket
	 */
	virtual void b_data_sent(string name) = 0;

	/**
	 * @brief Check if data has been sent on every socket.
	 * Blocking function: if data send is not finished yet, wait until
	 * the end of the transaction. If no send request has ever been made,
	 * considers the socket ready.
	 */
	virtual void b_all_data_sent(void) = 0;

	/**
	 * @brief Check if data has been sent on a particular socket.
	 * Non-blocking function. If no send request has ever been made,
	 * considers the socket ready.
	 * @param id Socket ID
	 */
	virtual bool nb_data_sent(int id) = 0;

	/**
	 * @brief Check if data has been sent on a particular socket.
	 * Non-blocking function. If no send request has ever been made,
	 * considers the socket ready.
	 * @param name Name of the connection bound to the socket
	 */
	virtual bool nb_data_sent(string name) = 0;

	/**
	 * @brief Check if data has been sent on every socket.
	 * Non-blocking function.
	 */
	virtual bool nb_all_data_sent(void) = 0;

	/**
	 * @brief Indicate that a new transaction sequence is beginning
	 * (i.e. sending new data) for a particular socket.
	 * @param id Socket ID
	 */
	virtual void start_new_transaction_sequence(int id) = 0;

	/**
	 * @brief Indicate that a new transaction sequence is beginning
	 * (i.e. sending new data) for a particular socket.
	 * @param name Name of the connection bound to the socket.
	 */
	virtual void start_new_transaction_sequence(string name) = 0;

	/**
	 * @brief Indicate that a new transaction sequence is beginning
	 * (i.e. sending new data) for every socket.
	 * @param id Socket ID
	 */
	virtual void start_new_transaction_sequence(void) = 0;



	//
	// ======== Transient channels ========
	//

	/**
	 * @brief Check if channel 'name' is transient or not
	 * @param name Name of the channel to check for transience
	 * @return true if channel is transient
	 */
	virtual bool is_channel_transient(string name) = 0;

	/**
	 * @brief Check if channel 'id' is transient or not
	 * @param id ID of the channel to check for transience
	 * @return true if channel is transient
	 */
	virtual bool is_channel_transient(int id) = 0;


	//
	// ======== Algorithm execution mode ========
	//

	/**
	 * @brief Get current algorithm execution mode
	 * @return String representation of the execution mode
	 */
	virtual string get_algorithm_execution_mode(void) = 0;

	/**
	 * @brief Check for a particular execution mode
	 * @param mode Execution mode to look after
	 * @return true if current execution mode is 'mode', false otherwise
	 */
	virtual bool is_algorithm_execution_mode(string mode) = 0;


	//
	// ======== Input socket management ========
	//

	/**
	 * @brief Release a particular input socket so that preceding module
	 * might send some more data
	 * @param socketID Socket ID
	 */
	virtual void release_input_socket(int socketID) = 0;

	/**
	 * @brief Release a particular input socket so that preceding module
	 * might send some more data
	 * @param connectionName Name of the connection bound to the socket
	 */
	virtual void release_input_socket(string connectionName) = 0;

	/**
	 * @brief Release all input socket so that preceding modules
	 * might send some more data
	 */
	virtual void release_all_input_sockets(void) = 0;


	//
	// ======== Misc ========
	//

	/**
	 * @brief Get task name
	 * @return char* representation of task name
	 */
	virtual const char* get_name(void) const = 0;

	/**
	 * @brief Get instance application top name
	 * @return char* representation of task name
	 */
	virtual string get_top_application_name(void) const = 0;

	/**
	 * @brief Get the number of preemption points that should
	 * be inserted in the algorithm
	 * @return number of preemption points
	 */
	virtual int get_nb_preemption_points(void) = 0;

	/**
	 * @brief Wait until the next period is started. For non-periodic tasks,
	 * return immediately.
	 */
	virtual void wait_until_next_period(void) = 0;

	/**
	 * @brief Get logfile
	 * @return Log file
	 */
	virtual ofstream& get_logfile(void) = 0;

};

#endif
