/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    trace.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-09-29
 * @section DESCRIPTION
 *			Some functions and structures used for VCD tracing
 */

#ifndef TRACE_H
#define TRACE_H

#include <cstdlib>
#include "systemc"
#include "tlm.h"
#include "utils.h"
#include "config.h"

#include "manager_interface.h"
#include "reconfigurable_zone.h"
#include "fpga.h"

using namespace sc_core;
using namespace sc_dt;
using namespace std;
using namespace tlm;

// Useful ASCII words

#define MAPPED_ASCII            0x4D617070656420
#define RUNNING_ASCII           0x52756E6E696E6720
#define CONFIGURING_ASCII       0x436F6E66696720
#define QUEUED_ASCII            0x51756575656420
#define CONTEXT_SAVE_ASCII		0x4374787420535620
#define CONTEXT_LOAD_ASCII		0x43747874204C4420
#define IDLE_ASCII				0x49646C65
#define WAITING_ASCII			0x57616974696E67
#define ACTIVE_ASCII			0x41637469766520

// ASCII words for VCD tracing
// TLM phases

#define BEGIN_REQ_ASCII			0x425F524551
#define END_REQ_ASCII			0x455F524551
#define BEGIN_RESP_ASCII		0x425F52455350
#define END_RESP_ASCII			0x455F52455350
#define INTERN_ASCII			0x494E5445524E
#define UNDEF_ASCII				0x554E444546
#define NULL_ASCII				0x4E554c4c

// TLM status

#define RESPONSE_OK_ASCII		0x4F4B
#define RESPONSE_ERROR_ASCII	0x455252

// TLM command

#define COMMAND_R_ASCII			0x52
#define COMMAND_W_ASCII			0x57

// Algorithm status

#define ALGORITHM_RUNNING_ASCII		RUNNING_ASCII
#define ALGORITHM_WAITING_ASCII		WAITING_ASCII

// Scheduling

#define SCHEDULER_ACTIVE_ASCII	ACTIVE_ASCII
#define ICAP_RECONF_T_ASCII		CONFIGURING_ASCII
#define RZ_RUNNING_T_ASCII		RUNNING_ASCII
#define RZ_MAPPED_T_ASCII		MAPPED_ASCII
#define RZ_BLANK_ASCII			0x424C414E4B
#define OFFSET_ASCII			0x4F6666736574

// RZ communication status

#define COM_WAITING_T_ASCII		WAITING_ASCII
#define COM_TX_T_ASCII			0x5478


/**
 * Basic packets definition
 */
struct transaction_packet_t {
	int command;
	int address;
	int data;
	int response_status;

	transaction_packet_t(void) {
		command = NULL_ASCII;
		address = 0;
		data = 0;
		response_status = NULL_ASCII;
	}

	void update_transaction(tlm_generic_payload* trans);
};

struct initiator_packet_t {
	sc_dt::uint64 phase;
	transaction_packet_t transaction;
};

struct target_packet_t {
	sc_dt::uint64 phase;
	transaction_packet_t transaction;
	transaction_packet_t transaction_pending;
};

/**
 * Module trace packet definition
 */
template<int Ni, int No>
struct trace_packet_t {

	bool trace_configuration_signals;
	bool trace_debug_signals;

	int address_in[Ni];
	int data_in[Ni];
	target_packet_t target_packet[Ni];

	int address_out[No];
	int data_out[No];
	initiator_packet_t initiator_packet[No];
	sc_dt::uint64 communication_status[No];

	sc_bigint<TRACE_LV_SIZE> algorithm_execution_mode;

	trace_packet_t(void) {
		trace_configuration_signals = false;
		trace_debug_signals = false;
	}
};


/**
 * Testbench trace packet definition
 */
template<int Ni, int No>
class trace_packet_testbench_t {
public:
	initiator_packet_t initiator_packet[No];
	target_packet_t target_packet[Ni];
};

// Template specialization if no inputs
template<int No>
class trace_packet_testbench_t<0, No> {
public:
	initiator_packet_t initiator_packet[No];
};

// Template specialization if no outputs
template<int Ni>
class trace_packet_testbench_t<Ni, 0> {
public:
	target_packet_t target_packet[Ni];
};

// Template specialization if no outputs
template<>
class trace_packet_testbench_t<0, 0> {
public:
	// Empty class
};


/**
 * Manager trace packet definition
 */
struct trace_packet_manager_t {

	// Manager parameters
	int nbRZs;
	int nbTasks;
	unsigned int nbCallsScheduler;
	int nb_hw_reconfiguration_units;
	int nb_sw_reconfiguration_units;

	// Trace configuration
	bool trace_debug_signals;

	// Trace for RZ
	sc_bigint<TRACE_LV_SIZE>* icap_status;
	sc_bigint<TRACE_LV_SIZE>* software_loader_status;
	sc_bigint<TRACE_LV_SIZE>  scheduler_status;
	sc_bigint<TRACE_LV_SIZE>* task_status;
	sc_bigint<TRACE_LV_SIZE>* rz_status;

	// Occupation rates
	double* icap_occupation_rate;
	double* software_loader_occupation_rate;
	double* rz_occupation_rate;
	double* rz_running_rate;
	double* rz_mapped_rate;
	double* rz_config_rate;
	double* rz_blank_rate;

	// Resource occupation rates
	double fpga_processor_occupation_rate;
	map<string, double> fpga_resource_occupation_rate;
	map<string, double>* rz_resource_occupation_rate;

	// RZ and tasks pointers
	vector<Manager_interface *> *modules_table_ptr;
	vector<RZ *> *rz_table_ptr;

	// Hyperperiod trace (ASCII)
	sc_bigint<TRACE_LV_SIZE> hyperperiod_status;

	trace_packet_manager_t(vector<Manager_interface *> *modules_table, vector<RZ *> *rz_table, int nb_hw_unit, int nb_sw_unit, FPGA* fpga) {

		modules_table_ptr = modules_table;
		rz_table_ptr = rz_table;

		nbTasks = modules_table_ptr->size();
		nbRZs = rz_table_ptr->size();
		nb_hw_reconfiguration_units = nb_hw_unit;
		nb_sw_reconfiguration_units = nb_sw_unit;
		fpga_processor_occupation_rate = 0;

		//initiator_packet = new initiator_packet_t[nbTasks];
		task_status = new sc_bigint<TRACE_LV_SIZE>[nbTasks];
		rz_status = new sc_bigint<TRACE_LV_SIZE>[nbRZs];

		icap_occupation_rate = new double[nb_hw_unit];
		icap_status = new sc_bigint<TRACE_LV_SIZE>[nb_hw_unit];
		software_loader_occupation_rate = new double[nb_sw_unit];
		software_loader_status = new sc_bigint<TRACE_LV_SIZE>[nb_sw_unit];
		rz_resource_occupation_rate = new map<string, double>[nbRZs];
		rz_occupation_rate = new double[nbRZs];
		rz_running_rate = new double[nbRZs];
		rz_mapped_rate = new double[nbRZs];
		rz_config_rate = new double[nbRZs];
		rz_blank_rate = new double[nbRZs];

		// HW reconfiguration units
		for(int i = 0; i < nb_hw_reconfiguration_units; i++) {
			icap_occupation_rate[i] = 0;
			icap_status[i] = IDLE_ASCII;
		}

		// SW reconfiguration units
		for(int i = 0; i < nb_sw_reconfiguration_units; i++) {
			software_loader_occupation_rate[i] = 0;
			software_loader_status[i] = IDLE_ASCII;
		}

		// FPGA resources
		map<string, int> fpgaResources(fpga->get_resources().getResourceMap());
		for(map<string, int>::const_iterator it = fpgaResources.begin(); it != fpgaResources.end(); it++) {
			fpga_resource_occupation_rate[it->first] = 0;
		}

		// RZ initialization
		for(int i = 0; i < nbRZs; i++) {
			rz_occupation_rate[i] = 0;
			rz_running_rate[i] = 0;
			rz_mapped_rate[i] = 0;
			rz_config_rate[i] = 0;
			rz_blank_rate[i] = 0;
			rz_status[i] = RZ_BLANK_ASCII;
			
			map<string, int> rzResources(rz_table->at(i)->get_resources_ptr()->getResourceMap());
			for(map<string, int>::const_iterator it = rzResources.begin(); it != rzResources.end(); it++) {
				rz_resource_occupation_rate[i][it->first] = 0;
			}
		}

		// Initialization
		nbCallsScheduler = 0;
		scheduler_status = IDLE_ASCII;
		for(int i = 0; i < nbTasks; i++) task_status[i] = IDLE_ASCII;

		trace_debug_signals = false;

		hyperperiod_status = OFFSET_ASCII;
	}

	~trace_packet_manager_t() {
		delete[] task_status;
		delete[] rz_status;
		delete[] icap_occupation_rate;
		delete[] icap_status;
		delete[] software_loader_occupation_rate;
		delete[] software_loader_status;
		delete[] rz_occupation_rate;
		delete[] rz_resource_occupation_rate;
	}
};

// sc_trace overload
template<int Ni, int No>	void sc_trace(sc_trace_file *tf, const trace_packet_t<Ni, No>& packet, const string& NAME);
template<int Ni, int No>	void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<Ni, No>& packet, const string& NAME);
template<int Ni>			void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<Ni, 0>& packet, const string& NAME);
template<int No>			void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<0, No>& packet, const string& NAME);
void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<0, 0>& packet, const string& NAME);
void sc_trace(sc_trace_file *tf, const trace_packet_manager_t& packet, const string& NAME);
void sc_trace(sc_trace_file *tf, const initiator_packet_t& packet, const string& NAME);
void sc_trace(sc_trace_file *tf, const target_packet_t& packet, const string& NAME);
void sc_trace(sc_trace_file *tf, const transaction_packet_t& packet, const string& NAME);

// Helper functions
sc_dt::uint64 tlm_phase_to_ascii(tlm_phase phase);


template<int Ni, int No>
void sc_trace(sc_trace_file *tf, const trace_packet_t<Ni, No>& packet, const string& NAME) {
	
	for(int i = 0; i < Ni; i++) {
		sc_trace(tf, packet.address_in[i], NAME + ".address_in_" + Utils::itoa(i));
		sc_trace(tf, packet.data_in[i], NAME + ".data_in_" + Utils::itoa(i));
		sc_trace(tf, packet.target_packet[i], NAME + ".target_" + Utils::itoa(i));
	}

	for(int i = 0; i < No; i++) {
		sc_trace(tf, packet.address_out[i], NAME + ".address_out_" + Utils::itoa(i));
		sc_trace(tf, packet.data_out[i], NAME + ".data_out_" + Utils::itoa(i));
		sc_trace(tf, packet.initiator_packet[i], NAME + ".initiator_" + Utils::itoa(i));
		sc_trace(tf, packet.communication_status[i],NAME + ".communication_" + Utils::itoa(i));
	}

	if(packet.trace_configuration_signals) sc_trace(tf, packet.algorithm_execution_mode, NAME + ".execution_mode");
}

template<int Ni, int No>
void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<Ni, No>& packet, const string& NAME) {
	for(int i = 0; i < Ni; i++) sc_trace(tf, packet.target_packet[i],		NAME + ".target_" + Utils::itoa(i));
	for(int i = 0; i < No; i++) sc_trace(tf, packet.initiator_packet[i],	NAME + ".initiator_" + Utils::itoa(i));
}

template<int Ni>
void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<Ni, 0>& packet, const string& NAME) {
	for(int i = 0; i < Ni; i++) sc_trace(tf, packet.target_packet[i],		NAME + ".target_" + Utils::itoa(i));
}

template<int No>
void sc_trace(sc_trace_file *tf, const trace_packet_testbench_t<0, No>& packet, const string& NAME) {
	for(int i = 0; i < No; i++) sc_trace(tf, packet.initiator_packet[i],	NAME + ".initiator_" + Utils::itoa(i));
}

#endif
