/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    task_to_schedule_interface.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2013-03-18
 * @section DESCRIPTION
 *			Interface to be passed as a parameter for task priority comparison and sorting
 */

#ifndef TASK_TO_SCHEDULE_INTERFACE_H
#define TASK_TO_SCHEDULE_INTERFACE_H

#include "reconfigurable_zone.h"
#include "task_implementation.h"
#include "task_priority_comparator_interface.h"

class Task_to_schedule_interface : public Task_priority_comparator_interface {

public:

	/**
	 * @brief Check if task has the implementation passed as parameter
	 * @param impl Implementation to look for
	 * @return true if task has implementation
	 */
	virtual bool has_implementation(TaskImplementation& impl) const = 0;

	/**
	 * @brief Check if task fits a reconfigurable zone (verifying every
	 * hardware implementations defined for the module)
	 * @param rz Pointer to the reconfigurable zone to check
	 * @return true if module fits the RZ
	 */
	virtual bool fits(RZ *rz) = 0;

	/**
	 * @brief Return module name
	 * @return string representation of module name
	 */
	virtual string getName(void) const = 0;

	/**
	 * @brief Retrieve time of the beginning for next period.
	 * @return Absolute time of the new period beginning
	 */
	virtual sc_time get_new_period_start_time(void) const = 0;

	/**
	 * @brief Change the task priority
	 * @param New task priority
	 */
	virtual void set_task_priority(int priority) = 0;

	/**
	 * @brief Get the current implementation
	 * @return current implementation
	 */
	virtual ModuleImplementation get_current_implementation(void) = 0;

	/**
	 * @brief Check if exist a current implementation
	 * @return true if exist, else false
	 */
	virtual bool has_current_implementation(void) = 0;

	/**
	 * @brief Get a list of tasks following this one in the block diagram. If an ID of vector is -1, the following task is a testbench.
	 * @return A vector containing IDs of following tasks
	 */
	virtual vector<int> getFollowingList(void) const = 0;

	/**
	 * @brief Get a list of tasks preceding this one in the block diagram. If an ID of vector is -1, the preceding task is a testbench.
	 * @return A vector containing IDs of preceding tasks
	 */
	virtual vector<int> getPrecedingList(void) const = 0;

};

#endif
