/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    scheduler_interface.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-02
 * @section DESCRIPTION
 *			Interface for scheduling algorithm development
 */

#ifndef SCHEDULER_INTERFACE_H
#define SCHEDULER_INTERFACE_H

#include <vector>
#include <ostream>
#include "task_implementation.h"
#include "reconfigurable_zone.h"
#include "scheduler_request.h"
#include "application_interface.h"

/* Macro to define User Mapping/Scheduling Algo */
#define MAPPING_ALGORITHM(nom) Return_mapping_information waiting_queue_handler_mapping_ ## nom  (Scheduler_interface &scheduler)
#define SCHEDULING_ALGORITHM(nom) void waiting_queue_handler_scheduling_ ## nom (Scheduler_interface &scheduler)

#define DEFAULT_MAPPING_ALGORITHM(nom) Return_mapping_information waiting_queue_handler_mapping_ ## nom (Scheduler_interface &scheduler) { \
	return waiting_queue_handler_mapping_default_algo(scheduler); \
	}

#define DEFAULT_SCHEDULING_ALGORITHM(nom) void waiting_queue_handler_scheduling_ ## nom (Scheduler_interface &scheduler) { \
	waiting_queue_handler_scheduling_default_algo(scheduler); \
	}

/* Macro only for Generation */
#define WAITING_QUEUE_HANDLER(nom) void waiting_queue_handler_## nom (Scheduler_interface &scheduler)
#define WAITING_QUEUE_HANDLER_REF(nom) waiting_queue_handler_## nom
#define CALL_WAITING_QUEUE_HANDLER(nom) waiting_queue_handler_## nom (scheduler)
#define CALL_MAPPING_ALGORITHM(nom) waiting_queue_handler_mapping_ ## nom (scheduler)
#define CALL_SCHEDULING_ALGORITHM(nom) waiting_queue_handler_scheduling_ ## nom (scheduler)


/* For User : Return information from Mapping Algorithm */
class Return_mapping_information {

	bool rz_found;
	bool exit_mapping_algo;
	RZ* implementation_RZ;
	int implementation_ID;
	Config_fct_point* cfgFctPt;

public:
	Return_mapping_information() {
		rz_found = false;
		exit_mapping_algo = true;
		cfgFctPt = NULL;
	}

		Return_mapping_information(RZ* _implementation_RZ, int _implementation_ID) {
		rz_found = true;
		exit_mapping_algo = true;
		implementation_RZ = _implementation_RZ;
		implementation_ID = _implementation_ID;
		cfgFctPt = NULL;
	}

	Return_mapping_information(RZ* _implementation_RZ, int _implementation_ID, Config_fct_point* _cfgFctPt) {
		rz_found = true;
		exit_mapping_algo = true;
		implementation_RZ = _implementation_RZ;
		implementation_ID = _implementation_ID;
		cfgFctPt = _cfgFctPt;
	}

	Return_mapping_information(RZ* _implementation_RZ, int _implementation_ID, bool exit_mapping_algorithm) {
		rz_found = true;
		exit_mapping_algo = exit_mapping_algorithm;
		implementation_RZ = _implementation_RZ;
		implementation_ID = _implementation_ID;
		cfgFctPt = NULL;
	}

		Return_mapping_information(RZ* _implementation_RZ, int _implementation_ID, Config_fct_point* _cfgFctPt, bool exit_mapping_algorithm) {
		rz_found = true;
		exit_mapping_algo = exit_mapping_algorithm;
		implementation_RZ = _implementation_RZ;
		implementation_ID = _implementation_ID;
		cfgFctPt = _cfgFctPt;
	}

	Return_mapping_information(const Return_mapping_information &other) {
		rz_found = other.rz_found;
		implementation_RZ = other.implementation_RZ;
		implementation_ID = other.implementation_ID;
		cfgFctPt = other.cfgFctPt;
	}

	bool RZ_found() {
		return rz_found;
	}

		
	bool hasCfgTcfPt() {
		return cfgFctPt != NULL;
	}

	bool exit_mapping_algorithm() {
		return exit_mapping_algo;
	}

	void set_exit_mapping_algo(bool exit_mapping_algorithm) {
		exit_mapping_algo = exit_mapping_algorithm;
	}

	RZ* get_implementation_RZ() {
		return implementation_RZ;
	}

	int get_implementation_ID() {
		return implementation_ID;
	}

	Config_fct_point* getConfigFctPoint() {
		return cfgFctPt;
	}
};



class Scheduler_interface : public sc_interface {

public:
	
	//
	// ======= Waiting queue =======
	//

	/**
	 * @brief Check if there are tasks waiting in the queue
	 * @return true if tasks are waiting to be instantiated
	 */
	virtual bool Scheduler_are_tasks_waiting(void) const = 0;

	/**
	 * brief Get request with the highest priority inside waiting queue
	 * return Top request
	 */
	//virtual SchedulerRequest Scheduler_get_top_request(void) = 0;

	/**
	 * @brief Get request with the current task in waiting queue
	 * @return request
	 */
	virtual SchedulerRequest& Scheduler_current_task_waiting() = 0;	 

	/**
	 * brief Remove highest priority request from the queue
	 */
	//virtual void Scheduler_pop_request(void) = 0;

	/**
	 * @brief Get waiting queue size
	 * @return Waiting queue size
	 */
	virtual int Scheduler_get_waiting_queue_size(void) = 0;

	/**
	 * @brief Get position of the current waiting queue element
	 * @return Position
	 */
	virtual int Scheduler_position_of_current_element(void) = 0;

	/**
	 * @brief Get a reference to an element in the queue
	 * @param position Position of the element to retrieve from the queue
	 * @return Reference to the scheduler request
	 */
	virtual SchedulerRequest& Scheduler_get_element(int position) = 0;

	/**
	 * @brief Update priority queue after changes (e.g. priority change)
	 */
	virtual void Scheduler_update_queue(void) = 0;

	/**
	 * @brief Get pointer to the task 'taskID'
	 */
	virtual Task_to_schedule_interface* Scheduler_get_task_to_schedule_ptr(int taskID) = 0;

	/**
	 * @brief Reserved
	 */
	virtual bool Scheduler_last_task_waiting(void) const = 0;	

	/**
	 * @brief Reserved
	 */
	virtual void Scheduler_erase_current_task_waiting() = 0;

	/**
	 * @brief Reserved
	 */
	virtual SchedulerRequest Scheduler_next_task_waiting() = 0;

	/**
	 * @brief Reserved
	 */
	virtual void Scheduler_reset_current_position_in_waiting_queue() = 0;


	//
	// ======= RZ & tasks =======
	//

	/**
	 * @brief Get the list of reconfigurable zones that fit task 'task_id'
	 * @param task_id Task identifier
	 * @return Vector of pointers to RZ objects
	 */
	virtual vector<RZ *> Scheduler_get_compatible_rz_vector(int task_id) = 0;

	/**
	 * @brief Get running time of the task since the last idle state
	 * @param task_id Task identifier
	 * @return running time
	 */
	virtual sc_time Scheduler_get_task_running_time(int task_id) = 0;

	/**
	 * @brief Get the list of all reconfigurable zones used in the architecture
	 * @return Vector of pointers to RZ objects
	 */
	virtual vector<RZ *> Scheduler_get_all_rz_vector(void) = 0;

	/**
	 * @brief Get ID of task currently instantiated on a reconfigurable zone
	 * @param rz Pointer to the reconfigurable zone
	 * @return Task ID
	 */
	virtual int Scheduler_get_current_module_ID(RZ* rz) = 0;

	/**
	 * @brief Check if RZ is blank or used by a task
	 * @param rz Pointer to the reconfigurable zone
	 * @return true if RZ is blank
	 */
	virtual bool Scheduler_is_RZ_blank(RZ *rz) = 0;

	/**
	 * @brief Retrieve task ID associated with shceduler request
	 * @param req Reference to the SchedulerRequest
	 * @return Task ID
	 */
	virtual int Scheduler_get_module_ID(SchedulerRequest& req) = 0;


	/**
	 * @brief Get the number of applications
	 * @return number of application
	 */
	virtual int Scheduler_get_application_number(void) = 0;

	/**
	 * @brief Get the application interface from an index
	 * @return Pointer to application interface
	 */
	virtual Application_interface* Scheduler_get_application_interface(int index) = 0;

	/**
	 * @brief test if the RZ properties of the instance exists (Processor or Hw RZ)
	 * @param inst_name Instance of the processor or default RZ ("RZDefaultGroupName")
	 * @return return true if the RZ exists
	 */
	virtual bool Scheduler_exist_rz_properties(string inst_name) = 0;

	/**
	 * @brief test if the RZ properties of the instance exists (Processor or Hw RZ)
	 * @param rz Pointer to the RZ
	 * @return return true if the RZ exists
	 */
	virtual bool Scheduler_exist_rz_properties(RZ *rz) = 0;

	/**
	 * @brief Get the RZ properties of the instance (Processor or Hw RZ)
	 * @param inst_name Instance of the processor or default RZ ("RZDefaultGroupName")
	 * @return Pointer to RZ properties
	 */
	virtual RZ_config& Scheduler_get_rz_properties(string inst_name) = 0;

	/**
	 * @brief Get the RZ properties of the RZ (Processor or Hw RZ)
	 * @param rz Pointer to the RZ
	 * @return Pointer to RZ properties
	 */
	virtual RZ_config& Scheduler_get_rz_properties(RZ *rz) = 0;

	/**
	 * @brief Get the current fct. point of the RZ (Processor or Hw RZ)
	 * @param rz Pointer to the RZ
	 * @return Pointer to current fct. point
	 */
	virtual Config_fct_point& Scheduler_get_current_fct_point(RZ *rz) = 0;

	
	//
	// ======= Scheduler execution =======
	//

	/**
	 * @brief Emulate scheduler behaviour in terms of execution time and energy consumption
	 */
	virtual void Scheduler_emulate_scheduler_behavior(void) = 0;

	/**
	 * @brief Set scheduler active for task 'taskid'
	 * @param taskid ID of the task being processed
	 */
	virtual void Scheduler_set_scheduler_active(int taskid) = 0;

	
	//
	// ======= Requests and owners =======
	//

	/**
	 * @brief Check if a task placement has already been demanded by the same predecessor
	 * @param task_to_map ID of the task which mapping is requested
	 * @param request_owner ID of the task requiring mapping of the task
	 * @return true if a similar request has already been performed
	 */
	virtual bool Scheduler_has_task_already_requested_mapping(int task_to_map, int request_owner) = 0;	

	/**
	 * @brief Add a new finished precedence to a task
	 * @param task_mapped ID of the task already mapped on the FPGA that has a new finished predecessor
	 * @param precedence ID of the preceding task
	 */
	virtual void Scheduler_add_finished_precedence(int task_mapped, int precedence) = 0;

	/**
	 * @brief Notify the owner of a request that it has been granted (i.e. the module is
	 * instantiated on the FPGA and ready to receive data).
	 * @param owner Request owner ID
	 * @param ready Ready module ID
	 */
	virtual void Scheduler_notify_request_owner_module_ready(int owner, int ready) = 0;

	/**
	 * @brief Prints the list of finished predecessors for a task
	 * @param task_id ID of the task to consider
	 */
	virtual void Scheduler_print_finished_precedence_vector(int task_id) = 0;

	/**
	 * @brief Send parameters to update algorithms
	 * @param rz Pointer to the RZ will be sent to algorithm
	 */
	virtual void Scheduler_send_update_parameters_to_module(RZ* rz) = 0;
	
	
	//
	// ======= Task configuration =======
	//

	/**
	 * @brief Launch task configuration through either HW of SW reconfiguration unit
	 * @param rz Reconfigurable zone (processing element) on which task is configured
	 * @param id ID of the module needing reconfiguration
	 * @param implID TaskImplementation ID to replace previous one
	 * @param requestOwner ID of the module that requested this configuration
	 */
	virtual void Scheduler_configure_task(RZ* rz, int id, int implID, int requestOwner) = 0;

	/**
	 * @brief Changes a task from MAPPED to RUNNING state
	 * @param Reference to the scheduler request issuing this state change
	 */
	virtual void Scheduler_rerun_task(SchedulerRequest& req) = 0;

	/**
	 * @brief Set current task hosted by a reconfigurable zone
	 * @param rz Pointer to the RZ to consider
	 * @param task_id Task which status changed
	 */
	virtual void Scheduler_set_current_task(RZ *rz, int task_id) = 0;


	//
	// ======= Task/RZ state =======
	//

	/**
	 * @brief Get task state
	 * @param task_id ID of the task to consider
	 * @return Task state
	 */
	virtual Task_state Scheduler_get_task_state(int task_id) = 0;

	/**
	 * @brief Get state of the task instantiated on a reconfigurable zone
	 * @param rz Pointer to the RZ to consider
	 * @return Task state
	 */
	virtual Task_state Scheduler_get_task_state(RZ *rz) = 0;


	/**
	 * @brief Set blank a reconfigurable zone 
	 * @param RZ id to consider
	 * @return true if possible (no task running on the rz)
	 */
	virtual bool Scheduler_set_blank(int hosting_rz_id) = 0;

	/**
	 * @brief Set blank a reconfigurable zone 
	 * @param rz Pointer to the RZ to consider
	 * @return true if possible (no task running on the rz)
	 */
	virtual bool Scheduler_set_blank(RZ *rz) = 0;

	

	//
	// ======= Preemption =======
	//

	/**
	 * @brief Check if task is preempted
	 * @param taskid Task to check for preemption
	 */
	virtual bool Scheduler_is_task_preempted(int taskid) const = 0;


	//
	// ======= Scheduler effort =======
	//

	/**
	 * @brief Get scheduler performance effort
	 * @return Performance effort
	 */
	virtual double Scheduler_get_performance_effort(void) const = 0;

	/**
	 * @brief Get scheduler power effort
	 * @return Power effort
	 */
	virtual double Scheduler_get_power_effort(void) const = 0;

	/**
	 * @brief Get scheduler area effort
	 * @return Area effort
	 */
	virtual double Scheduler_get_area_effort(void) const = 0;


	//
	// ======= Misc =======
	//

	/**
	 * @brief Get scheduler instance name
	 * @return char* represnetation of reconfiguration manager name
	 */
	virtual const char* Scheduler_get_name(void) const = 0;

	/**
	 * @brief Get output stream
	 * @return Reference to the output stream
	 */
	virtual ostream& Scheduler_get_output_stream(void) = 0;

	/**
	 * @brief Display task state table on the standard output
	 */
	virtual void Scheduler_display_task_state_table(void) const = 0;

	/**
	 * @brief Displays RZ occupation on the standard output
	 */
	virtual void Scheduler_display_rz_current_module_table(void) const = 0;

	/**
	 * @brief Display waiting queue
	 */
	virtual void Scheduler_display_waiting_queue(void) const = 0;

};

/* Default */
WAITING_QUEUE_HANDLER(default_algo);
MAPPING_ALGORITHM(default_algo);
SCHEDULING_ALGORITHM(default_algo);

/* Mapping/Sheduling Algorithm : AMAP_EDF */
WAITING_QUEUE_HANDLER(AMAP_EDF);
MAPPING_ALGORITHM(AMAP_EDF);
SCHEDULING_ALGORITHM(AMAP_EDF);

#endif
