/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    Reconfiguration_manager_base.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-07-01
 * @section DESCRIPTION
 *			The reconfiguration manager
 */

#ifndef RECONFIGURATION_MANAGER_BASE_H
#define RECONFIGURATION_MANAGER_BASE_H

#include "utils.h"
#include "memory_manager.h"
#include "trace.h"
#include "utils.h"
#include "manager_interface.h"
#include "reconfigurable_zone.h"
#include "task_priority_comparator.h"
#include "my_priority_queue.h"
#include "simulation_controller.h"
#include "fpga.h"
#include "task.h"
#include "application.h"
#include "scheduler_interface.h"
#include "scheduling.h"
#include "reconfiguration_unit.h"
#include "scheduler_request.h"
#include "configuration_request.h"

#ifdef RECOSIM_PART
	#define SC_INCLUDE_DYNAMIC_PROCESSES

	#include <systemc.h>
	#include "tlm.h"
	#include "tlm_utils/peq_with_cb_and_phase.h"
	#include "tlm_utils/multi_passthrough_initiator_socket.h"
	#include "tlm_utils/multi_passthrough_target_socket.h"
	#include "tlm_utils/simple_target_socket.h"
	#include "tlm_utils/simple_initiator_socket.h"

	#include "trace.h"
	#include "simulation_controller.h"
	#include "monitoring_interface.h"
	#include "qos_management.h"

	using namespace tlm;
	using namespace tlm_utils;
	using namespace sc_core;
	using namespace sc_dt;
#endif

#ifdef MIDDLEWARE_GEN
	#include "rs_event.h"
	#include "rs_time.h"
	#include "rs_buffer.h"
	#include "rs_socket.h"

	#include <thread>
	#include <sys/types.h>
#endif

#include <map>
#include <vector>
#include <list>

using namespace sc_core;
using namespace sc_dt;
using namespace std;
using namespace tlm;
using namespace tlm_utils;

//#define GENERATE_RZ_STATE_FILE
//#define GENERATE_TASK_STATE_FILE

/* For User : Return information from Mapping Algorithm */
class Running_time_information {

	Task_state state;
	sc_time last_running_time;
	sc_time running_time_duration;

public:
	Running_time_information() {
		reset_time();
	}

	Running_time_information(const Running_time_information &other) {
		state = other.state;
		last_running_time = other.last_running_time;
		running_time_duration = other.running_time_duration;
	}

	// New Task Execution
	void reset_time() {
		state = IDLE;
		last_running_time = sc_time(0, SC_NS);
		running_time_duration = sc_time(0, SC_NS);
	}

	void update_running_time(Task_state new_state) {

		sc_time current_time = sc_time_stamp();

		if (new_state == IDLE) {
			reset_time(); 
		} 
		else if (new_state == RUNNING && state != RUNNING) {
			 last_running_time = current_time;
		} 
		else if (state == RUNNING) {
			running_time_duration += (current_time - last_running_time);
			last_running_time = current_time;
		}

		state = new_state;
	}

	sc_time get_running_time_duration() {
		if (state == RUNNING)
			return running_time_duration + (sc_time_stamp() - last_running_time);
		else
			return running_time_duration;
	}

};


class Reconfiguration_manager_base : public sc_module, public Scheduler_interface, public Monitoring_interface, public QoS_interface {

protected:
	//
	// Simulation and scheduler parameters
	//
	static sc_time SCHEDULER_EXECUTION_TIME;
	static double SCHEDULER_ENERGY_CONSUMPTION;
	static double ENERGY_CONSUMPTION_LIMIT;
	static bool GENERATE_ZERO_FILLED_LINE_IN_CSV;
	static double SCHEDULER_PERFORMANCE_EFFORT;
	static double SCHEDULER_POWER_EFFORT;
	static double SCHEDULER_AREA_EFFORT;
	static int NB_HW_RECONFIGURATION_UNITS;
	static int NB_SW_RECONFIGURATION_UNITS;
	static bool ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST;
	static sc_time TRACE_WINDOW_FUNCTION_LENGTH;
	static sc_time MAXIMUM_SIMULATION_TIME;

	//
	// Simulation controller part
	//
	static int nbApplicationTerminated; /**< Number of terminated applications */

	//
	// Constants
	//
	static int RZ_TASK_BLANK;
	static bool CONSIDER_WAITING_STATE_AS_RUNNING_MODE;
	static bool SEND_CONFIGURATION_NOTIFICATION_AFTER_IDLE_STATE;

public:

		// Configuration interface to sub-modules
	multi_passthrough_target_socket<Reconfiguration_manager_base> module_target_socket;
	
	// Communication interface, taking orders from user
	multi_passthrough_target_socket<Reconfiguration_manager_base> testbench_target_socket;

	// Configuration interface to sub-modules
	multi_passthrough_initiator_socket<Reconfiguration_manager_base> module_initiator_socket;

	// Communication interface, taking orders from user
	multi_passthrough_initiator_socket<Reconfiguration_manager_base> testbench_initiator_socket;


protected:
	// Memory manager
	Memory_manager m_mm;

	// Events
	sc_event scheduler_thread_event;	// Event to check waiting queue

	// Log file stream
	ofstream* fout;
#ifdef GENERATE_TASK_STATE_FILE
	ofstream  file_task_state;
#endif
#ifdef GENERATE_RZ_STATE_FILE
	ofstream  file_rz_state;
#endif

	// Trace file
	sc_trace_file* tf;
	trace_packet_manager_t* trace_packet;

	// Maps for scheduling
	map<int, vector<RZ *> > map_compatible_rz;
	map<RZ *, int> map_current_module;
	vector<Task_state> task_state_table;

	// Application & Architecture
	vector<Manager_interface *> modules_table;
	vector<RZ *> rz_table;
	TaskImplementation* previous_implementation_on_RZ;
	TaskImplementation* current_implementation_on_RZ;
	static vector<Application> applicationVector;

	// Preemption gestion
	vector<int> preempted_tasks_list;

	// Running time of tasks
	vector<Running_time_information> running_time_table;

	// Reconfiguration engine
	ReconfigurationUnit* hw_reconfiguration_units;
	list<ConfigurationRequest> reconfiguration_waiting_list;
	sc_event* hw_reconfiguration_event_table;

	// Software reconfiguration engine
	ReconfigurationUnit* sw_reconfiguration_units;
	list<ConfigurationRequest> software_reconfiguration_waiting_list;
	sc_event* sw_reconfiguration_event_table;

	// Scheduler
	void (*waiting_queue_handler_ptr)(Scheduler_interface &);
	int nb_calls_scheduler;
	sc_time scheduler_busy_time;
	vector<vector<int> > finished_precedence_vectors;

	// Monitoring
	vector<sc_event* > monitoring_end_of_scheduling_event_list;
	vector<sc_event* > monitoring_start_of_scheduling_event_list;
	vector<sc_event* > monitoring_update_rz_state_event_list;
	vector<sc_event* > monitoring_update_task_state_event_list;

	// QoS Management
	QoS_management *qos_management;

	// Misc
	static FPGA* fpga;
	vector<Task> task_table;
	double total_energy_consumption;

	map<string, RZ_config*> rz_parameter_table;
	

	sc_time *deadlines_table;

	sc_buffer<ConfigurationRequest> *rz_configuration_request_buffer;
	sc_event *rz_configured_event;

	bool simulationOK;

public:
	SC_HAS_PROCESS(Reconfiguration_manager_base);
	Reconfiguration_manager_base(sc_module_name instname, vector<RZ *> rzs, vector<Task> tasks, void (*handler)(Scheduler_interface &), FPGA* device, ofstream& logfile, sc_trace_file* mtf, map<string, RZ_config*>& table) :
			sc_module(instname),
			module_initiator_socket("Manager2Module_initiator_socket"),
			testbench_initiator_socket("Manager2Testbench_initiator_socket"),
			module_target_socket("Manager2Module_target_socket"),
			testbench_target_socket("Manager2Testbench_target_socket"),
			rz_table(rzs),
			fout(&logfile),

#ifdef GENERATE_TASK_STATE_FILE
			file_task_state("log/task.csv"),
#endif
#ifdef GENERATE_RZ_STATE_FILE
			file_rz_state("log/rz.csv"),
#endif
			scheduler_busy_time(SC_ZERO_TIME),
			tf(mtf),
			task_table(tasks),
			rz_parameter_table(table),
			waiting_queue_handler_ptr(handler) {

#ifdef GENERATE_RZ_STATE_FILE
		for(int i = 0; i < (int) rz_table.size(); i++) file_rz_state << "; " << rz_table[i]->getName();
		file_rz_state << endl;
#endif

		nb_calls_scheduler = 0;
		total_energy_consumption = 0;

		simulationOK = false;

		//fpga = new FPGA(device);

		fpga = device;

		previous_implementation_on_RZ = new TaskImplementation[(int) rz_table.size()];
		current_implementation_on_RZ = new TaskImplementation[(int) rz_table.size()];
		for(int i = 0; i < (int) rz_table.size(); i++) {
			previous_implementation_on_RZ[i] = TaskImplementation();
			current_implementation_on_RZ[i] = TaskImplementation();
		}

		// Set Nominal Point for RZ
		if (Scheduler_exist_rz_properties("RZDefaultGroupName") == true) {
			RZ_config rz_config = Scheduler_get_rz_properties("RZDefaultGroupName");
			Config_fct_point hwRZdefaultPt;
			string hwRZdomainName = "";
			if (rz_config.getConfigDomain() != NULL) {
				hwRZdefaultPt = rz_config.getConfigDomain()->getNominalFctPoint();
				hwRZdomainName =  rz_config.getConfigDomain()->getGroupName();
			}

			for(int i = 0; i < (int) rz_table.size(); i++) {
				RZ *rz = rz_table.at(i);
				if (rz->get_implementation_type() == HARD) {
					rz->setConfigFctPoint(&hwRZdefaultPt);
					rz->setGroupDomainName(hwRZdomainName);
				}
				else {
					rz_config = Scheduler_get_rz_properties(rz);
					if (rz_config.getConfigDomain() != NULL) {
						Config_fct_point swRZdefaultPt = rz_config.getConfigDomain()->getNominalFctPoint();
						rz->setConfigFctPoint(&swRZdefaultPt);
						rz->setGroupDomainName(rz_config.getConfigDomain()->getGroupName());
					}
				}
			}
		}

		hw_reconfiguration_units = new ReconfigurationUnit[NB_HW_RECONFIGURATION_UNITS];
		sw_reconfiguration_units = new ReconfigurationUnit[NB_SW_RECONFIGURATION_UNITS];

		hw_reconfiguration_event_table = new sc_event[NB_HW_RECONFIGURATION_UNITS];
		sw_reconfiguration_event_table = new sc_event[NB_SW_RECONFIGURATION_UNITS];

		SC_THREAD(scheduler_thread);				// Scheduler process
		SC_THREAD(occupation_rate_update_thread);	// Occupation rate automatic update thread

		// Register transport methods
		module_target_socket.register_b_transport(this, &Reconfiguration_manager_base::b_transport_module);
		testbench_target_socket.register_b_transport(this, &Reconfiguration_manager_base::b_transport_testbench);

		// HW reconfiguration engines dynamic threads
		for(int i = 0; i < NB_HW_RECONFIGURATION_UNITS; i++) {
			string process_name("HW_reconfiguration_unit_");
			ostringstream out;
			out << i;
			process_name.append(out.str());

			sc_spawn(sc_bind(&Reconfiguration_manager_base::hardware_reconfiguration_engine_thread, this, i), process_name.c_str());
		}

		// SW reconfiguration engines dynamic threads
		for(int i = 0; i < NB_SW_RECONFIGURATION_UNITS; i++) {
			string process_name("SW_reconfiguration_unit_");
			ostringstream out;
			out << i;
			process_name.append(out.str());

			sc_spawn(sc_bind(&Reconfiguration_manager_base::software_reconfiguration_engine_thread, this, i), process_name.c_str());
		}

		rz_configuration_request_buffer = new sc_buffer<ConfigurationRequest>[(int) rz_table.size()];
		rz_configured_event = new sc_event[(int) rz_table.size()];
		for(int i = 0; i < (int) rz_table.size(); i++) {
			string process_name("RZ_configuration_request_handler_");
			process_name.append(Utils::itoa(i));
			sc_spawn(sc_bind(&Reconfiguration_manager_base::rz_configuration_request_thread, this, i), process_name.c_str());
		}
	}

	~Reconfiguration_manager_base() {
		delete trace_packet;
		delete deadlines_table;
		delete[] previous_implementation_on_RZ;
		delete[] current_implementation_on_RZ;
		delete[] hw_reconfiguration_units;
		delete[] sw_reconfiguration_units;
		delete[] hw_reconfiguration_event_table;
		delete[] sw_reconfiguration_event_table;
		//delete qos_management;

		for (int i=0; i<(int)monitoring_end_of_scheduling_event_list.size(); i++) {
			delete monitoring_end_of_scheduling_event_list.at(i);
			delete monitoring_start_of_scheduling_event_list.at(i);
			delete monitoring_update_rz_state_event_list.at(i);
			delete monitoring_update_task_state_event_list.at(i);
		}

		for(int i = 0; i < (int) rz_table.size(); i++) delete rz_table.at(i);
	}

	// Application definition
	int addApplication(string appName, string appPrefix, vector<Manager_interface *> table, double qos, int appInstanceID = -1);
	void applicationDefined(void);

	// TLM transport methods
	virtual void b_transport_module(int id, tlm_generic_payload& trans, sc_time& delay) = 0;
	virtual void b_transport_testbench(int id, tlm_generic_payload& trans, sc_time& delay) = 0;

	// User interface
	void set_scheduler_execution_time(sc_time t);
	void set_generate_zeros_in_csv(bool gen);
	void display_occupation_rates(void);
	void generateSimulationResultFile(void);
	void update_occupation_rate_end_simulation(void);
	void enable_debug_signals_trace(void);
	void activate_trace(void);
	void generateCSVResultFile(string filename);

	static vector<Application> * getApplicationVectorPtr(void);

	// Simulation controller part
	bool isSimulationOK(void);
	static void endApplication(string appname, bool ok);
	static void verifySimulationTimes(void);
	static sc_time getMaximumSimulationTime(void);

	// ===== Scheduler interface =====
	vector<RZ *> Scheduler_get_compatible_rz_vector(int task_id);
	sc_time Scheduler_get_task_running_time(int task_id);
	vector<RZ *> Scheduler_get_all_rz_vector(void);
	int Scheduler_get_current_module_ID(RZ* rz);
	bool Scheduler_is_RZ_blank(RZ *rz);
	int Scheduler_get_module_ID(SchedulerRequest&);
	int Scheduler_get_application_number(void);
	Application_interface* Scheduler_get_application_interface(int index);
	bool Scheduler_exist_rz_properties(string inst_name);
	bool Scheduler_exist_rz_properties(RZ *rz);
	RZ_config& Scheduler_get_rz_properties(string inst_name);
	RZ_config& Scheduler_get_rz_properties(RZ* rz);
	Config_fct_point& Scheduler_get_current_fct_point(RZ *rz);
	void Scheduler_emulate_scheduler_behavior(void);
	void Scheduler_set_scheduler_active(int taskid);
	bool Scheduler_has_task_already_requested_mapping(int task_to_map, int request_owner);
	void Scheduler_add_finished_precedence(int task_mapped, int precedence);
	void Scheduler_notify_request_owner_module_ready(int owner, int ready);
	void Scheduler_send_update_parameters_to_module(RZ* rz);
	void Scheduler_print_finished_precedence_vector(int task_id);
	void Scheduler_configure_task(RZ* rz, int id, int implID, int requestOwner);
	void Scheduler_rerun_task(SchedulerRequest&);
	void Scheduler_set_current_task(RZ *rz, int task_id);
	Task_state Scheduler_get_task_state(int task_id);
	bool Scheduler_set_blank(RZ *rz);
	bool Scheduler_set_blank(int hosting_rz_id);
	Task_state Scheduler_get_task_state(RZ *rz);
	bool Scheduler_is_task_preempted(int taskid) const;
	double Scheduler_get_performance_effort(void) const;
	double Scheduler_get_power_effort(void) const;
	double Scheduler_get_area_effort(void) const;
	const char* Scheduler_get_name(void) const;
	ostream& Scheduler_get_output_stream(void);
	void Scheduler_display_task_state_table(void) const;
	void Scheduler_display_rz_current_module_table(void) const;
	Task_to_schedule_interface* Scheduler_get_task_to_schedule_ptr(int taskID);

	static FPGA* getFPGAptr(void);
	//static bool isInterfaceDefined(string if_name);

	// Monitoring
	void setMonitoringEvent(int threadNumber);
	void notifyStartOfSchedulingEventMonitoringEvent();
	void notifyEndOfSchedulingEventMonitoringEvent();
	void notifyUpdateRZStateEvent();
	void notifyUpdateTaskStateEvent();
	RZ_config* getRZProperties(string inst_name);

	// Monitoring Interface
	sc_event& endOfSchedulingEvent(int id);
	sc_event& startOfSchedulingEvent(int id);
	sc_event& updateRZStateEvent(int id);
	sc_event& updateTaskStateEvent(int id);

	// QoS Interface
	vector<Application>& getApplications();

	// Common QoS & Monitoring Interface
	sc_trace_file* getTraceFile() const;
	Scheduler_interface& getShedulerInterface();


protected:
	// Communication with submodules / scheduling
	void scheduler_thread(void);
	void send_next_module_ready(int module_id, int module_ready_id);
	void send_current_module_ready(int module_id);
	void send_module_ready_to_testbench(int tbid, int module_ready_id);
	void notify_request_owner_module_ready(int owner_id, int ready_module_id);
	void send_update_parameters(int module_id, Algorithm_Parameters* rz);

	// Initialization
	void init_maps(void);
	void transcoder_initialisation_method(int id, sc_dt::uint64 data);

	// Reconfiguration engine
	void configure_task(int id, int implID, RZ* rz);
	void reconfiguration_engine_thread(void);
	void hardware_reconfiguration_engine_thread(int unitID);
	void software_reconfiguration_engine_thread(int unitID);

	// Simulation control
	void check_deadline(int task_id);
	void notify_simulation_controller(void);

	// Preemption stuff
	bool is_task_preempted(int id) const;
	bool is_task_preempted(string taskname) const;
	void add_task_to_preemption_list(int id);
	void remove_task_from_preemption_list(int id);

	// Trace packet & monitoring methods
	void set_target_phase(tlm_phase phase);
	void set_icap_reconf_task(int hw_unit, int id);
	void set_icap_idle(int hw_unit);
	void set_software_loader_config_task(int sw_unit, int id);
	void set_software_loader_idle(int sw_unit);
	void set_task_state(int id, int rz_id, Task_state state);
	void set_rz_state(int rz_id, RZState state);
	int get_hosting_rz_id(int taskid);
	void update_occupation_rate(int rz_id, RZState newState);
	void set_initiator_phase(int id, tlm_phase phase);
	void set_scheduler_idle(void);
	void set_scheduler_active(int taskID);
	void occupation_rate_update_thread(void);
	void update_resources_trace(int rz_id, int task_id);
	void update_processor_occupation_rate(void);
	void set_hyperperiod_status(int hpid);

	// Misc
	void display_map_compatible_RZ(void);
	void display_map_current_module(void);
	void display_task_state_table(void) const;
	void display_rz_current_module_table(void) const;
	bool task_runs_on_implementation_type(int taskID, PEImplementation impl);
	int getApplicationIncludingTask(int taskID);
	void deadline_check_thread(int appID);

	int getRZID(RZ *rz);
	void set_task_context_save(int task_id, int rz_id);
	void set_task_context_restore(int task_id, int rz_id);

	int get_module_ID(string moduleName);

	void update_deadline(int modID);

	void rz_configuration_request_thread(int rzid);
	
};

#endif
