/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    reconfiguration_manager.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-07-01
 * @section DESCRIPTION
 *			Implementation file for the reconfiguration manager
 */

#include "reconfiguration_manager_base.h"
#include "reconfiguration_manager.h"
#include <time.h>
#include "algorithm_mode_command.h"

#define FOUT if(generate_logfile()) (*fout)

void Reconfiguration_manager::display_waiting_queue(void) const {

	FOUT << sc_time_stamp() << ": " << name() << ": Tasks waiting in the queue to be executed (size: " << waiting_queue.size() << ")" << endl;

	//REM: Permet de savoir qui a dclench la demande de placement de la tche
	FOUT << sc_time_stamp() << ": Waiting queue size: " << dec << waiting_queue.size();
	for(int i = 0; i < waiting_queue.size(); i++) {
		FOUT << " - " << waiting_queue.at(i).getName() << " (" << waiting_queue.at(i).get_request_owner() << ", " << waiting_queue.at(i).get_deadline_time() << ")";
	}
	FOUT << endl;
}

/**
 * Blocking transport function, called by any module identified by its id.
 * The function checks whether it is a 'End of computation' request or a
 * 'Preemption point found' request (not handled yet). The function then puts
 * into the waiting queue every following module, known using the corresponding
 * list accessible via the Manager interface.
 */
//REM: fonction bloquante (mais pas de wait  l'intrieur donc finalement non bloquante)
//REM: appel d'un module
//REM: id : id du socket qui est d'ailleurs le mme que l'id de la table des modules_table (voir dans recosim_main.cpp)
void Reconfiguration_manager::b_transport_module(int id, tlm_generic_payload& trans, sc_time& delay) {

	// Prepare transaction for response
	trans.set_response_status(TLM_OK_RESPONSE);
	int *data = reinterpret_cast<int *>(trans.get_data_ptr());

	// Update scheduling
	if(trans.get_address() == MANAGER_UPLINK_TASK_END_COMPUTATION) {

		// Task id is no longer running. Care for static modules
		if(!modules_table[id]->is_static()) set_task_state(id, get_hosting_rz_id(id), MAPPED);

		// Check if time constraints are met
		check_deadline(id);

		// Update finished precedence list
		//REM: remise  zero de la liste des modules qui ont demand l'instanciation d'un module car end_computation
		if(!ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) finished_precedence_vectors.at(id).clear();

		// Check if there are some tasks to execute in the waiting queue
		//REM: normalement ici ??
		scheduler_thread_event.notify(SC_ZERO_TIME);
		
	} else if(trans.get_address() == MANAGER_UPLINK_TASK_CHECK_SOCKET) {
		//REM: demande d'instantiation d'un module suivant

		// Check for a particular module
		//REM: nid = id du module  instancier
		int nid = *reinterpret_cast<int*>(trans.get_data_ptr());

		if(nid == -1) {
			//REM: c'est le testbench qui est obligatoirement statique donc on peut envoyer directement les donnes

			// Last module (connected to a TB)
			// Nothing to do, assuming testbench is always ready to receive stuff
			FOUT << sc_time_stamp() << ": " << name() << " Last module reached" << endl;
			*data = (int) NEXT_MODULE_READY;
		} else {

			//REM: utilise pour la suite, voir question_manager_b_transport dans module.h
			// By default, wait for next module
			*data = (int) WAIT_FOR_NEXT_MODULE;
					
			// Update deadline
			//REM: mise  jour de la deadline absolue
			update_deadline(nid);

			// Verify if this following module is static
			if(modules_table[nid]->is_static()) {
				//REM: met le module suivant en file d'attente

				FOUT << sc_time_stamp() << ": " << name() << ": task " << nid << " (" << modules_table[nid]->getName() << ") is static" << endl;

				modules_table[nid]->set_current_implementation(modules_table.at(nid)->get_implementation_vector().at(0).get_name());

				// Notify module
				send_next_module_ready(id, nid);

			} else {
				
				if(ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) {

					//cout << sc_time_stamp() << " Looking for owner " << id << " for module " << nid << endl;
					if(!Scheduler_has_task_already_requested_mapping(nid, id)) {
						//cout << "Not found" << endl;
						Scheduler_add_finished_precedence(nid, id);
					} // ERR: not working when two requests come from the same module (the second one is lost)

					//cout << "Finished precedence size " << finished_precedence_vectors.at(nid).size() << " (" << modules_table.at(nid)->getPrecedingList().size() << ")" << endl;
					
					// QUESTION: POURQUOI 2 ?, code inutile ?
					if(finished_precedence_vectors.at(nid).size() == 2) {
						//cerr << "ERROR!!!!!!" << endl;
						//cerr << "Finished precedence vector: " << finished_precedence_vectors.at(nid).at(0) << ", " << finished_precedence_vectors.at(nid).at(1) << endl;
					}

					if(finished_precedence_vectors.at(nid).size() == modules_table.at(nid)->getPrecedingList().size() ||
						(modules_table.at(nid)->get_nb_processed_transactions() == 0 && finished_precedence_vectors.at(nid).size() == (modules_table.at(nid)->getPrecedingList().size() - modules_table.at(nid)->get_nb_transient_channels()))) {

						// Put task in waiting queue
						FOUT << sc_time_stamp() << ": " << name() << ": pushing element " << dec << nid << " (" << modules_table[nid]->getName() << ")" << " into queue. Deadline " << modules_table[nid]->get_deadline_time() << endl;
						waiting_queue.push(SchedulerRequest(modules_table[nid], id));

						//finished_precedence_vectors.at(nid).clear();

						// Check waiting queue
						scheduler_thread_event.notify(SC_ZERO_TIME);
					}
				} else {
					// Put task in waiting queue
					FOUT << sc_time_stamp() << ": " << name() << ": pushing element " << dec << nid << " (" << modules_table[nid]->getName() << ")" << " into queue. Deadline " << modules_table[nid]->get_deadline_time() << endl;
					waiting_queue.push(SchedulerRequest(modules_table[nid], id));

					// Check waiting queue
					scheduler_thread_event.notify(SC_ZERO_TIME);
				}
			}
		}

	} else if(trans.get_address() == MANAGER_UPLINK_TASK_PREEMPTION) {
		
		// Preemption
		FOUT << sc_time_stamp() << ": " << name() << " preemption point reached for task " << id << " (" << modules_table[id]->getName() << ")" << endl;
		FOUT << sc_time_stamp() << ": " << name() << " New deadline: " << modules_table[id]->get_deadline_time() << endl;

		//REM: on arrte la tche (tat MAPPED) et on l'ajoute  la file d'attente et on appelle le scheduler

		add_task_to_preemption_list(id);

		// Task id is no longer running
		//set_task_state(id, get_hosting_rz_id(id), MAPPED);
		set_task_state(id, get_hosting_rz_id(id), PREEMPTED_MAPPED);

		// By default, wait for current module
		//REM: on considre que le module n'est pas prt, c'est le scheduler qui dcidera.
		*data = (int) WAIT_FOR_CURRENT_MODULE;

		// Put task in waiting queue
		FOUT << sc_time_stamp() << ": " << name() << ": pushing element " << dec << id << " (" << modules_table[id]->getName() << ")" << " into queue. Deadline " << modules_table[id]->get_deadline_time() << endl;
		waiting_queue.push(SchedulerRequest(modules_table[id], id));
		//finished_precedence_vectors.at(id).push_back(id);

		// Check waiting queue
		scheduler_thread_event.notify(SC_ZERO_TIME);

	} else if(trans.get_address() == MANAGER_UPLINK_ALGO_STATE_UPDATE) {
		
		Task_state newState = (Task_state) *reinterpret_cast<int*>(trans.get_data_ptr());
		FOUT << sc_time_stamp() << ": " << name() << ": New algorithm state received for task " << id << " (" << modules_table[id]->getName() << ")" << endl;
		FOUT << sc_time_stamp() << ": " << name() << ": New state " << Task_state_string[(int) newState] << endl;

		/* Update Trace */
		set_task_state(id, get_hosting_rz_id(id), newState);

		if(SEND_CONFIGURATION_NOTIFICATION_AFTER_IDLE_STATE && !ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST&& newState == WAITING) {
			// Notify previous modules
			for(int i = 0; i < (int) finished_precedence_vectors[id].size(); i++) notify_request_owner_module_ready(finished_precedence_vectors[id].at(i), id);
		}

	} else if(trans.get_address() == MANAGER_UPLINK_UPDATE_PARAMETER_COMMAND_SCHEDULING) {
		//toto
		//cout << sc_time_stamp() << "Send RZ parameters ..." << endl;

		int rz_id = get_hosting_rz_id(id);
		RZ *rz = rz_table.at(rz_id);
		Scheduler_send_update_parameters_to_module(rz);

	} else {
		cerr << "Manager does not recognize the address coming from module " << id << endl;
		SC_REPORT_FATAL("TLM-2", "Manager does not recognize the address during uplink communication");
	}

	// Wait, but delay should be 0
	wait(delay);
}

/**
 * Blocking transport function, called by the testbench. The testbench has to
 * check the availability of the entry point it wants to use.
 * The function puts the entry point into the waiting queue for instantiation.
 */
void Reconfiguration_manager::b_transport_testbench(int id, tlm_generic_payload& trans, sc_time& delay) {

	// Update scheduling
	if(trans.get_address() == TB_TO_MANAGER_CHECK_SOCKET) {

		int nid = *reinterpret_cast<int*>(trans.get_data_ptr());

		// Update deadline
		update_deadline(nid);

		// Verify if this following module is static
		if(modules_table[nid]->is_static()) {

			FOUT << sc_time_stamp() << ": " << name() << ": task " << nid << " (" << modules_table[nid]->getName() << ") is static" << endl;

			/* Update Implementation */
			modules_table[nid]->set_current_implementation(modules_table.at(nid)->get_implementation_vector().at(0).get_name());
			
			// Notify testbench
			send_module_ready_to_testbench(id, nid);

		} else {
				
			if(ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) {
				if(!Scheduler_has_task_already_requested_mapping(nid, (id + 1) * -1)) Scheduler_add_finished_precedence(nid, (id + 1) * -1);

				if(finished_precedence_vectors.at(nid).size() == modules_table.at(nid)->getPrecedingList().size() ||
						(modules_table.at(nid)->get_nb_processed_transactions() == 0 && finished_precedence_vectors.at(nid).size() == (modules_table.at(nid)->getPrecedingList().size() - modules_table.at(nid)->get_nb_transient_channels()))) {

					// Put task in waiting queue
					FOUT << sc_time_stamp() << ": " << name() << ": pushing element " << dec << nid << " (" << modules_table[nid]->getName() << ")" << " into queue. Deadline " << modules_table[nid]->get_deadline_time() << endl;

					waiting_queue.push(SchedulerRequest(modules_table[nid], (id + 1) * -1));

					//finished_precedence_vectors.at(nid).clear();
				}
			} else {
				// Put task in waiting queue
				FOUT << sc_time_stamp() << ": " << name() << ": pushing element " << dec << nid << " (" << modules_table[nid]->getName() << ")" << " into queue. Deadline " << modules_table[nid]->get_deadline_time() << endl;

				waiting_queue.push(SchedulerRequest(modules_table[nid], (id + 1) * -1));

			}
		}

		// Check waiting queue
		scheduler_thread_event.notify(SC_ZERO_TIME);

		// Update response
		trans.set_response_status(TLM_OK_RESPONSE);

	} else if(trans.get_address() == TB_TO_MANAGER_ALGORITHM_COMMAND) {

		//REM: possiblit de changer de mode d'excution de l'algo en cours de simulation
		AlgorithmModeCommand *data = reinterpret_cast<AlgorithmModeCommand *>(trans.get_data_ptr());
		
		//REM: fabrication du nom complet du module avec prefix, app ...
		string moduleName(applicationVector.at(id).getFullModuleName(data->get_target_module()));
		int moduleID(get_module_ID(moduleName));

		FOUT << sc_time_stamp() << ": " << name() << ": New command received: " << data->get_command() << " for module " << moduleName << " (" << moduleID << ") from testbench " << id << endl;

		//REM: met juste  jour le champ algorithm_execution_mode de module.h et la trace
		modules_table.at(moduleID)->update_algorithm_execution_mode(data->get_command());

		// Update response
		trans.set_response_status(TLM_OK_RESPONSE);

	} else {
		SC_REPORT_FATAL("TLM-2", "Manager does not recognize the address during uplink communication with TB");
	}
}

/*********************************
 *      Scheduler interface      *
 *********************************/
 
bool Reconfiguration_manager::Scheduler_are_tasks_waiting(void) const {
	return !waiting_queue.empty();
}

/*
SchedulerRequest Reconfiguration_manager::Scheduler_get_top_request(void) {
	return waiting_queue.top();
}
*/

/*
void Reconfiguration_manager::Scheduler_pop_request(void) {
	waiting_queue.pop();
}
*/

void Reconfiguration_manager::Scheduler_display_waiting_queue(void) const {
	display_waiting_queue();
}

int Reconfiguration_manager::Scheduler_get_waiting_queue_size(void) {
	return waiting_queue.size();
}

int Reconfiguration_manager::Scheduler_position_of_current_element(void) {
	return waiting_queue.current_position();
}

SchedulerRequest& Reconfiguration_manager::Scheduler_get_element(int position) {
	return waiting_queue.at(position);
}

void Reconfiguration_manager::Scheduler_update_queue(void) {
	waiting_queue.update_queue();
}

bool Reconfiguration_manager::Scheduler_last_task_waiting(void) const {
	return waiting_queue.last_element();

}

void Reconfiguration_manager::Scheduler_erase_current_task_waiting() {
	waiting_queue.erase_current_element();
}

SchedulerRequest& Reconfiguration_manager::Scheduler_current_task_waiting() {
	return waiting_queue.current_element();
}

SchedulerRequest Reconfiguration_manager::Scheduler_next_task_waiting() {
	return waiting_queue.next_element();
}

void Reconfiguration_manager::Scheduler_reset_current_position_in_waiting_queue() {
	waiting_queue.resetCurrentPosition();
}
