/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    module_implementation.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-06-01
 * @section DESCRIPTION
 *			Defines a the algorithm type of implementation, either
 *			hard or soft. There might be more than one hard/soft implementation
 *			for a single module
 */

#ifndef MODULE_IMPLEMENTATION_H
#define MODULE_IMPLEMENTATION_H

#include <systemc.h>
#include <string>
#include <set>
#include <map>
#include "pe_implementation.h"
#include "user_algorithm_interface.h"

using namespace std;

typedef void (*algorithm_thread_type)(void *);

struct ModuleImplementation {

	string name;
	string netlist_name;
	string algorithm_name;
	PEImplementation implementation_type;
	sc_time worst_case_execution_time;
	sc_time best_case_execution_time;
	
	double energy_consumption;	//  virer
	bool enable_implementation;
	int nb_preemption_points;
	bool use_context_switch;

	// Only HwImpl
	double frequency;
	double p_idle;
	double p_run;


	map<string, int> physical_channels;

	// Pointer to the thread associated with the implementation
	algorithm_thread_type algorithm_thread_ptr;

	ModuleImplementation(string _name, PEImplementation impl, algorithm_thread_type algo, sc_time wcet, sc_time bcet = SC_ZERO_TIME, double freq = 0, double pidle = 0, double prun = 0) 
		: name(_name), implementation_type(impl), algorithm_thread_ptr(algo), worst_case_execution_time(wcet), best_case_execution_time(bcet), frequency(freq), p_idle(pidle), p_run(prun) {
		// Enable implementation by default
		enable_implementation = true;
		if(bcet == SC_ZERO_TIME) best_case_execution_time = worst_case_execution_time;
		nb_preemption_points = 0;
		use_context_switch = false;
		energy_consumption = 0;
	}

	ModuleImplementation(string _name, PEImplementation impl, algorithm_thread_type algo) : name(_name), implementation_type(impl), algorithm_thread_ptr(algo) {
		// Enable implementation by default
		enable_implementation = true;
		best_case_execution_time = SC_ZERO_TIME;
		worst_case_execution_time = SC_ZERO_TIME;
		frequency = 0;
		energy_consumption = 0;
		nb_preemption_points = 0;
		use_context_switch = false;
		p_idle = 0;
		p_run = 0;
	}

	string get_name(void);
	PEImplementation get_implementation_type(void);
	sc_time get_worst_case_execution_time(void) const;
	sc_time get_best_case_execution_time(void) const;
	double get_frequency(void) const;
	//double get_energy_consumption(void) const;
	double get_p_idle(void) const;
	double get_p_run(void) const;
	string get_netlist_name(void) const;
	string get_algorithm_name(void) const;

	void activate_implementation(void);
	void deactivate_implementation(void);
	bool is_implementation_enabled(void) const;

	void info(ostream &st) const;

	void add_physical_channels(string connection, int n = 1);
	map<string, int> get_physical_channel_map(void);

	// Setters
	void set_worst_case_execution_time(sc_time t);
	void set_best_case_execution_time(sc_time t);
	void set_frequency(double f);
	//void set_energy_consumption(double e);
	void set_p_idle(double e);
	void set_p_run(double e);
	void set_netlist_name(string n);
	void set_algorithm_name(string n);

	algorithm_thread_type get_algorithm_thread_ptr(void);

	void set_nb_preemption_points(int n);
	int get_nb_preemption_points(void);

	void set_use_context_switch_mode(bool);
	bool use_context_switch_mode(void);

};

#endif
