/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    manager_interface.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-11-22
 * @section DESCRIPTION
 *			Interface to be passed as a parameter for the reconfiguration manager
 */

#ifndef MANAGER_INTERFACE_H
#define MANAGER_INTERFACE_H

#ifdef MIDDLEWARE_GEN
	#include <sys/types.h>
#endif

#include <vector>
#include "module_implementation.h"
#include "task_to_schedule_interface.h"

class Manager_interface : public Task_to_schedule_interface {

public:

	//
	// ======= Computation time =======
	//

	/**
	 * @brief Get task implementations (HW and SW implementations are merged)
	 * @return Vector containing module implementation (class ModuleImplementation)
	 */
	virtual vector<ModuleImplementation> get_implementation_vector(void) const = 0;

	/**
	 * @brief Set current implementation for the module, based on its name ('implementationName'). 
	 * @param implementationName Current implementation name
	 */
	virtual void set_current_implementation(string implementationName) = 0;

	/**
	 * @brief Get current implementation ID
	 * @return Current implementation id
	 */
	virtual string get_current_implementation_fullname(void) = 0;

	/**
	 * @brief Get current implementation
	 * @return Current implementation
	 */
	virtual ModuleImplementation get_current_implementation(void) = 0;

	/**
	 * @brief Check if exist a current implementation
	 * @return return true if exists
	 */
	virtual bool has_current_implementation(void) = 0;

	//
	// ======= Task properties =======
	//

	/**
	 * @brief Check if the module has been defined static (i.e. always implemented
	 * on the FPGA)
	 * @return true if module is static
	 */
	virtual bool is_static(void) const = 0;
	

	//
	// ======= Misc =======
	//

	/**
	 * @brief Set the time at which a configuration request has been made in order to update deadlines
	 */
	virtual void set_request_time(void) = 0;
	
	/**
	 * @brief Get the number of transactions processed so far
	 * @return Number of processed transactions
	 */
	virtual int get_nb_processed_transactions(void) const = 0;

	/**
	 * @brief Update lists after the entire application has been defined.
	 * Called once by the reconfiguration manager.
	 * @param offset Offset to apply to IDs within lists
	 */
	virtual void updateLists(int offset) = 0;

	/**
	 * @brief Transmit algorithm execution mode from issued by the testbench
	 * to the module (e.g. change algorithm implementation to 'VHDL'
	 * @param mode String representation of the execution mode
	 */
	virtual void update_algorithm_execution_mode(string mode) = 0;

	/**
	 * @brief Check if context switch mode is enabled for the implementation
	 * named 'implementationName'. In this mode, specific to HW implementations,
	 * a context switch is performed each time the task is being replaced on a 
	 * reconfigurable zone (not only when the task is being preempted).
	 * @param implementationName Implementation name
	 * @return true if context switch mode is enabled
	 */
	virtual bool use_context_switch_mode(string implementationName) = 0;

	/**
	 * @brief Send a notification to the module in order to start the user algorithm.
	 */
	virtual void send_start_algorithm_event(void) = 0;

	/**
	 * @brief Get number of transient channels within the module
	 * @return Number of transient channels
	 */
	virtual int get_nb_transient_channels(void) = 0;
	
};

#endif
