/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    hardwareTask.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-08-12
 * @section DESCRIPTION
 *			Representation of a synthesized task
 */

#ifndef HARDWARE_TASK_H
#define HARDWARE_TASK_H

#include <iostream>
#include <string>
#include <vector>
#include <set>
using namespace std;

#include "task.h"
#include "RZ.h"
#include "device.h"
#include "synthesisReport.h"
#include "resourceRequirements.h"
#include "RZFinderManager.h"
#include "coordinates.h"
#include "interface_location.h"
#include "XMLTaskDescription.h"


class HwTask : public Task {
	
protected:
	ResourceRequirements resourceRequirements;
	vector<Coordinates> requiredColumns;
	vector<InterfaceLocation> possibleInterfaceLocations;

private:
	string netlistName;					/*< Netlist name, may be different from task name */
	string resourceFileName;			/*< Resource file name, may be different from task name */
	TaskResourceFile* resourceFile;		/*< Resource file linked to the task */
	double contextSwitchTime;			/*< Time in microseconds */

public:
	/* Constructor */
	HwTask(string n, string netlist, string synth, double contextSwitch) : Task(n), netlistName(netlist), resourceFileName(synth), contextSwitchTime(contextSwitch) {
		HwTaskInit();
	}

	HwTask(string n, string netlist, string synth, double contextSwitch, bool static_mod) : Task(n, static_mod), netlistName(netlist), resourceFileName(synth), contextSwitchTime(contextSwitch) {
		HwTaskInit();
	}

	HwTask(){
		HwTaskInit();
	}

	HwTask(string n, string netlist, string synth, double contextSwitch, vector<Coordinates> vec) : Task(n), netlistName(netlist), resourceFileName(synth), requiredColumns(vec), contextSwitchTime(contextSwitch){
		HwTaskInit();
	}

	HwTask(string n, string netlist, string synth, double contextSwitch, bool static_mod, vector<Coordinates> vec) : Task(n, static_mod), netlistName(netlist), resourceFileName(synth), requiredColumns(vec), contextSwitchTime(contextSwitch){
		HwTaskInit();
	}

	HwTask(const HwTask &other) : Task(other) {
		resourceRequirements = other.resourceRequirements;
		requiredColumns = other.requiredColumns;
		possibleInterfaceLocations = other.possibleInterfaceLocations;
		netlistName = other.netlistName;
		resourceFileName = other.resourceFileName;
		contextSwitchTime = other.contextSwitchTime;
		if(other.resourceFile == 0) resourceFile = 0;
		else HwTaskInit(other.resourceFile);
	}

	~HwTask() {
		if(resourceFile != 0) delete resourceFile;
		//if(resourceRequirements) delete resourceRequirements;
	}

	/* Getters */
	string getNetlistName(void);
	string getDeviceID(void);
	DeviceClass getDeviceClass(void);
	Device* getDevicePtr(void);
	vector<RZ> getRZSet(void);
	vector<RZ>* getRZSetPtr(void);
	ResourceRequirements * getResourceRequirementsPtr(void);
	ResourceRequirements& getResourceRequirements(void);
	TaskResourceFile* getTaskResourceFile(void);

	/* Misc */
	void info(void);
	void findRZs(void);
	void defineResourceRequirements(void);
	bool fitsRZ(RZ *rz);
	int computeInternalFragmentation(RZ &rz);
	int getTaskCost(void);
	int readResourceFile(void);
	bool hasPlacementRequirements(void);
	void setResourceMargin(double v);
	void setLOCConstraint(vector<Coordinates> &vec);
	void setPossibleInterfaceLocations(vector<InterfaceLocation> &vec);

	bool isInterfaceDefined(string interfaceType);

	void HwTaskInit(void);
	void HwTaskInit(TaskResourceFile* res);

	double getContextSwitchTime(void);

};

#endif
