/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    RZSetDescriptionFile.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-12-18
 * @section DESCRIPTION
 *			RZ set description file
 */

#include "RZSetDescriptionFile.h"

#include <xercesc/sax/HandlerBase.hpp>
#include <xercesc/framework/LocalFileFormatTarget.hpp>

#include <string>
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdexcept>
#include <list>

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include "utils.h"


// Helper function
ResourceConstraint convertStringToResourceConstraint(string str) {
	if(str.compare("UNUSED") == 0) return UNUSED;
	if(str.compare("USED") == 0) return USED;
	if(str.compare("STATIC") == 0) return STATIC;
	if(str.compare("NOT_RECONFIGURABLE") == 0) return NOT_RECONFIGURABLE;

	cerr << "ERROR: Unable to convert string to ResourceConstraint! Defaulting to UNUSED" << endl;
	return UNUSED;
}

string convertResourceConstraintToString(ResourceConstraint constraint) {

	switch(constraint) {
		case UNUSED: return "UNUSED";
		case USED: return "USED";
		case STATIC: return "STATIC";
		case NOT_RECONFIGURABLE: return "NOT_RECONFIGURABLE";
		default: 
			cerr << "ERROR: Unrecognized ResourceConstraint while writing XML file" << endl;
			return "NULL";
	}
}


int RZSetDescriptionFile::readConfigFile(Device *device, vector<RZ> &RZset) {

	// Try opening the file read-only to check existence
	ifstream file(filename.c_str());
	if(!file) {
		cerr << "ERROR: XML file does not exist!" << endl;
		return -1;
	}

	// Configure DOM parser
	m_ConfigFileParser->setValidationScheme(XercesDOMParser::Val_Never);
	m_ConfigFileParser->setDoNamespaces(false);
	m_ConfigFileParser->setDoSchema(false);
	m_ConfigFileParser->setLoadExternalDTD(false);

	try {
		m_ConfigFileParser->parse(filename.c_str());

		// no need to free this pointer - owned by the parent parser object
		DOMDocument* xmlDoc = m_ConfigFileParser->getDocument();

		// Get the top-level element: Name is "root". No attributes for "root"
      
		DOMElement* elementRoot = xmlDoc->getDocumentElement();
		if( !elementRoot ) throw(std::runtime_error( "empty XML document" ));


		DOMNodeList*      children = elementRoot->getChildNodes();
		const  XMLSize_t nodeCount = children->getLength();

		// For all nodes, children of "root" in the XML tree.

		for( XMLSize_t xx = 0; xx < nodeCount; ++xx ) {
			DOMNode* currentNode = children->item(xx);
			if( currentNode->getNodeType() &&  // true is not NULL
				currentNode->getNodeType() == DOMNode::ELEMENT_NODE ) // is element 
			{
				// Found node which is an Element. Re-cast node as element
				DOMElement* currentElement = dynamic_cast< xercesc::DOMElement* >( currentNode );

				if( XMLString::equals(currentElement->getTagName(), TAG_RZ_description)) readRZDescription(xmlDoc, currentElement, device, RZset);
			}
		}
	} catch( xercesc::XMLException& e ) {
		char* message = xercesc::XMLString::transcode( e.getMessage() );
		ostringstream errBuf;
		errBuf << "Error parsing file: " << message << flush;
		XMLString::release( &message );
		return -1;
	}

	return 0;
}

void RZSetDescriptionFile::readRZDescription(DOMDocument* xmlDoc, DOMElement* elt, Device *device, vector<RZ> &RZset) {

	// Retrieving RZ attributes
	const XMLCh* xmlch_rzid = elt->getAttribute(ATTR_RZ_ID);
	char *rzid_cstr = XMLString::transcode(xmlch_rzid);

	const XMLCh* xmlch_column = elt->getAttribute(ATTR_Column_number);
	char *rzcolumn_cstr = XMLString::transcode(xmlch_column);
	int rz_column = Utils::atoi(rzcolumn_cstr);

	const XMLCh* xmlch_line = elt->getAttribute(ATTR_Line_number);
	char *rzline_cstr = XMLString::transcode(xmlch_line);
	int rz_line = Utils::atoi(rzline_cstr);

	const XMLCh* xmlch_height = elt->getAttribute(ATTR_Height);
	char *rzheight_cstr = XMLString::transcode(xmlch_height);
	int rz_height = Utils::atoi(rzheight_cstr);

	const XMLCh* xmlch_reverse = elt->getAttribute(ATTR_Reverse);
	char *rzreverse_cstr = XMLString::transcode(xmlch_reverse);
	string reverse_str(rzreverse_cstr);
	bool rz_reverse = (reverse_str.compare("true") == 0 ? true : false);	// Defaulting to false

	// Creating RZ
	RZ currentRZ(rzid_cstr, rz_column, rz_line, rz_height, device, rz_reverse);

	DOMNodeList*      children = elt->getChildNodes();
	const  XMLSize_t nodeCount = children->getLength();


	vector<vector<pair<RBType, ResourceConstraint> > > resourcesArray;

	for( XMLSize_t xx = 0; xx < nodeCount; ++xx ) {
		DOMNode* currentNode = children->item(xx);
		if( currentNode->getNodeType() &&  // true is not NULL
			currentNode->getNodeType() == DOMNode::ELEMENT_NODE ) // is element 
		{
			// Found node which is an Element. Re-cast node as element
			DOMElement* currentElement = dynamic_cast< xercesc::DOMElement* >( currentNode );

			if( XMLString::equals(currentElement->getTagName(), TAG_Column)) {

				const XMLCh* xmlch_ALL = currentElement->getAttribute(ATTR_Column_all);
				char *all_cstr = XMLString::transcode(xmlch_ALL);
				string str_ALL(all_cstr);
				XMLString::release(&all_cstr);

				int repeat_value;
				const XMLCh* xmlch_repeat = currentElement->getAttribute(ATTR_Column_repeat);
				char *repeat_cstr = XMLString::transcode(xmlch_repeat);
				string str_REPEAT(repeat_cstr);
				XMLString::release(&repeat_cstr);

				if(str_REPEAT.empty()) repeat_value = 1;
				else repeat_value = Utils::atoi(str_REPEAT);

				for(int col = 0; col < repeat_value; col++) {
					vector<pair<RBType, ResourceConstraint> > current_column;

					if(!str_ALL.empty()) {

						for(int i = 0; i < rz_height; i++) {
							current_column.push_back(pair<RBType, ResourceConstraint>(device->getRB(rz_line + i, rz_column + (int) resourcesArray.size()), convertStringToResourceConstraint(str_ALL)));
						}

					} else {

						// Dynamically search for LX attributes
						for(int i = 0; i < rz_height; i++) {

							string lineX("L");
							lineX.append(Utils::itoa(i));

							XMLCh* ATTR_LX = XMLString::transcode(lineX.c_str());
							const XMLCh* xmlch_LX = currentElement->getAttribute(ATTR_LX);
							char *lx_value_cstr = XMLString::transcode(xmlch_LX);
							string str_LX(lx_value_cstr);

							current_column.push_back(pair<RBType, ResourceConstraint>(device->getRB(rz_line + i, rz_column + (int) resourcesArray.size()), convertStringToResourceConstraint(str_LX)));

							XMLString::release(&ATTR_LX);
							XMLString::release(&lx_value_cstr);
						}
					}

					resourcesArray.push_back(current_column);
				}
			}
		}
	}

	currentRZ.setResourcesArray(resourcesArray);
	currentRZ.updateConstrainedResources();
	currentRZ.info();
	currentRZ.printRZ();

	RZset.push_back(currentRZ);

	// Clean
	XMLString::release(&rzid_cstr);
	XMLString::release(&rzcolumn_cstr);
	XMLString::release(&rzline_cstr);
	XMLString::release(&rzheight_cstr);
	XMLString::release(&rzreverse_cstr);
}

void RZSetDescriptionFile::writeConfigFile(Device *device, vector<RZ> &vec) {

	// Create XML character pointers
	XMLCh *ioMode(XMLString::transcode("LS"));
	XMLCh *rootName(XMLString::transcode("RZSet"));
	XMLCh *newLine(XMLString::transcode("\r\n"));

	//Return the first registered implementation that has the desired features. In this case, we are after a DOM implementation that has the LS feature... or Load/Save. 
    DOMImplementation *implementation = DOMImplementationRegistry::getDOMImplementation(ioMode);
	DOMDocument* doc = implementation->createDocument(0, rootName, 0);
	createDOMDocument(doc, device, vec);

    // Create a DOMLSSerializer which is used to serialize a DOM tree into an XML document
    DOMLSSerializer *serializer = ((DOMImplementationLS*)implementation)->createLSSerializer();

    // Make the output more human readable by inserting line feeds
    if (serializer->getDomConfig()->canSetParameter(XMLUni::fgDOMWRTFormatPrettyPrint, true)) 
        serializer->getDomConfig()->setParameter(XMLUni::fgDOMWRTFormatPrettyPrint, true);

    // The end-of-line sequence of characters to be used in the XML being written out
    serializer->setNewLine(newLine);

    // Convert the path into Xerces compatible XMLCh*
	XMLCh *tempFilePath = XMLString::transcode(filename.c_str());

    // Specify the target for the XML output
    XMLFormatTarget *formatTarget = new LocalFileFormatTarget(tempFilePath);

    // Create a new empty output destination object
    DOMLSOutput *output = ((DOMImplementationLS*)implementation)->createLSOutput();

    // Set the stream to our target
    output->setByteStream(formatTarget);

    // Write the serialized output to the destination
    serializer->write(doc, output);

    // Cleanup
    serializer->release();
    XMLString::release(&tempFilePath);
    delete formatTarget;
    output->release();
	XMLString::release(&rootName);
	XMLString::release(&newLine);
	XMLString::release(&ioMode);
	XMLString::release(&tempFilePath);

}

void RZSetDescriptionFile::createDOMDocument(DOMDocument* doc, Device *device, vector<RZ> &vec) {

	try {

		XMLCh *devName(XMLString::transcode(device->getDeviceID().c_str()));
		XMLCh *devPackage(XMLString::transcode(device->getDevicePackage().c_str()));
		XMLCh *devSpeedGrade(XMLString::transcode(device->getDeviceSpeedGrade().c_str()));

		DOMElement* rootElem = doc->getDocumentElement();

		// Device Name
		DOMElement* deviceNameElt = doc->createElement(TAG_Device_name);
		deviceNameElt->setAttribute(ATTR_Value, devName);
		rootElem->appendChild(deviceNameElt);

		// Device Package
		DOMElement* devicePackageElt = doc->createElement(TAG_Device_package);
		DOMElement* DPElt = doc->createElement(TAG_DP);
		DPElt->setAttribute(ATTR_Value, devPackage);
		devicePackageElt->appendChild(DPElt);
		rootElem->appendChild(devicePackageElt);

		// Speed Grade
		DOMElement* speedGradeElt = doc->createElement(TAG_Speed_grade);
		DOMElement* SGElt = doc->createElement(TAG_SG);
		SGElt->setAttribute(ATTR_Value, devSpeedGrade);
		speedGradeElt->appendChild(SGElt);
		rootElem->appendChild(speedGradeElt);

		for(int i = 0; i < (int) vec.size(); i++) {

			XMLCh *rzid_xmlch(XMLString::transcode(vec.at(i).getID().c_str()));
			XMLCh *column_xmlch(XMLString::transcode(Utils::itoa(vec.at(i).getX0()).c_str()));
			XMLCh *line_xmlch(XMLString::transcode(Utils::itoa(vec.at(i).getY0()).c_str()));
			XMLCh *height_xmlch(XMLString::transcode(Utils::itoa(vec.at(i).getHeight()).c_str()));
			XMLCh *reverse_xmlch = vec.at(i).isReversed() ? XMLString::transcode("true") : XMLString::transcode("false");

			DOMElement* RZElement = doc->createElement(TAG_RZ_description);

			RZElement->setAttribute(ATTR_RZ_ID, rzid_xmlch);
			RZElement->setAttribute(ATTR_Column_number, column_xmlch);
			RZElement->setAttribute(ATTR_Line_number, line_xmlch);
			RZElement->setAttribute(ATTR_Height, height_xmlch);
			RZElement->setAttribute(ATTR_Reverse, reverse_xmlch);

			// Create Columns descriptions
			for(int j = 0; j < (int) vec.at(i).getResourcesArrayPtr()->size(); j++) {

				// Count the number of times the exact same column is being repeated
				int columnID = j;
				while((columnID < (int) vec.at(i).getResourcesArrayPtr()->size() - 1) && vec.at(i).areColumnsIdentical(columnID, columnID+1)) columnID++;
				int repeatValue = columnID - j + 1;

				DOMElement* CNElement = doc->createElement(TAG_Column);

				if(repeatValue != 1) {
					XMLCh *repeat_xmlch(XMLString::transcode(Utils::itoa(repeatValue).c_str()));
					CNElement->setAttribute(ATTR_Column_repeat, repeat_xmlch);
					XMLString::release(&repeat_xmlch);
				}

				if(vec.at(i).isColumnIdenticallyConstrained(j)) {
					XMLCh *all_xmlch(XMLString::transcode(convertResourceConstraintToString(vec.at(i).getResourcesArrayPtr()->at(j).front().second).c_str()));
					CNElement->setAttribute(ATTR_Column_all, all_xmlch);
					XMLString::release(&all_xmlch);
				} else {
					for(int k = 0; k < (int) vec.at(i).getResourcesArrayPtr()->at(j).size(); k++) {
						string lineX("L");
						lineX.append(Utils::itoa(k));

						XMLCh* ATTR_LX = XMLString::transcode(lineX.c_str());
						XMLCh* ATTR_LX_value = XMLString::transcode(convertResourceConstraintToString(vec.at(i).getResourcesArrayPtr()->at(j).at(k).second).c_str());
						CNElement->setAttribute(ATTR_LX, ATTR_LX_value);

						XMLString::release(&ATTR_LX);
						XMLString::release(&ATTR_LX_value);
					}
				}

				RZElement->appendChild(CNElement);

				// Update loop iterator
				j += repeatValue - 1;
			}

			rootElem->appendChild(RZElement);

			// Clean
			XMLString::release(&rzid_xmlch);
			XMLString::release(&column_xmlch);
			XMLString::release(&line_xmlch);
			XMLString::release(&height_xmlch);
			XMLString::release(&reverse_xmlch);

		}

		// Clean
		XMLString::release(&devName);
		XMLString::release(&devPackage);
		XMLString::release(&devSpeedGrade);

	} catch (...) {
		cerr << "Unknown exception!" << endl;
	}
}
