/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    RZ.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-08-18
 * @section DESCRIPTION
 *			Representation of a Reconfigurable Zone (RZ) for RZ generation.
 *			Coordinates represent a particulat resources, i.e. if a RZ is
 *			defined by coordinates {(0, 0), (0, 0)}, it is only composed of
 *			the resources located at (0, 0) in the FPGA model. If it is
 *			defined by {(1, 2), (3, 4)}:
 *
 *				x-----|-----|-----x
 *				|(1,2)|(2,2)|(3,2)|
 *				------|-----|------
 *				|(1,3)|(2,3)|(3,3)|
 *				------|-----|------
 *				|(1,4)|(2,4)|(3,4)|
 *				x-----|-----|-----x
 *
 *			Size of the array is : A = (x1-x0+1)*(y1-y0+1)
 */

#include "RZ.h"

#include <cstdlib>
#include <fstream>
#include "time.h"
#include "resourceRequirements.h"

int RZ::static_rz_id = 0;
double RZ::VERTEX_COST = 1;
double RZ::TASK_CORRELATION_COST = 1;
double RZ::INTERNAL_FRAGMENTATION_COST = 1;
int RZ::MINIMUM_DISTANCE_BETWEEN_RZ = 0;
bool RZ::IGNORE_RZ_POSITION_CONSTRAINTS = false;

/** Getters and setters */
int RZ::getX0(void) {
	return x0;
}

int RZ::getY0(void) {
	return y0;
}

int RZ::getHeight(void) {
	return height;
}

map<RBType, int> RZ::getConstrainedResourcesCount(void) {
	return constrainedResourcesCount;
}

vector<vector<pair<RBType, ResourceConstraint> > > * RZ::getResourcesArrayPtr(void) {
	return &resourcesArray;
}

int RZ::getRZCost(void) const {
	return (int)cost;
}

string RZ::getID(void) const {
	return id;
}

void RZ::setID(string s) {
	id = s;
}

/**
 * Computes the resources associated with the given RZ
 * using our specific FPGA hardware architecture models.
 * Translates CLBs into their Slices equivalent.
 */
void RZ::updateConstrainedResources(void) {

	// First, clean old table (may be optimized: TODO)
	constrainedResourcesCount.clear();

	for(int i = 0; i < (int) resourcesArray.size(); i++) {
		for(int j = 0; j < height; j++) {
			pair<RBType, ResourceConstraint> tmp = resourcesArray.at(i).at(j);

			// Check whether current resource is constrained to the RZ or not
			if(tmp.second == USED) {
				// Update table
				// If resources is not yet contained in the table, it will be automatically created
				// CLB -> Slice
				if(tmp.first == CLBm) {
					constrainedResourcesCount[SliceL]++;
					constrainedResourcesCount[SliceM]++;
				} else if(tmp.first == CLBl) constrainedResourcesCount[SliceL] += 2;
				else constrainedResourcesCount[tmp.first]++;
			}
		}
	}
}

/**
 * This method extends the current RZ size. Extension
 * can be done on the right or left side depending on
 * the boolean parameter.
 */
RZExtensionReturnValue RZ::extendRZ(void) {

	if(!reverse) {
		// If it is the RZ first extension, check if the column can either start/finish a RZ
		if(resourcesArray.size() == 0 && !device->canColumnBeginRZ(x0)) return RESTRICTED_COLUMN;

		// Extend RZ on the right side
		// First, check if RZ can be extended
		if((x0 + (int) resourcesArray.size()) > device->getNbColumns()) return END_OF_LINE;
		else {
			vector<pair<RBType, ResourceConstraint> > tmp_table(height);
			for(int i = 0; i < height; i++)
				tmp_table[i] = pair<RBType, ResourceConstraint>(device->getRB(y0 + i, x0 + resourcesArray.size()), UNUSED); // Careful with X and Y coordinates
			resourcesArray.push_back(tmp_table);

			updateConstrainedResources();

			for(int i = 0; i < height; i++) {
				if(!device->isColumnUsable(y0 + i, x0 + (int) resourcesArray.size() - 1)) return RESTRICTED_COLUMN;
			}
		}
	} else {
		// If it is the RZ first extension, check if the column can either start/finish a RZ
		if(resourcesArray.size() == 0 && !device->canColumnEndRZ(x0)) return RESTRICTED_COLUMN;

		// Extend RZ on the left side
		// First, check if RZ can be extended
		if((x0 - (int) resourcesArray.size()) < 0) return END_OF_LINE;
		else {
			vector<pair<RBType, ResourceConstraint> > tmp_table(height);
			for(int i = 0; i < height; i++)
				tmp_table[i] = pair<RBType, ResourceConstraint>(device->getRB(y0 + i, x0 - resourcesArray.size()), UNUSED); // Careful with X and Y coordinates
			resourcesArray.push_back(tmp_table);

			updateConstrainedResources();

			for(int i = 0; i < height; i++) {
				if(!device->isColumnUsable(y0 + i, x0 - (int) resourcesArray.size() + 1)) return RESTRICTED_COLUMN;
			}
		}
	}

	return SUCCESS;
}

/**
 * Displays a string representation of the RZ.
 * Displays the model upside down in order to comply
 * with PlanAhead floorplanning view.
 */
void RZ::printRZ(void) {

	if(resourcesArray.size() != 0) {
		for(int i = height - 1; i >= 0; i--) {
			for(int j = 0; j < (int) resourcesArray.size(); j++) {
				if(!reverse) cout << "| " << (resourcesArray.at(j)[i]).first << ", " << (resourcesArray.at(j)[i]).second << " ";
				else cout << "| " << (resourcesArray.at(resourcesArray.size() - j - 1)[i]).first << ", " << (resourcesArray.at(resourcesArray.size() - j - 1)[i]).second << " ";
			}
			cout << "|" << endl;
		}
	} else cout << "RZ is empty..." << endl;
}

/**
 * Displays information about the RZ.
 */
void RZ::info(void) const {

	cout << "RZ defined by x0(" << x0 << "), y0(" << y0 << "), length(" << resourcesArray.size() <<"), height(" << height << "), reverse(" << reverse << ") and ID(" << id << ")" << endl;
	cout << "Constrains the following resources:" << endl;
	for(map<RBType,int>::const_iterator it = constrainedResourcesCount.begin(); it != constrainedResourcesCount.end(); it++) {
		cout << "\t" << (*it).second << " " << device->getCellName(it->first) << " column(s)" << endl;
	}
}

/**
 * Computes cost associated with the reconfigurable zone. It is
 * composed of two parts: application compliance (number of tasks
 * that can be hosted by the RZ and RZ shape (rectangular zones
 * prefered to weird shapes). RZ shape cost is related to the number
 * of right angles inside the RZ.
 */
void RZ::computeCost(void) {

	int nb_vertices = 2; // Total number of vertices, including last two vertices
	int down_unused_height, up_unused_height;
	int last_down_unused_height = -1;
	int last_up_unused_height = -1;
	double shapeCost = 0;
	double taskCost = 0;
	double resourceCost = 0;

	// Shape cost
	for(int i = 0; i < (int) resourcesArray.size(); i++) {

		int j = 0;
		while(j < height && resourcesArray.at(i)[j].second != USED) j++;
		up_unused_height = j;
		if(up_unused_height != last_up_unused_height && up_unused_height != height) nb_vertices++;
		last_up_unused_height = up_unused_height;

		j = height;
		while(j > 0 && resourcesArray.at(i)[j-1].second != USED) j--;
		down_unused_height = j;
		if(down_unused_height != last_down_unused_height && down_unused_height != 0) nb_vertices++;
		last_down_unused_height = down_unused_height;
	}

	shapeCost = nb_vertices * VERTEX_COST;
	taskCost = nbIncompatibleTasks * TASK_CORRELATION_COST;
	resourceCost = resourceWastage * INTERNAL_FRAGMENTATION_COST;
	cost = taskCost + shapeCost + resourceCost;
}


/**
 * Generates the constraint file (.ucf) related to the RZ
 */
void RZ::generateUCF(string taskname) {

	// Compute taskname without extension (.syr)
	string taskname_wo_ext;
	taskname_wo_ext.append(taskname);
	size_t found = taskname_wo_ext.find(".syr");
	if (found != string::npos) taskname_wo_ext.erase(taskname_wo_ext.begin() + found, taskname_wo_ext.begin() + found + 4);

	// Generate unique filename
	string filename;
	filename.append("constraints/");
	filename.append(taskname_wo_ext);

	// Device ID
	filename.append("_");
	filename.append(device->getDeviceID());

	// RZ unique identifier
	filename.append("_RZ_");
	filename.append(id);
	filename.append(".ucf");

	// Open file in Write mode, deleting previous contents if any
	ofstream file(filename.c_str(), ios::out | ios::trunc);

	// Header
	time_t rawtime;
	time(&rawtime);
	struct tm *timeinfo = localtime(&rawtime);

	file << "#######################################################" << endl;
	file << "# Reconfigurable zone constraints" << endl;
	file << "# This file has been automatically generated by RZGen" << endl;
	file << "# on " << asctime(timeinfo);
	file << "#######################################################" << endl << endl;

	string pblock("pblock_");
	pblock.append(taskname_wo_ext);
	generateUCF(taskname_wo_ext, pblock, file);

	// Close file
	file.close();
}

void RZ::generateUCF(string taskname_wo_ext, string pblock_name, ofstream& file) {

	// Constraints
	file << "INST \"" << taskname_wo_ext << "\" AREA_GROUP = \"" << pblock_name << "\";" << endl;

	for(int i = 0; i < (int) resourcesArray.size(); i++) {
		for(int j = 0; j < height; j++) {
			if(resourcesArray.at(i)[j].second == USED) {

				// Add constraint
				switch(resourcesArray.at(i)[j].first) {

					case CLBl:
					case CLBm:
						file << "AREA_GROUP \"" << pblock_name << "\" RANGE=";
						if(!reverse) {
							file << "SLICE_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << device->getCell(CLBl).getNbCellsPerColumn() * (y0 + j)
								<< ":SLICE_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) + 1 << "Y" << device->getCell(CLBl).getNbCellsPerColumn() * (y0 + j) + device->getCell(CLBl).getNbCellsPerColumn() - 1 << ";" << endl;
						} else {
							file << "SLICE_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << device->getCell(CLBl).getNbCellsPerColumn() * (y0 + j)
								<< ":SLICE_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) + 1 << "Y" << device->getCell(CLBl).getNbCellsPerColumn() * (y0 + j) + device->getCell(CLBl).getNbCellsPerColumn() - 1 << ";" << endl;
						}
						break;

					case DSP:
						file << "AREA_GROUP \"" << pblock_name << "\" RANGE=";
						if(!reverse) {
							file << "DSP48_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << device->getCell(DSP).getNbCellsPerColumn() * (y0 + j)
								<< ":DSP48_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << device->getCell(DSP).getNbCellsPerColumn() * (y0 + j) + device->getCell(DSP).getNbCellsPerColumn() - 1 << ";" << endl;
						} else {
							file << "DSP48_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << device->getCell(DSP).getNbCellsPerColumn() * (y0 + j)
								<< ":DSP48_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << device->getCell(DSP).getNbCellsPerColumn() * (y0 + j) + device->getCell(DSP).getNbCellsPerColumn() - 1 << ";" << endl;
						}
						break;

					case BRAM:
						if(device->getDeviceClass() == Virtex6 || device->getDeviceClass() == Virtex7 || device->getDeviceClass() == Artix7
							|| device->getDeviceClass() == Kintex7 || device->getDeviceClass() == Zynq7) {
							// RAMB18
							file << "AREA_GROUP \"" << pblock_name << "\" RANGE=";
							if(!reverse) {
								file << "RAMB18_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << 2 * device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j)
									<< ":RAMB18_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << 2 * device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j) + device->getCell(BRAM).getNbCellsPerColumn() * 2 - 1 << ";" << endl;
							} else {
								file << "RAMB18_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << 2 * device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j)
									<< ":RAMB18_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << 2 * device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j) + device->getCell(BRAM).getNbCellsPerColumn() * 2 - 1 << ";" << endl;
							}
						}

						// RAMB36
						file << "AREA_GROUP \"" << pblock_name << "\" RANGE=";
						if(!reverse) {
							file << "RAMB36_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j)
								<< ":RAMB36_X" << device->getPhysicalXCoordinate(y0 + j, x0 + i) << "Y" << device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j) + device->getCell(BRAM).getNbCellsPerColumn() - 1 << ";" << endl;
						} else {
							file << "RAMB36_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j)
								<< ":RAMB36_X" << device->getPhysicalXCoordinate(y0 + j, x0 - i) << "Y" << device->getCell(BRAM).getNbCellsPerColumn() * (y0 + j) + device->getCell(BRAM).getNbCellsPerColumn() - 1 << ";" << endl;
						}
						break;

					default:
						cerr << "ERROR: Unrecognized reconfigurable resource" << endl;
				}
			}
		}
	}
}

bool RZ::isReversed(void) {
	return reverse;
}

void RZ::setNbIncompatibleTasks(int nbTasks) {
	nbIncompatibleTasks = nbTasks;
}

int RZ::getNbIncompatibleTasks(void) {
	return nbIncompatibleTasks;
}

bool RZ::isOverlappingRZ(const RZ &other) {

	int x00 = reverse ? x0 - (int) resourcesArray.size() + 1 : x0;
	int x10 = other.reverse ? other.x0 - (int) other.resourcesArray.size() + 1 : other.x0;

	int x01 = x00 + (int) resourcesArray.size() - 1;
	int x11 = x10 + (int) other.resourcesArray.size() - 1;

	int y00 = y0;
	int y10 = other.y0;

	int y01 = y00 + height - 1;
	int y11 = y10 + other.height - 1;

	if(!(((x01 + MINIMUM_DISTANCE_BETWEEN_RZ) < x10) || ((x11 + MINIMUM_DISTANCE_BETWEEN_RZ) < x00) || (y01 < y10) || (y11 < y00))) {

		// Possible overlap
		// Check with constrained resources

		// GREEDY IMPLEMENTATION


		return true;

	} else return false;
}

void RZ::addCompatibleTask(string task) {
	compatibleTasks.push_back(task);
}

vector<string> RZ::getCompatibleTaskVector(void) {
	return compatibleTasks;
}

bool RZ::overlapsRZInSet(vector<RZ> &rzs) {

	for(int i = 0; i < (int) rzs.size(); i++) {
		if(isOverlappingRZ(rzs.at(i))) return true;
	}

	return false;
}

int RZ::horizontalDistance(const RZ &other) {

	int x00 = reverse ? x0 - (int) resourcesArray.size() : x0;
	int x10 = other.reverse ? other.x0 - (int) other.resourcesArray.size() : other.x0;

	int x01 = x00 + (int) resourcesArray.size();
	int x11 = x10 + (int) other.resourcesArray.size();

	if(x10 > x01) return x10 - x01;
	else if(x00 > x11) return x00 - x11;
	else return 0;
}

int RZ::verticalDistance(const RZ &other) {

	int y00 = y0;
	int y10 = other.y0;

	int y01 = y00 + height;
	int y11 = y10 + other.height;

	if(y10 > y01) return y10 - y01;
	else if(y00 > y11) return y00 - y11;
	else return 0;
}

Device * RZ::getDevicePtr(void) {
	return device;
}

void RZ::clearCompatibleTasks(void) {
	compatibleTasks.clear();
}

/**
 * Returns bitstream size in words depending on the constrained resources
 * and on the device model
 */
int RZ::computeBitstreamSize(void) {

	int bitstreamSize = 0;

	for(int i = 0; i < (int) resourcesArray.size(); i++) {
		for(int j = 0; j < height; j++) {
			pair<RBType, ResourceConstraint> tmp = resourcesArray.at(i).at(j);

			// Check whether current resource is constrained to the RZ or not
			if(tmp.second == USED) {
				bitstreamSize += (device->getCell(tmp.first).getBitstreamSizePerColumn() / 4); // in words
			}
		}
	}

	return bitstreamSize;
}

bool RZ::constrainsColumn(Coordinates c) {
	int x00 = reverse ? x0 - (int) resourcesArray.size() + 1 : x0;
	int x10 = c.getX();

	int x01 = x00 + (int) resourcesArray.size() - 1;
	int x11 = c.getX();

	int y00 = y0;
	int y10 = c.getY();

	int y01 = y00 + height - 1;
	int y11 = c.getY();

	if(!(((x01 + MINIMUM_DISTANCE_BETWEEN_RZ) < x10) || ((x11 + MINIMUM_DISTANCE_BETWEEN_RZ) < x00) || (y01 < y10) || (y11 < y00))) return true;
	else return false;
}

bool RZ::operator==(const RZ &rz) const {

	bool sameLocation;

	if(y0 == rz.y0 && height == rz.height) {
		if((reverse && rz.reverse) || (!reverse && !rz.reverse))
			sameLocation = (x0 == rz.x0 && resourcesArray.size() == rz.resourcesArray.size());
		else {
			if(reverse) sameLocation = ((x0 - resourcesArray.size()) == (unsigned int)rz.x0 && resourcesArray.size() == rz.resourcesArray.size());
			else sameLocation = ((unsigned int)x0 == ((unsigned int)rz.x0 - rz.resourcesArray.size()) && resourcesArray.size() == rz.resourcesArray.size());
		}
	} else sameLocation = false;

	if(sameLocation) {
		// Now check constrained resources count
		ResourceRequirements req1(constrainedResourcesCount, device);
		ResourceRequirements req2(rz.constrainedResourcesCount, rz.device);
		return (req1 == req2);
	} else return false;
}

bool RZ::operator!=(const RZ &rz) const {
	return !(*this == rz);
}

void RZ::setTaskCorrelationCost(double c) {
	VERTEX_COST = c;
}

void RZ::setRZShapeCost(double c) {
	TASK_CORRELATION_COST = c;
}

void RZ::setInternalFragmentationCost(double c) {
	INTERNAL_FRAGMENTATION_COST = c;
}

void RZ::setMinimumDistanceBetweenRZs(int val) {
	MINIMUM_DISTANCE_BETWEEN_RZ = val;
}

void RZ::setResourceWastage(double val) {
	resourceWastage = val;
}

void RZ::setIgnoreRZPositionCOnstraints(bool val) {
	IGNORE_RZ_POSITION_CONSTRAINTS = val;
}

bool RZ::isResourcePositionOK(InterfaceLocation loc) {

	bool positionOK = false;

	if(constrainsColumn(loc.getInterfaceLocation())) {
		// Verify interface relative position inside RZ

		if(loc.getInterfacePosition() == DONT_CARE) positionOK = true;
		else {
			Coordinates interfaceCoordinates = loc.getInterfaceLocation();
			switch(loc.getInterfacePosition()) {
			case NORTH:
				positionOK = (interfaceCoordinates.getY() == (y0 + height - 1));
				break;
			case NORTHEAST:
				if(reverse) positionOK = (interfaceCoordinates.getY() == (y0 + height - 1) && interfaceCoordinates.getX() == x0);
				else positionOK = (interfaceCoordinates.getY() == (y0 + height - 1) && interfaceCoordinates.getX() == (x0 + (int)resourcesArray.size() - 1));
				break;
			case EAST:
				if(reverse) positionOK = (interfaceCoordinates.getX() == x0);
				else positionOK = (interfaceCoordinates.getX() == (x0 + (int)resourcesArray.size() - 1));
				break;
			case SOUTHEAST:
				if(reverse) positionOK = (interfaceCoordinates.getY() == y0 && interfaceCoordinates.getX() == x0);
				else positionOK = (interfaceCoordinates.getY() == y0 && interfaceCoordinates.getX() == (x0 + (int)resourcesArray.size() - 1));
				break;
			case SOUTH:
				positionOK = (interfaceCoordinates.getY() == y0);
				break;
			case SOUTHWEST:
				if(!reverse) positionOK = (interfaceCoordinates.getY() == y0 && interfaceCoordinates.getX() == x0);
				else positionOK = (interfaceCoordinates.getY() == y0 && interfaceCoordinates.getX() == (x0 - (int)resourcesArray.size() + 1));
				break;
			case WEST:
				if(!reverse) positionOK = (interfaceCoordinates.getX() == x0);
				else positionOK = (interfaceCoordinates.getX() == (x0 - (int)resourcesArray.size() + 1));
				break;
			case NORTHWEST:
				if(!reverse) positionOK = (interfaceCoordinates.getY() == (y0 + height - 1) && interfaceCoordinates.getX() == x0);
				else positionOK = (interfaceCoordinates.getY() == (y0 + height - 1) && interfaceCoordinates.getX() == (x0 - (int)resourcesArray.size() + 1));
				break;
			case DONT_CARE:
				positionOK = true; break;
			default:
				cerr << "Unknown interface position value..." << endl;
				positionOK = false;
			}
		}
	} else positionOK = false;

	return positionOK;
}

void RZ::setResourcesArray(vector<vector<pair<RBType, ResourceConstraint> > > &vec) {
	resourcesArray = vec;
}

bool RZ::areColumnsIdentical(int col1, int col2) {

	// Check if columns are identical in terms of constraints (USED/UNUSED...) and do not consider RBType
	if(col1 < 0 || col1 >= (int)resourcesArray.size() || col2 < 0 || col2 >= (int)resourcesArray.size()) {
		cerr << "ERROR: Index out of bounds while comparing columns!" << endl;
		exit(-1);
	}

	for(int i = 0; i < height; i++) {
		if(resourcesArray.at(col1).at(i).second != resourcesArray.at(col2).at(i).second) return false;
	}

	return true;
}

bool RZ::isColumnIdenticallyConstrained(int col) {

	if(col < 0 || col >= (int)resourcesArray.size()) {
		cerr << "ERROR: Index out of bounds while comparing columns!" << endl;
		exit(-1);
	}

	ResourceConstraint constraint(resourcesArray.at(col).front().second);
	for(int i = 1; i < height; i++) {
		if(resourcesArray.at(col).at(i).second != constraint) return false;
	}

	return true;
}

/**
 * @brief Check if first and last constrained columns are consistent with device description
 */
bool RZ::isRZWellShaped(void) {

	if(IGNORE_RZ_POSITION_CONSTRAINTS) return true;
	else {

		// Check if first/last column has some UNUSED stuff
		bool firstColumnfullyUsed = true;
		bool lastColumnfullyUsed = true;
		for(int i = 0; i < height; i++) {
			if(resourcesArray.front().at(i).second != USED) firstColumnfullyUsed = false;
			if(resourcesArray.back().at(i).second != USED) lastColumnfullyUsed = false;
		}

		int firstX = (reverse ? x0 - resourcesArray.size() + 1 : x0);
		int lastX = (reverse ? x0 : x0 + resourcesArray.size() - 1);

		//cout << "firstX: " << firstX << ", lastX: " << lastX << endl;

		return (device->canColumnBeginRZ(firstX)
			&&  device->canColumnEndRZ(lastX)
			&&  (firstColumnfullyUsed || device->canColumnBeginRZ(firstX + 1))
			&&  (lastColumnfullyUsed || device->canColumnEndRZ(lastX - 1)));
	}
}

bool RZ::isPurelyReconfigurable(void) {
	for(int i = 0; i < (int) resourcesArray.size(); i++) {
		for(int j = 0; j < (int) height; j++) {
			if(resourcesArray.at(i).at(j).second == NOT_RECONFIGURABLE) return false;
		}
	}

	return true;
}
