/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    my_priority_queue.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-04-28
 * @section DESCRIPTION
 *			Definition of a priority queue, similar to the one defined in STL
 *			but with some more functions for debugging purposes
 */

#ifndef MY_PRIORITY_QUEUE_H
#define MY_PRIORITY_QUEUE_H

#include <mutex>
#include <vector>
using namespace std;


template <class T, class Compare> 
class MyPriorityQueue {

private:
	vector<T> waiting_queue; /**< Queue implemented as a vector */
	int current_pos;
	std::mutex	mtx;

public:
	/* Typical priority queue functions (see priority queue in STL) */
	T top(void);
	void pop(void);
	int push(const T& elt);
	int size(void) const;
	bool empty(void) const;

	/* Additionnal functions for debug/scheduling purposes */
	T& at(int pos);
	const T& at(int pos) const;

	/* Look into the vector with current_pos variable (used by scheduling) */
	void resetCurrentPosition();
	T next_element(void);
	const T& current_element() const;
	T& current_element();
	void erase_current_element(); 
	const int current_position() const;
	bool last_element() const;

	void update_queue(void);
};

/**
 * Returns the first element in the priority queue
 */
template <class T, class Compare> 
T MyPriorityQueue<T, Compare>::top(void) {
	std::unique_lock<std::mutex> lck (mtx);
	return waiting_queue.front();
}

/**
 * Removes the first element in the priority queue
 */
template <class T, class Compare> 
void MyPriorityQueue<T, Compare>::pop(void) {
	std::unique_lock<std::mutex> lck (mtx);
	waiting_queue.erase(waiting_queue.begin());
	current_pos -= 1;
}

/**
 * Inserts an element into the priority queue. The priority queue
 * is sorted at any time during the simulation. To insert the element
 * properly, the method uses the Compare template class.
 * @param elt A reference to the element to insert. The element is copied into the queue
 */
template <class T, class Compare> 
int MyPriorityQueue<T, Compare>::push(const T& elt) {
	std::unique_lock<std::mutex> lck (mtx);
	// Version 1: basic FIFO, Always put the element at the end
	//waiting_queue.push_back(T(elt));

	// Version 2: real priority queue coupled with FIFO when comp is equal
	typename vector<T>::iterator it;
	Compare comp;
	bool inserted = false;
	int position = 0;
	for(it = waiting_queue.begin(); it < waiting_queue.end(); it++) {
		if(comp(elt, *it)) {
			// Insert element
			waiting_queue.insert(it, T(elt));
			inserted = true;
			break;
		}
		position++;
	}

	if(!inserted) waiting_queue.push_back(T(elt)); // Insert at the end

	return position;
}

/**
 * Returns the size of the priority queue
 * @return The waiting queue size
 */
template <class T, class Compare> 
int MyPriorityQueue<T, Compare>::size(void) const {
	return (int) waiting_queue.size();
}

/**
 * Checks if the priority queue is empty
 * @return True if the queue is empty, false otherwise
 */
template <class T, class Compare> 
bool MyPriorityQueue<T, Compare>::empty(void) const {
	return waiting_queue.empty();
}

/**
 * Retrieve element at position pos
 * @param pos Position in the queue
 * @return Element at position pos
 */
template <class T, class Compare> 
T& MyPriorityQueue<T, Compare>::at(int pos) {
	std::unique_lock<std::mutex> lck (mtx);
	return waiting_queue.at(pos);
}

/**
 * Retrieve element at position pos
 * @param pos Position in the queue
 * @return Element at position pos
 */
template <class T, class Compare> 
const T& MyPriorityQueue<T, Compare>::at(int pos) const {
	return waiting_queue.at(pos);
}

/**
 * Update queue after a priority change
 */
template <class T, class Compare> 
void MyPriorityQueue<T, Compare>::update_queue(void) {
	std::unique_lock<std::mutex> lck (mtx);
	vector<T> waiting_queue_copy(waiting_queue);

	waiting_queue.clear();
	for(int i = 0; i < (int) waiting_queue_copy.size(); i++) push(waiting_queue_copy.at(i));
}

/**
 * Reset the current position
 */
template <class T, class Compare> 
void MyPriorityQueue<T, Compare>::resetCurrentPosition() {
	std::unique_lock<std::mutex> lck (mtx);
	current_pos = -1;
}

/**
 * Retrieve next element according to current position
 * @return next Element at current position
 */
template <class T, class Compare> 
T MyPriorityQueue<T, Compare>::next_element(void) {
	std::unique_lock<std::mutex> lck (mtx);
	current_pos += 1;
	return waiting_queue.at(current_pos);
}
	
/**
 * Retrieve current element at current position
 * @return current Element at current position
 */
template <class T, class Compare> 
const T& MyPriorityQueue<T, Compare>::current_element() const {
	return waiting_queue.at(current_pos);
}

/**
 * Retrieve current element at current position
 * @return current Element at current position
 */
template <class T, class Compare> 
T& MyPriorityQueue<T, Compare>::current_element() {
	std::unique_lock<std::mutex> lck (mtx);
	return waiting_queue.at(current_pos);
}

/** 
 * Erase element at current pos
 * @param pos Position in the queue
 */
template <class T, class Compare> 
void MyPriorityQueue<T, Compare>::erase_current_element() {
	std::unique_lock<std::mutex> lck (mtx);
	waiting_queue.erase(waiting_queue.begin() + current_pos);
	current_pos -= 1;
}

/**
 * Retrieve position of the current element
 * @return position of the current element
 */
template <class T, class Compare> 
const int MyPriorityQueue<T, Compare>::current_position() const {
	return current_pos;
}

/**
 * End of element
 * @return True if it is the last element according to the current position, false otherwise
 */
template <class T, class Compare> 
bool MyPriorityQueue<T, Compare>::last_element() const {
	return (current_pos >= (int)(waiting_queue.size()-1));
}

#endif
