/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    Reconfiguration_manager_base.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-07-01
 * @section DESCRIPTION
 *			Implementation file for the reconfiguration manager
 */

#include "reconfiguration_manager_base.h"
#include <time.h>
#include "algorithm_mode_command.h" 

#define FOUT if(generate_logfile()) (*fout)

bool Reconfiguration_manager_base::CONSIDER_WAITING_STATE_AS_RUNNING_MODE = false;
bool Reconfiguration_manager_base::SEND_CONFIGURATION_NOTIFICATION_AFTER_IDLE_STATE = true;


// Static members
vector<Application> Reconfiguration_manager_base::applicationVector;
int Reconfiguration_manager_base::nbApplicationTerminated = 0;
FPGA* Reconfiguration_manager_base::fpga = 0;
int Reconfiguration_manager_base::RZ_TASK_BLANK = -1;

//REM: vrifie si les applications sont excutes correctement
bool Reconfiguration_manager_base::isSimulationOK(void) {

	bool allApplicationsOK = true;

	bool maxSimTimeReached = false;//(rs_time_stamp() >= MAXIMUM_SIMULATION_TIME);
	bool maxTimeNotEnough(false);
	
	if(maxSimTimeReached) {
		//cout << "CRITICAL WARNING: Simulation time reached its limit: " << MAXIMUM_SIMULATION_TIME << endl;
		maxTimeNotEnough = true;
	}

	for(int i = 0; i < (int) applicationVector.size(); i++) {
		
		// Check for particular case: application stalled
		if(maxSimTimeReached && applicationVector.at(i).isQoSSatisfied() && !applicationVector.at(i).getStatus()) {
			cout << "CRITICAL WARNING: Application " << applicationVector.at(i).getName() << " has an error status. Possible reasons:" << endl;
			cout << " - Application stalled at some point due to scheduling infeasability" << endl;
			cout << " - Tesbench is waiting for too much transactions" << endl;
			allApplicationsOK = false;
			maxTimeNotEnough = false;
		}

		// QoS check
		if(!applicationVector.at(i).isQoSSatisfied()) allApplicationsOK = false;
	}

	if(maxTimeNotEnough) {
		cout << "Maximum simulation time not enough for proper application execution" << endl;
		cout << "Consider modifying testbench or maximum simulation time in FoRTReSS preferences" << endl;
	}

	// Energy consumption check
	bool energyConsumptionOK = true;//(total_energy_consumption < ENERGY_CONSUMPTION_LIMIT);
	if(!energyConsumptionOK) cout << "Implementation does not satisfy energy consumption requirements" << endl;

	simulationOK = (allApplicationsOK && energyConsumptionOK);
	return simulationOK;
}

//REM: termine l'application "appname" en lui indiquant si l'excution est ok ou pas (boolean)
void Reconfiguration_manager_base::endApplication(string appname, bool ok) {

	static bool stop_requested = false;

	// Search for application in vector
	for(int i = 0; i < (int) applicationVector.size(); i++) {
		if(applicationVector.at(i).getName().compare(appname) == 0) {
			if(ok) applicationVector.at(i).checkQoSOnHyperperiod(); // last QoS check
			applicationVector.at(i).setStatus(ok);
			applicationVector.at(i).setAchievedSimulationTime(rs_time_stamp());
			break;
		}
	}

	nbApplicationTerminated++;
	if(!ok || nbApplicationTerminated == (int) applicationVector.size()) {
		//if(!stop_requested) sc_stop();
		stop_requested = true;
	}
}

void Reconfiguration_manager_base::verifySimulationTimes(void) {
	for(int i = 0; i < (int) applicationVector.size(); i++) {
		if(applicationVector.at(i).isSimulationTimeEnough()) 
			cout << "Application " << applicationVector.at(i).getName() << " has been simulated long enough to ensure QoS" << endl;
		else {
			if(applicationVector.at(i).getStatus()) {
				cout << "CRITICAL WARNING: Application " <<  applicationVector.at(i).getName() << " must be simulated longer to ensure QoS" << endl;
				cout << "Time simulated: " << applicationVector.at(i).getAchievedSimulationTime() << endl;
				cout << "Required simulation time: " << Application::getMinimumSimulationTime() << endl;
			}
		}
	}
}

/***********************************
 ******     User Interface    ******
 ***********************************/

//REM: Mthodes Utiles pour la gnration du Recosim_main.cpp

int Reconfiguration_manager_base::addApplication(string appName, string appPrefix, vector<Manager_interface *> table, double qos, int appInstanceID) {

	// New entry point
	int newAppOffset = (int) modules_table.size();
	for(int i = 0; i < (int) table.size(); i++) {
		modules_table.push_back(table.at(i));
		modules_table.back()->updateLists(newAppOffset); // Add offset
	}

	applicationVector.push_back(Application(appName, appPrefix, appInstanceID, qos, newAppOffset, (int) modules_table.size() - 1));
	applicationVector.back().computeMinimumSimulationTime(modules_table);
	
	return newAppOffset;
}

//REM: Appel  la fin de cration de toute les applis
void Reconfiguration_manager_base::applicationDefined(void) {

	//REM: initialisation par dfaut des tables de tous les modules de toutes les applis
	for(int i = 0; i < (int) modules_table.size(); i++) {
		task_state_table.push_back(INACTIVE);
		running_time_table.push_back(Running_time_information());

	}
//
//	// Add signals to trace
//	//trace_packet = new trace_packet_manager_t(&modules_table, &rz_table, NB_HW_RECONFIGURATION_UNITS, NB_SW_RECONFIGURATION_UNITS, fpga);
//
//	//REM: init les maps
	init_maps();

	vector<int> vec;
	for(int i = 0; i < (int) modules_table.size(); i++) finished_precedence_vectors.push_back(vec);

	// Verify that all tasks can be hosted
	//REM: permet de vrifier que chaque tache trouve une RZ ou processeur
	for(int i = 0; i < (int) modules_table.size(); i++) {
		if(!modules_table.at(i)->is_static()) {
			bool hosted = false;
			for(int j = 0; j < (int) rz_table.size(); j++) {
				if(modules_table.at(i)->fits(rz_table.at(j))) {
					hosted = true;
					break;
				}
			}

			if(!hosted) {
				cerr << endl << endl << "ERROR: Not all tasks hosted by RZ simulation subset! Check configuration file" << endl;
				FOUT << endl << endl << "ERROR: Not all tasks hosted by RZ simulation subset! Check configuration file" << endl;
				//exit(RECOSIM_NOT_ALL_TASKS_HOSTED_ERRCODE);
			}
		}
	}

//	Application::computeMinimumSimulationTime(applicationVector);
//
//	cout << "Minimum simulation time: " << Application::getMinimumSimulationTime() << endl;
//
//	//Creation des threads de vrification de l'hyperperiode (pour chaque appli)
////	for(int i = 0; i < (int) applicationVector.size(); i++) {
////		sc_spawn_options method_options;
////		string process_name("Dynamic_process_Application");
////		ostringstream out;
////		out << i;
////		process_name.append(out.str());
////
////		sc_spawn(sc_bind(&Reconfiguration_manager_base::deadline_check_thread, this, i), process_name.c_str(), &method_options);
////	}

	deadlines_table = new rs_time[(int)modules_table.size()];
//
	// QoS Management
	//qos_management = new QoS_management("QoS_Management", *this, fout);
	scheduler_thread_event.notify();
}

void Reconfiguration_manager_base::deadline_check_thread(int appID) {

	int current_hyperperiod = 0;

	//wait(Application::getGlobalOffset());
	while(true) {
		// New hyperperiod -> update VCD trace
		//set_hyperperiod_status(++current_hyperperiod);

		//wait(Application::getGlobalHypeperiod());
		//FOUT << rs_time_stamp() << ": " << name << ": QoS check for application " << applicationVector.at(appID).getName() << endl;
		applicationVector.at(appID).checkQoSOnHyperperiod();
	}
}

/***********************************
 ******       Scheduling      ******
 ***********************************/
void Reconfiguration_manager_base::init_maps(void) {

	/** Update implementation vector */
	for(int i = 0; i < (int) rz_table.size(); i++) {
		for(int j = 0; j < (int) rz_table.at(i)->getTaskImplementationVectorPtr()->size(); j++) {

			int moduleID = get_module_ID(rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).getTaskName());
			vector<ModuleImplementation> implVector = modules_table.at(moduleID)->get_implementation_vector();
			for(int k = 0; k < (int) implVector.size(); k++) {
				if(implVector.at(k).get_name().compare(rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).getImplementationName()) == 0) {
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setBestCaseExecutionTime(implVector.at(k).get_best_case_execution_time());
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setWorstCaseExecutionTime(implVector.at(k).get_worst_case_execution_time());
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setFrequency(implVector.at(k).get_frequency());
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setPidle(implVector.at(k).get_p_idle());
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setPrun(implVector.at(k).get_p_run());
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setNetlistName(implVector.at(k).get_netlist_name());
					rz_table.at(i)->getTaskImplementationVectorPtr()->at(j).setAlgorithmName(implVector.at(k).get_algorithm_name());
				}
			}
		}
	}

	/** Find potential allocations for each module */
	for(int i = 0; i < (int)modules_table.size(); i++) {

		if(!modules_table.at(i)->is_static()) {
			vector<RZ *> tmp_vect;
			for(int j = 0; j < (int)rz_table.size(); j++) {
				if(modules_table[i]->fits(rz_table[j])) tmp_vect.push_back(rz_table[j]);
			}

			map_compatible_rz.insert(pair<int,vector<RZ *> >(i, tmp_vect));
		}
	}

	/** Initialize the RZs with a implementation (defaulting to blank) */
	for(int i = 0; i < (int)rz_table.size(); i++) {
		map_current_module.insert(pair<RZ *, int> (rz_table[i], RZ_TASK_BLANK));
		//REM: trace
		set_rz_state(i, RZ_BLANK);
	}

	/** Display maps */
	display_map_compatible_RZ();
	display_map_current_module();
}

void Reconfiguration_manager_base::display_map_compatible_RZ(void) {

	//FOUT << "== Possible allocations for each task ==" << endl;
	map<int, vector<RZ *> >::iterator it;
	for(it = map_compatible_rz.begin(); it != map_compatible_rz.end(); it++) {
	//	FOUT << "Module " << modules_table[it->first]->getName() << " fits: ";
		for(int j = 0; j < (int) it->second.size(); j++) {
			if(j != ((int)it->second.size() - 1))
				FOUT << it->second.at(j)->getName() << ", ";
			else 
				FOUT << it->second.at(j)->getName();
		}
		FOUT << endl;
	}
	FOUT << endl;
}

void Reconfiguration_manager_base::display_map_current_module(void) {

	FOUT << "== RZ initialization ==" << endl;
	for(int i = 0; i < (int)rz_table.size(); i++) {
		if(map_current_module[rz_table[i]] == RZ_TASK_BLANK)
			FOUT << "RZ \"" << rz_table[i]->getName() << "\" is blank" << endl;
		else
			FOUT << "RZ \"" << rz_table[i]->getName() << "contains module" << modules_table[map_current_module[rz_table[i]]]->getName() << endl;
	}
	FOUT << endl;
}

void Reconfiguration_manager_base::display_task_state_table(void) const{

#ifdef GENERATE_TASK_STATE_FILE
	file_task_state << rs_time_stamp();
#endif
	
	int nb_rz = 0;

	for(int i = 0; i < (int) modules_table.size(); i++) {

		FOUT << "\t" << modules_table[i]->getName() << ": " << Task_state_string[(int) task_state_table[i]] << endl;
#ifdef GENERATE_TASK_STATE_FILE
		file_task_state << "; " << Task_state_string[(int) task_state_table[i]] << ": " << modules_table[i]->get_nb_processed_transactions();
#endif

	}

#ifdef GENERATE_TASK_STATE_FILE
	file_task_state << " -- Nb RZ: " << nb_rz << endl;
#endif
}

void Reconfiguration_manager_base::display_rz_current_module_table(void) const {

#ifdef GENERATE_RZ_STATE_FILE
	file_rz_state << rs_time_stamp();
	for(int i = 0; i < (int) rz_table.size(); i++) file_rz_state << "; " << map_current_module[rz_table[i]];
	file_rz_state << endl;
#endif
}

/***************************************
 ******   TLM communication part  ******
 ***************************************/

//REM: Communication du Manager vers les modules
//REM: TLM bloquant avec temps nul

//REM: permet de dire au module qu'un successeur est pret, NEXT_MODULE_READY (si plusieurs successeur alors on aura plusieurs appels)
//REM: appeler lorsque qu'une config est faite

void Reconfiguration_manager_base::send_next_module_ready(int module_id, int module_ready_id) {
	
	rs_payload* trans;

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	int local_data = (int) NEXT_MODULE_READY + (module_ready_id << 16);

	// Set all attributes except byte_enable_length (unused)
	trans->set_command(COM_WRITE_COMMAND);
	trans->set_address(MANAGER_COMMAND_SCHEDULING);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(&local_data));
	trans->set_data_length(4);
	//trans->set_streaming_width(4);							// = data_length to indicate no streaming
	//trans->set_byte_enable_ptr(0);							// 0 indicates unused
	//trans->set_dmi_allowed(false);							// Mandatory initial value
	//trans->set_response_status(TLM_INCOMPLETE_RESPONSE);	// Mandatory initial value

	//rs_time delay(RS_ZERO_TIME);
	module_initiator_socket[module_id]->b_transport(*trans/*, delay*/);
	//module_initiator_socket.b_transport(*trans);

	//Utils::check_transaction(*trans);
	trans->release();
}

//REM: indique  un module (CURRENT_MODULE_READY) qu'il peut reprendre son excution apres une demande de preemption
void Reconfiguration_manager_base::send_current_module_ready(int module_id) {

	rs_payload* trans;

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	int local_data = (int) CURRENT_MODULE_READY;

	// Set all attributes except byte_enable_length (unused)
	trans->set_command(COM_WRITE_COMMAND);
	trans->set_address(MANAGER_COMMAND_SCHEDULING);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(&local_data));
	trans->set_data_length(4);
	//trans->set_streaming_width(4);							// = data_length to indicate no streaming
	//trans->set_byte_enable_ptr(0);							// 0 indicates unused
	//trans->set_dmi_allowed(false);							// Mandatory initial value
	//trans->set_response_status(TLM_INCOMPLETE_RESPONSE);	// Mandatory initial value

	//rs_time delay(RS_ZERO_TIME);
	module_initiator_socket[module_id]->b_transport(*trans/*, delay*/);
	//module_initiator_socket.b_transport(*trans);

	//Utils::check_transaction(*trans);
	trans->release();
}

void Reconfiguration_manager_base::send_module_ready_to_testbench(int tbid, int module_ready_id) {

	rs_payload* trans;

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	int local_data = (int) NEXT_MODULE_READY + (module_ready_id << 16);

	// Set all attributes except byte_enable_length (unused)
	trans->set_command(COM_WRITE_COMMAND);
	trans->set_address(MANAGER_COMMAND_SCHEDULING);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(&local_data));
	trans->set_data_length(4);
	//trans->set_streaming_width(4);							// = data_length to indicate no streaming
	//trans->set_byte_enable_ptr(0);							// 0 indicates unused
	//trans->set_dmi_allowed(false);							// Mandatory initial value
	//trans->set_response_status(TLM_INCOMPLETE_RESPONSE);	// Mandatory initial value

	//rs_time delay(RS_ZERO_TIME);
	testbench_initiator_socket[tbid]->b_transport(*trans/*, delay*/);

	//Utils::check_transaction(*trans);
	trans->release();
}

void Reconfiguration_manager_base::Scheduler_send_update_parameters_to_module(RZ* rz) {
	//for simulation onlie
}

//REM: permet de signaler  l'owner que le module 
//REM: owner_id < 0 : testbench, autrement un module
void Reconfiguration_manager_base::notify_request_owner_module_ready(int owner_id, int ready_module_id) {

	if(owner_id < 0) { // Means entry point

		// Search entry point ID
		//REM: il est forcment positif
		int epid = -1 * (owner_id + 1);
		send_module_ready_to_testbench(epid, ready_module_id); 
	} else {
		if(owner_id != ready_module_id) 
			send_next_module_ready(owner_id, ready_module_id); //REM: Cas d'instanciation d'un successeur
		else 
			send_current_module_ready(owner_id); //REM: Cas de la preemption : owner = ready_module_id
	}
}



/***********************************
 ***       Scheduling part       ***
 ***********************************/

void *Reconfiguration_manager_base::scheduler_handler(void) {
	
	while(true) {

		scheduler_thread_event.wait();

		(*waiting_queue_handler_ptr)(*this);	// Perform scheduling & simulate scheduler execution time
	}
}


/***********************************
 ***  Communication uplink part  ***
 ***********************************/

void Reconfiguration_manager_base::update_deadline(int modID) {
	// Update configuration request time
	modules_table[modID]->set_request_time();

	// Update local table
	deadlines_table[modID] = modules_table[modID]->get_deadline_time();
}


int Reconfiguration_manager_base::get_module_ID(string moduleName) {
	for(int i = 0; i < (int) modules_table.size(); i++) {
		if(modules_table.at(i)->getName().compare(moduleName) == 0) return i;
	}

	cerr << "ERROR: Manager unable to retrieve module ID from module name! Exiting..." << endl;
	FOUT << "ERROR: Manager unable to retrieve module ID from module name! Exiting..." << endl;
}


/**
 * This function asks for the reconfiguration of a given task 'id'. It simply
 * puts the task in the reconfiguration list and sends an event to the reconfiguration
 * engine which handles the process.
 */
void Reconfiguration_manager_base::configure_task(int id, pid_t tid, int implID, RZ *rz) {

	//REM: mise  jour du sc_buffer de la RZ qui va tre configure
	rz_configuration_request_buffer[getRZID(rz)].write(ConfigurationRequest(id, tid ,implID, rz));

}

//REM: Thread dynamique (1 par RZ)
void Reconfiguration_manager_base::rz_configuration_request_thread(int rzid) {

	while(true) {

		//REM: attente d'une reconfiguration
		rz_configuration_request_buffer[rzid].wait_value_change();

		ConfigurationRequest request(rz_configuration_request_buffer[rzid].read());

		// Set if the Bitstream is the same between actual and new implementation (only for Hw Impl)
		// if yes, no reconfiguration time

		TaskImplementation newTaskImplementation = rz_table.at(rzid)->getTaskImplementationVector().at(request.getTaskImplementationID());

		bool same_bitstream = newTaskImplementation.hasSameNetlist(current_implementation_on_RZ[rzid]);

		FOUT << rs_time_stamp() << ": new : " << newTaskImplementation.getNetlistName() << " current : " << current_implementation_on_RZ[rzid].getNetlistName() << " same = " << same_bitstream << endl;

		//FOUT << rs_time_stamp() << ": same bitstream 1: " << same_bitstream << " new: " << newTaskImplementation.getFullName() << " current: " << current_implementation_on_RZ[rzid].getFullName() << endl;
		//FOUT << rs_time_stamp() << ": same bitstream 2: " << same_bitstream << " previous: " << previous_implementation_on_RZ[rzid].getFullName() << " current: " << current_implementation_on_RZ[rzid].getFullName() << endl;

		//
		// CONTEXT SAVE
		//
		previous_implementation_on_RZ[rzid] = current_implementation_on_RZ[rzid];
		if(previous_implementation_on_RZ[rzid].isValid() && 
			(is_task_preempted(previous_implementation_on_RZ[rzid].getTaskName()) || modules_table.at(get_module_ID(previous_implementation_on_RZ[rzid].getTaskName()))->use_context_switch_mode(previous_implementation_on_RZ[rzid].getImplementationName()))) {

			//REM: Trace
			set_task_state(get_module_ID(previous_implementation_on_RZ[rzid].getTaskName()), rzid, CONTEXT_SAVE);
			set_task_context_save(get_module_ID(previous_implementation_on_RZ[rzid].getTaskName()), rzid);
			//REM: attente ctx switch
			//wait(previous_implementation_on_RZ[rzid].getContextSwitchTime());
		}

		// Update task & RZ states (Trace)
		if(previous_implementation_on_RZ[rzid].isValid()) {
			if(is_task_preempted(get_module_ID(previous_implementation_on_RZ[rzid].getTaskName()))) set_task_state(get_module_ID(previous_implementation_on_RZ[rzid].getTaskName()), rzid, PREEMPTED_INACTIVE);
			else set_task_state(get_module_ID(previous_implementation_on_RZ[rzid].getTaskName()), rzid, INACTIVE);
		}
		
		// Met  jour l'impl courante de la RZ
		rz_table.at(rzid)->setCurrentImplementationID(request.getTaskImplementationID());

		if (same_bitstream == false) {
			//
			// RECONFIGURATION
			//
			if(rz_table.at(rzid)->get_implementation_type() == HARD) {

				//REM: liste des taches hw  reconfigurer (file d'attente)
				reconfiguration_waiting_list.push_back(request);

				// Search for an idle reconfiguration unit
				for(int i = 0; i < NB_HW_RECONFIGURATION_UNITS; i++) {
					if(!hw_reconfiguration_units[i].isBusy()) {
						hw_reconfiguration_event_table[i].notify();
						break;
					}
				}

				//REM: Si pas trouv de ICAP libre, alors, il y a un thread de gestion ICAP qui le fera ds que possible.

			} else {

				//REM: liste des taches sw  reconfigurer (file d'attente)
				software_reconfiguration_waiting_list.push_back(request);

				// Search for an idle reconfiguration unit
				for(int i = 0; i < NB_SW_RECONFIGURATION_UNITS; i++) {
					if(!sw_reconfiguration_units[i].isBusy()) {
						sw_reconfiguration_event_table[i].notify();
						break;
					}
				}
			}


			// Wait for the end of configuration  titi
			rz_configured_event[rzid].wait();
		}

		//
		// CONTEXT RESTORE
		//
		TaskImplementation taskImplementation = rz_table.at(rzid)->getTaskImplementationVector().at(request.getTaskImplementationID());
		// If task is preempted (or context switch mode is enabled), add context switch time to the reconfiguration overhead (context load)
		if(is_task_preempted(request.getTaskID()) || modules_table.at(request.getTaskID())->use_context_switch_mode(taskImplementation.getImplementationName())) {
			//REM: Trace
			set_task_state(request.getTaskID(), rzid, CONTEXT_RESTORE);
			set_task_context_restore(request.getTaskID(), rzid);
			//REM: attente cxt switch
			//wait(taskImplementation.getContextSwitchTime());
		}


		// Set execution time depending on implementation
		modules_table[request.getTaskID()]->set_current_implementation(taskImplementation.getImplementationName());
		modules_table[request.getTaskID()]->send_start_algorithm_event();

		//if(rz_table.at(rzid)->get_implementation_type() == HARD) update_resources_trace(rzid, request.getTaskID());	// HW only

		current_implementation_on_RZ[rzid] = taskImplementation;

		//set_task_state(request.getTaskID(), rzid, RUNNING);
		set_task_state(request.getTaskID(), rzid, IDLE);


		if(is_task_preempted(request.getTaskID())) {
			// Remove task from list
			remove_task_from_preemption_list(request.getTaskID());

			// Notify current module
			send_current_module_ready(request.getTaskID());

		} else {

			if(ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST || !SEND_CONFIGURATION_NOTIFICATION_AFTER_IDLE_STATE) {
				// Notify previous modules
				for(int i = 0; i < (int) finished_precedence_vectors[request.getTaskID()].size(); i++) notify_request_owner_module_ready(finished_precedence_vectors[request.getTaskID()].at(i), request.getTaskID());
			}
			if(ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) finished_precedence_vectors.at(request.getTaskID()).clear();
		}

		// Display current state for each task
		display_task_state_table();
		display_rz_current_module_table();

	}
}

int Reconfiguration_manager_base::getRZID(RZ *rz) {
	for(int i = 0; i < (int) rz_table.size(); i++) {
		if(rz_table.at(i) == rz) return i;
	}

	cerr << "ERROR: Could not retrieve RZ ID from pointer!" << endl;
	//exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
}

/**
 * This thread controls the reconfiguration process. It ensures
 * the mutual exclusion needed on the reconfiguration port (ICAP).
 * It treats the tasks in a FIFO style, since the task priority is
 * handled before the actual reconfiguration process.
 */

//REM: thread de reconfiguration Hw (1 thread par ICAP)
void Reconfiguration_manager_base::hardware_reconfiguration_engine_thread(int unitID) {

	int id;
	RZ* hostingRZ;
	TaskImplementation taskImplementation;
	int taskImplementationID;

	while(true) {

		// Wait for a reconfiguration to be needed
		hw_reconfiguration_event_table[unitID].wait();

		hw_reconfiguration_units[unitID].setBusy(true);

		// Loop while queue is not empty
		while(!reconfiguration_waiting_list.empty()) {

			FOUT << rs_time_stamp() << ": Reconfiguration waiting list size: " << reconfiguration_waiting_list.size() << endl;
			for(list<ConfigurationRequest>::iterator it = reconfiguration_waiting_list.begin(); it != reconfiguration_waiting_list.end(); it++) FOUT << it->getTaskID() << " - ";
			FOUT << endl;

			// Retrieve first element
			id = reconfiguration_waiting_list.front().getTaskID();
			hostingRZ = reconfiguration_waiting_list.front().getHostingRZ();
			taskImplementationID = reconfiguration_waiting_list.front().getTaskImplementationID();
			taskImplementation = hostingRZ->getTaskImplementationVector().at(taskImplementationID);

			// Remove first elements from lists
			reconfiguration_waiting_list.pop_front();


			// Update request time
			//REM: permet de mettre  jour la partie analog de la trace (figure 2)

			// Display current state for each task
			display_task_state_table();
			display_rz_current_module_table();

			// Simulate reconfiguration time
			set_task_state(id, getRZID(hostingRZ), CONFIG);
			//rs_time reconfigurationTimeOverhead(taskImplementation.getCompressedReconfigurationTime());

			rz_configured_event[getRZID(hostingRZ)].notify();

			FOUT << rs_time_stamp() << ": Module " << modules_table[id]->getName() << " (" << id << ") configured" << endl;

			//hw_reconfiguration_units[unitID].getBusyTime() += reconfigurationTimeOverhead;

			// VCD Tracing purpose
			//set_icap_idle(unitID);

			// Call Scheduler toto
			scheduler_thread_event.notify();
		}

		hw_reconfiguration_units[unitID].setBusy(false);
	}
}

/**
 Same as previous thread but for software reconfiguration
 */
void Reconfiguration_manager_base::software_reconfiguration_engine_thread(int unitID) {
	
	int id;
	pid_t tid;
	RZ* hostingRZ;
	TaskImplementation taskImplementation;
	int taskImplementationID;

	while(true) {

		// Wait for a reconfiguration to be needed
		sw_reconfiguration_event_table[unitID].wait();

		sw_reconfiguration_units[unitID].setBusy(true);

		// Loop while queue is not empty
		while(!software_reconfiguration_waiting_list.empty()) {

			FOUT << rs_time_stamp() << ": Software reconfiguration waiting list size: " << software_reconfiguration_waiting_list.size() << endl;
			for(list<ConfigurationRequest>::iterator it = software_reconfiguration_waiting_list.begin(); it != software_reconfiguration_waiting_list.end(); it++) FOUT << it->getTaskID() << " - ";
			FOUT << endl;

			// Retrieve first element
			id  = software_reconfiguration_waiting_list.front().getTaskID();
			tid = software_reconfiguration_waiting_list.front().getTaskTID();
			hostingRZ = software_reconfiguration_waiting_list.front().getHostingRZ();
			taskImplementationID = software_reconfiguration_waiting_list.front().getTaskImplementationID();
			taskImplementation = hostingRZ->getTaskImplementationVector().at(taskImplementationID);

			// Remove first elements from lists
			software_reconfiguration_waiting_list.pop_front();

			// VCD Tracing purpose
			//set_software_loader_config_task(unitID, id);
			set_task_state(id, getRZID(hostingRZ), CONFIG);

			// Update request time
			//sw_reconfiguration_units[unitID].getRequestTime() = rs_time_stamp();

			// Display current state for each task
			display_task_state_table();
			display_rz_current_module_table();

			// Simulate reconfiguration time
			set_task_state(id, getRZID(hostingRZ), CONFIG);
			//rs_time reconfigurationTimeOverhead(taskImplementation.getCompressedReconfigurationTime());
			//wait(reconfigurationTimeOverhead);

			rz_configured_event[getRZID(hostingRZ)].notify();

			if (tid == 0){
				modules_table[id]->wait_tid_valide();
				tid = modules_table[id]->get_thread_id();
			}

			FOUT << " \033[33;4m"<< rs_time_stamp() << ": Module " << modules_table[id]->getName() << " thread id is : " << hex << tid << dec << " configured on core : " << hostingRZ->getName()   << "\033[39;24m"<< endl;

			cpu_set_t set;
			CPU_ZERO(&set);
			CPU_SET(hostingRZ->get_cpu_id(), &set);
			if (sched_setaffinity(tid, sizeof(set), & set)  == -1){
				FOUT <<" \033[33;4m"<< rs_time_stamp() << ": Module " << modules_table[id]->getName() << " set faild"   << "\033[39;24m"<< endl;
			}
			else {
				FOUT <<" \033[33;4m"<< rs_time_stamp() << ": Module " << modules_table[id]->getName() << " set succes"   << "\033[39;24m"<< endl;
			}
			//sw_reconfiguration_units[unitID].getBusyTime() += reconfigurationTimeOverhead;

			// VCD Tracing purpose
			set_software_loader_idle(unitID);
		}

		sw_reconfiguration_units[unitID].setBusy(false);
	}
}


/*********************************
 *      Scheduler interface      *
 *********************************/

vector<RZ *> Reconfiguration_manager_base::Scheduler_get_compatible_rz_vector(int task_id) {
	return map_compatible_rz[task_id];
}

rs_time Reconfiguration_manager_base::Scheduler_get_task_running_time(int task_id) {
	return running_time_table[task_id].get_running_time_duration();
}

vector<RZ *> Reconfiguration_manager_base::Scheduler_get_all_rz_vector(void) {
	return rz_table;
}

int Reconfiguration_manager_base::Scheduler_get_application_number(void) {
	return applicationVector.size();
}

Application_interface* Reconfiguration_manager_base::Scheduler_get_application_interface(int index) {
	return &applicationVector.at(index);
}

bool Reconfiguration_manager_base::Scheduler_exist_rz_properties(string inst_name) {
	return (rz_parameter_table[inst_name] != NULL);
}

bool Reconfiguration_manager_base::Scheduler_exist_rz_properties(RZ *rz) {
	RZ_config* ptr;
	if (rz->get_implementation_type() == HARD)
		ptr = rz_parameter_table["RZDefaultGroupName"];
	else
		ptr = rz_parameter_table[rz->getName()];
	return (ptr != NULL);
}

RZ_config& Reconfiguration_manager_base::Scheduler_get_rz_properties(RZ *rz) {
	if (rz->get_implementation_type() == HARD)
		return *rz_parameter_table["RZDefaultGroupName"];
	else
		return *rz_parameter_table[rz->getName()];
}

RZ_config& Reconfiguration_manager_base::Scheduler_get_rz_properties(string inst_name) {
	return *rz_parameter_table[inst_name];
}

Config_fct_point& Reconfiguration_manager_base::Scheduler_get_current_fct_point(RZ *rz) {
	return rz->getConfigFctPoint();
}

int Reconfiguration_manager_base::Scheduler_get_current_module_ID(RZ* rz) {
	return map_current_module[rz];
}

bool Reconfiguration_manager_base::Scheduler_is_RZ_blank(RZ *rz) {
	return map_current_module[rz] == RZ_TASK_BLANK;
}

int Reconfiguration_manager_base::Scheduler_get_module_ID(SchedulerRequest& req) {
	int waiting_task_id = -1;
		
	// Get corresponding ID
	for(int i = 0; i < (int) modules_table.size(); i++) {
		//if(req.get_task_to_map_ptr() == modules_table.at(i)) {
		if(req.getName().compare(modules_table.at(i)->getName()) == 0) {
			waiting_task_id = i;
			break;
		}
	}

	if(waiting_task_id == -1) {
		cerr << "ERROR: Could not retrieve waiting task ID from interface! Exiting..." << endl;
		//exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
	}

	return waiting_task_id;
}

pid_t Reconfiguration_manager_base::Scheduler_get_module_tid(int waiting_task_id){
	return(modules_table.at(waiting_task_id)->get_thread_id()); 
}

void Reconfiguration_manager_base::Scheduler_emulate_scheduler_behavior(void) {
	//for simulation onlie
}

void Reconfiguration_manager_base::Scheduler_set_scheduler_active(int taskid) {
//	FOUT << rs_time_stamp() << ": " << name << " trying to allocate task " << taskid << " (" << modules_table.at(taskid)->getName() << ")" << endl;
	//set_scheduler_active(taskid);
}

bool Reconfiguration_manager_base::Scheduler_has_task_already_requested_mapping(int task_to_map, int request_owner) {
	for(int i = 0; i < (int) finished_precedence_vectors.at(task_to_map).size(); i++) {
		if(finished_precedence_vectors.at(task_to_map).at(i) == request_owner) {
			// Task has already been taken into account
			return true;
		}
	}

	return false;
}

void Reconfiguration_manager_base::Scheduler_add_finished_precedence(int task_mapped, int precedence) {
	finished_precedence_vectors.at(task_mapped).push_back(precedence);
}

void Reconfiguration_manager_base::Scheduler_notify_request_owner_module_ready(int owner, int ready) {
	notify_request_owner_module_ready(owner, ready);
}

void Reconfiguration_manager_base::Scheduler_print_finished_precedence_vector(int task_id) {
//	FOUT << "Finished precedence list size for task " << task_id << ": " << finished_precedence_vectors.at(task_id).size() << endl;
	for(int i = 0; i < (int) finished_precedence_vectors.at(task_id).size(); i++) {
//		FOUT << finished_precedence_vectors.at(task_id).at(i) << " - ";
	}
//	FOUT << endl;
}

void Reconfiguration_manager_base::Scheduler_configure_task(RZ* rz, int id, pid_t tid, int implID, int requestOwner) {

	// Set status for previous task implemented on RZ to WAITING (if necessary)
	//if(!Scheduler_is_RZ_blank(rz)) set_task_state(map_current_module[rz], getRZID(rz), WAITING);	// TODO: UTILE ?

	// Visual2010
	//map_current_module.at(rz) = id; // 'Reserve' RZ

	// Visual2008
	map_current_module[rz] = id;

	//rz->setCurrentImplementationID(implID);
	//configure_task(id, rz->getTaskImplementationVector().at(implID));
	configure_task(id, tid, implID, rz);
	set_task_state(id, getRZID(rz), QUEUED); // NOT Moved to configure_task
	//finished_precedence_vectors.at(id).push_back(requestOwner);
	if(!ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) finished_precedence_vectors.at(id).push_back(requestOwner);
}

void Reconfiguration_manager_base::Scheduler_rerun_task(SchedulerRequest& request) {

	int waiting_task_id = Scheduler_get_module_ID(request);

	//REM: Trace + mise  jour d'une map
	set_task_state(waiting_task_id, get_hosting_rz_id(waiting_task_id), IDLE);

//	if(is_task_preempted(waiting_task_id)) {
//		//REM: enlve la tache de la liste de preemption (voir Reconfiguration_manager::b_transport_module)
//		remove_task_from_preemption_list(waiting_task_id);
//		//REM: continu l'execution de la tache 
//		notify_request_owner_module_ready(request.get_request_owner(), waiting_task_id);
//	} else {
		//REM: tache  redmarrer
		//REM: on ajoute le module qui a fait la requete  la liste des prdcesseurs qui sont prts  envoyer les donnes. (figure 5)
		if(!ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) finished_precedence_vectors.at(waiting_task_id).push_back(request.get_request_owner());

		// Notify previous modules
		//REM: notifie les modules qui en ont fait la demande
		for(int i = 0; i < (int) finished_precedence_vectors[waiting_task_id].size(); i++) notify_request_owner_module_ready(finished_precedence_vectors[waiting_task_id].at(i), waiting_task_id);
		if(ALL_DEPENDENCIES_REQUIRED_BEFORE_REQUEST) finished_precedence_vectors.at(waiting_task_id).clear();

		
	//}

	//notify_request_owner_module_ready(request.get_request_owner(), waiting_task_id);

	
	modules_table.at(waiting_task_id)->send_start_algorithm_event();
	
}

void Reconfiguration_manager_base::Scheduler_set_current_task(RZ *rz, int task_id) {
	map_current_module[rz] = task_id;
}

Task_state Reconfiguration_manager_base::Scheduler_get_task_state(int task_id) {
	return task_state_table.at(task_id);
}

Task_state Reconfiguration_manager_base::Scheduler_get_task_state(RZ *rz) {
	return Scheduler_get_task_state(map_current_module[rz]);
}

bool Reconfiguration_manager_base::Scheduler_set_blank(RZ *rz) {
	return Scheduler_set_blank(getRZID(rz));
}

bool Reconfiguration_manager_base::Scheduler_set_blank(int hosting_rz_id) {

	// getRZID
	// Task_state Scheduler_get_task_state(RZ *rz);
	// Task_state_string

	FOUT << rs_time_stamp() << " *** SET BLANK RZ" << endl;
	cout << rs_time_stamp() << " *** SET BLANK RZ" << endl;

	RZ* rz = rz_table.at(hosting_rz_id);
	FOUT << "RZ : " << rz->getName() << endl;
	FOUT << "RZ state : " << RZState_string[rz->getRZState()] << endl;

	cout << "RZ : " << rz->getName() << endl;
	cout << "RZ state : " << RZState_string[rz->getRZState()] << endl;

	if (rz->getRZState() != RZ_BLANK) {

		int taskID = map_current_module[rz];
		// Check if the task state on the RZ is not BLANK 
		
		if (taskID != RZ_TASK_BLANK) {

			FOUT << "task id : " << taskID << ", name : " << task_table.at(taskID).get_task_ID() << endl;
			cout << "task id : " << taskID << ", name : " << task_table.at(taskID).get_task_ID() << endl;

			Task_state taskState = Scheduler_get_task_state(rz);
			FOUT << "task state : " << Task_state_string[taskState] << endl;
			cout << "task state : " << Task_state_string[taskState] << endl;

			if ((taskState == IDLE || taskState == MAPPED || taskState == INACTIVE) && modules_table.at(taskID)->is_static() == false) {
				// ** set BLANK **
				
				// Update Tables
				map_current_module[rz] = RZ_TASK_BLANK;		
				rz_table[hosting_rz_id]->setRZState(RZ_BLANK);
				
				// No Implementation
				rz->setCurrentImplementationID(0);

				// Update Task State and RZ State (INACTIVE = No association with a RZ)
				set_task_state(taskID, hosting_rz_id, INACTIVE);

				// Update RZ State
				//set_rz_state(hosting_rz_id, RZ_BLANK);

				FOUT << "UPDATE RZ : BLANK" << endl;
				cout << "UPDATE RZ : BLANK" << endl;
			}
			else { 
				FOUT << " *** NO POSSIBLE RZ BLANKING ! " << endl;
				cout << " *** NO POSSIBLE RZ BLANKING ! " << endl;
				return false;	// Error: task is not IDLE, MAPPED, INACTIVE or it is a static task
			}
		}
		else {
				FOUT << " *** RZ ALREADY BLANK ! " << endl;
				cout << " *** RZ ALREADY BLANK ! " << endl;		
		}
	}
	else {
			FOUT << " *** RZ ALREADY BLANK ! " << endl;
			cout << " *** RZ ALREADY BLANK ! " << endl;		
	}
	
	return true;
}

//bool Reconfiguration_manager_base::Scheduler_is_task_preempted(int taskid) const {
//	return is_task_preempted(taskid);
//}

double Reconfiguration_manager_base::Scheduler_get_performance_effort(void) const {
	return SCHEDULER_PERFORMANCE_EFFORT;
}

double Reconfiguration_manager_base::Scheduler_get_power_effort(void) const {
	return SCHEDULER_POWER_EFFORT;
}

double Reconfiguration_manager_base::Scheduler_get_area_effort(void) const {
	return SCHEDULER_AREA_EFFORT;
}

string Reconfiguration_manager_base::Scheduler_get_name(void) const {
	return "\033[36;4m"+name+"\033[39;24m";
}

ostream& Reconfiguration_manager_base::Scheduler_get_output_stream(void) {
	return *fout;
}

void Reconfiguration_manager_base::Scheduler_display_task_state_table(void) const {
	display_task_state_table();
}

void Reconfiguration_manager_base::Scheduler_display_rz_current_module_table(void) const{
	display_rz_current_module_table();
}

Task_to_schedule_interface* Reconfiguration_manager_base::Scheduler_get_task_to_schedule_ptr(int taskID) {
	return modules_table.at(taskID);
}


/**********************************
 *      Packet trace methods      *
 **********************************/

//void Reconfiguration_manager_base::set_icap_reconf_task(int hw_unit, int id) {
//	trace_packet->icap_status[hw_unit] = ((sc_bigint<TRACE_LV_SIZE>)ICAP_RECONF_T_ASCII << (modules_table.at(id)->getName().size() * 8))
//								+ Utils::string_to_ascii(modules_table.at(id)->getName());
//}

//void Reconfiguration_manager_base::set_icap_idle(int hw_unit) {
//	trace_packet->icap_status[hw_unit] = IDLE_ASCII;
//}
//
//void Reconfiguration_manager_base::set_software_loader_config_task(int sw_unit, int id) {
//	trace_packet->software_loader_status[sw_unit] = ((sc_bigint<TRACE_LV_SIZE>)ICAP_RECONF_T_ASCII << (modules_table.at(id)->getName().size() * 8))
//								+ Utils::string_to_ascii(modules_table.at(id)->getName());
//}

//void Reconfiguration_manager_base::update_processor_occupation_rate(void) {
//	int nbProcRunning = 0;
//	int nbProc = 0;
//	for(int i = 0; i < (int) rz_table.size(); i++) {
//		if(rz_table.at(i)->get_implementation_type() == SOFT) {
//			nbProc++;
//			if(map_current_module.find(rz_table.at(i)) != map_current_module.end()) nbProcRunning++;
//		}
//	}
//	trace_packet->fpga_processor_occupation_rate = (double)nbProcRunning / nbProc * 100.0;
//}

void Reconfiguration_manager_base::set_software_loader_idle(int sw_unit) {
	//trace_packet->software_loader_status[sw_unit] = IDLE_ASCII;
}

void Reconfiguration_manager_base::set_task_context_save(int task_id, int rz_id) {
	//trace_packet->task_status[task_id] = (((sc_bigint<TRACE_LV_SIZE>)CONTEXT_SAVE_ASCII << 24) << (rz_table.at(rz_id)->getName().size() * 8))
									//+ Utils::string_to_ascii(rz_table.at(rz_id)->getName());
}

void Reconfiguration_manager_base::set_task_context_restore(int task_id, int rz_id) {
	//trace_packet->task_status[task_id] = (((sc_bigint<TRACE_LV_SIZE>)CONTEXT_LOAD_ASCII << 24) << (rz_table.at(rz_id)->getName().size() * 8))
									//+ Utils::string_to_ascii(rz_table.at(rz_id)->getName());
}

void Reconfiguration_manager_base::set_task_state(int id, int hosting_rz_id, Task_state state) {

	task_state_table[id] = state;

	// Update running time of the task
	running_time_table[id].update_running_time(state);

	// Tache sur aucune RZ (soft ou hard)
	if(state == INACTIVE) {
		//trace_packet->task_status[id] = Utils::string_to_ascii("Inactive");

		// Update for rz state
		RZState actual_rz_state = rz_table.at(hosting_rz_id)->getRZState();
		if (actual_rz_state == RZ_BLANK)
			set_rz_state(hosting_rz_id, RZ_BLANK);
		else
			set_rz_state(hosting_rz_id, RZ_MAPPED);
	}
	else if(state == PREEMPTED_INACTIVE) {
		//trace_packet->task_status[id] = Utils::string_to_ascii("Preempted/Inactive");
		// No update for rz state

	}
	else if(state == QUEUED) {
		//trace_packet->task_status[id] = QUEUED_ASCII;
		// No update for rz state

	}
	else {
		//REM: associ  une RZ
		RZState rz_state;
		//sc_bigint<TRACE_LV_SIZE> ascii_word;

		switch(state) {
			case RUNNING:
				//rz_state = RZ_ACTIVE;
				rz_state = RZ_RUNNING;
				//ascii_word = RUNNING_ASCII;
				//ascii_word = ACTIVE_ASCII;
				//ascii_word = Utils::string_to_ascii("Running ");
				break;

			case CONFIG:
				rz_state = RZ_RECONFIG;
				//ascii_word = CONFIGURING_ASCII;
				break;

			case MAPPED:
				rz_state = RZ_MAPPED;
				//ascii_word = MAPPED_ASCII;
				break;

			case PREEMPTED_MAPPED:
				rz_state = RZ_MAPPED;
				//ascii_word = Utils::string_to_ascii("Preempted/Mapped ");
				break;

			case WAITING:
				if(CONSIDER_WAITING_STATE_AS_RUNNING_MODE) rz_state = RZ_RUNNING;
				else rz_state = RZ_ACTIVE;
				//ascii_word = Utils::string_to_ascii("Waiting ");
				break;

			case IDLE:
				rz_state = RZ_ACTIVE;
				//ascii_word = Utils::string_to_ascii("Idle ");
				break;

			case CONTEXT_SAVE:
				//rz_state = RZ_ACTIVE;
				rz_state = RZ_RUNNING;
				//ascii_word = Utils::string_to_ascii("Context save");
				break;

			case CONTEXT_RESTORE:
				//rz_state = RZ_ACTIVE;
				rz_state = RZ_RUNNING;
				//ascii_word = Utils::string_to_ascii("Context restore");
				break;

			default:
				rz_state = RZ_BLANK;
				//ascii_word = NULL_ASCII;
		}

		if (modules_table.at(id)->is_static() == false) {
			
			//REM: trace l'tat de la RZ
			set_rz_state(hosting_rz_id, rz_state);

			//REM: trace l'tat de la tache
			//trace_packet->task_status[id] = (((sc_bigint<TRACE_LV_SIZE>)ascii_word << 24) << (rz_table.at(hosting_rz_id)->getName().size() * 8))
									//+ Utils::string_to_ascii(rz_table.at(hosting_rz_id)->getName());
		}
		else {

			string wnone = "Static";
			//trace_packet->task_status[id] = (((sc_bigint<TRACE_LV_SIZE>)ascii_word << 24) << (wnone.size() * 8))
									//+ Utils::string_to_ascii(wnone);

		}
	}

	// Event to Monitoring
	notifyUpdateTaskStateEvent();
}

int Reconfiguration_manager_base::get_hosting_rz_id(int task_id) {
	int rz_id = -1;

	if (modules_table.at(task_id)->is_static() == false) {

		for(int i = 0; i < (int) map_current_module.size(); i++) {
			if(map_current_module[rz_table[i]] == task_id) rz_id = i;
		}

		if(rz_id == -1) {
			cerr << rs_time_stamp() << ": Cannot retrieve RZ ID... (task " << modules_table.at(task_id)->getName() << " : " << task_id << ")" << endl;
		}
	}
	
	return rz_id;
}

void Reconfiguration_manager_base::set_rz_state(int rz_id, RZState state) {
//
//	if(state == RZ_BLANK) trace_packet->rz_status[rz_id] = RZ_BLANK_ASCII;
//	else {
//
//		sc_bigint<TRACE_LV_SIZE> ascii_word;
//		switch(state) {
//			case RZ_RUNNING: ascii_word = RZ_RUNNING_T_ASCII; break;
//			case RZ_ACTIVE: ascii_word = ACTIVE_ASCII; break;
//			case RZ_RECONFIG: ascii_word = ICAP_RECONF_T_ASCII; break;
//			case RZ_MAPPED: ascii_word = RZ_MAPPED_T_ASCII; break;
//			default: ascii_word = NULL_ASCII;
		}
//
//		trace_packet->rz_status[rz_id] = ((sc_bigint<TRACE_LV_SIZE>)ascii_word << (rz_table.at(rz_id)->getCurrentImplementation().getFullName().size() * 8))
//			+ Utils::string_to_ascii(rz_table.at(rz_id)->getCurrentImplementation().getFullName());
//	}
//
//	update_occupation_rate(rz_id, state);
//}

void Reconfiguration_manager_base::set_scheduler_idle(void) {
	//trace_packet->scheduler_status = IDLE_ASCII;
}

//void Reconfiguration_manager_base::set_scheduler_active(int taskID) {
//	trace_packet->scheduler_status = ((sc_bigint<TRACE_LV_SIZE>)SCHEDULER_ACTIVE_ASCII << (modules_table.at(taskID)->getName().size() * 8))
//								+ Utils::string_to_ascii(modules_table.at(taskID)->getName());
//}


//void Reconfiguration_manager_base::update_occupation_rate(int rz_id, RZState newState) {
//	
//	rz_table.at(rz_id)->setRZState(newState);
//
//	// Event to Monitoring
//	notifyUpdateRZStateEvent();
//}

//void Reconfiguration_manager_base::occupation_rate_update_thread(void) {
//
//	rs_time sampling_period(200, SC_US);
//	rs_time icap_busy_time(RS_ZERO_TIME);
//	rs_time loader_busy_time(RS_ZERO_TIME);
//
//	WindowFunction::setWindowFunctionPeriod(TRACE_WINDOW_FUNCTION_LENGTH);
//	WindowFunction::setAnalogTraceSamplingPeriod(sampling_period);
//
//	while(true) {
//
//		//wait(sampling_period);
//
//		// ICAP occupation rate
//		for(int i = 0; i < NB_HW_RECONFIGURATION_UNITS; i++) {
//			trace_packet->icap_occupation_rate[i] = hw_reconfiguration_units[i].getOccupationRateOverLastWindow();
//		}
//
//		// Software loader occupation rate
//		for(int i = 0; i < NB_SW_RECONFIGURATION_UNITS; i++) {
//			trace_packet->software_loader_occupation_rate[i] = sw_reconfiguration_units[i].getOccupationRateOverLastWindow();
//		}
//
//		// RZs occupation rates
//		for(int i = 0; i < (int) rz_table.size(); i++) {
//			trace_packet->rz_occupation_rate[i] = rz_table.at(i)->getStateOccupationRateOverLastWindow(RZ_ACTIVE);
//			trace_packet->rz_running_rate[i] = rz_table.at(i)->getStateOccupationRateOverLastWindow(RZ_RUNNING);
//			trace_packet->rz_mapped_rate[i] = rz_table.at(i)->getStateOccupationRateOverLastWindow(RZ_MAPPED);
//			trace_packet->rz_config_rate[i] = rz_table.at(i)->getStateOccupationRateOverLastWindow(RZ_RECONFIG);
//			trace_packet->rz_blank_rate[i] = rz_table.at(i)->getStateOccupationRateOverLastWindow(RZ_BLANK);
//			
//			rz_table.at(i)->resetWindow();
//		}
//
//		// FPGA processor occupation rate
//		//update_processor_occupation_rate();
//	}
//}

//REM: Mise  jour de l'utilisation des ressources des RZ HARD uniquement
//void Reconfiguration_manager_base::update_resources_trace(int rz_id, int task_id) {
//
//	Resources taskResources;
//
//	// Search for HW task by name
//	for(int i = 0; i < (int) task_table.size(); i++) {
//		if(task_table.at(i).get_task_ID().compare(modules_table.at(task_id)->get_current_implementation_fullname()) == 0) {
//			taskResources = task_table.at(i).get_resources();
//		}
//	}
//
//	// Update trace
//	//REM: first : nom de la ressource (exemple: SLICE), second : nombre de ressource (SLICE) sur la RZ
//	for(map<string, int>::iterator it = rz_table.at(rz_id)->get_resources_ptr()->getResourceMap().begin(); it != rz_table.at(rz_id)->get_resources_ptr()->getResourceMap().end(); it++) {
//		if(it->second != 0) {
//			trace_packet->rz_resource_occupation_rate[rz_id][it->first] = (double)taskResources.getResourceCount(it->first) / (double) it->second * 100.0;
//		} else {
//			trace_packet->rz_resource_occupation_rate[rz_id][it->first] = 0;
//		}
//	}
//
//	//REM: NB de ressources reconfigurables utilis  cet instant
//	//REM: RZ1 120 Slices, T1 : 38 SLices
//	//REM: RZ2 : 200 SLICES, T2:144 Slices
//	//REM: Aggregate : 38 + 144 = 182
//	map<string, double> aggregate_resource_count;
//	for(int i = 0; i < (int) rz_table.size(); i++) {
//		for(map<string, int>::iterator it = rz_table.at(i)->get_resources_ptr()->getResourceMap().begin(); it != rz_table.at(i)->get_resources_ptr()->getResourceMap().end(); it++) {
//			aggregate_resource_count[it->first] += trace_packet->rz_resource_occupation_rate[i][it->first] / 100.0 * (double) rz_table.at(i)->get_resources_ptr()->getResourceCount(it->first);
//		}
//	}
//
//	//REM: result = 182 / (Nb slice sur le fpga)
//	for(map<string, double>::iterator it = aggregate_resource_count.begin(); it != aggregate_resource_count.end(); it++) {
//		trace_packet->fpga_resource_occupation_rate[it->first] = it->second / (double)fpga->get_resources().getResourceCount(it->first) * 100.0;
//	}
//}

//void Reconfiguration_manager_base::set_hyperperiod_status(int hpid) {
//	string hpid_str(Utils::itoa(hpid));
//	trace_packet->hyperperiod_status = ((sc_bigint<TRACE_LV_SIZE>) Utils::string_to_ascii("Hyperperiod ") << (hpid_str.size() * 8))
//								+ Utils::string_to_ascii(hpid_str);
//}


//void Reconfiguration_manager_base::display_occupation_rates(void) {
//
//	//for(int i = 0; i < NB_HW_RECONFIGURATION_UNITS; i++) FOUT << "HW reconfiguration unit " << i << " occupation rate: " << hw_reconfiguration_units[i].getBusyTime() / rs_time_stamp() * 100 << " %" << endl;
//	//for(int i = 0; i < NB_SW_RECONFIGURATION_UNITS; i++) FOUT << "SW reconfiguration unit " << i << " occupation rate: " << sw_reconfiguration_units[i].getBusyTime() / rs_time_stamp() * 100 << " %" << endl;
//
//	for(int i = 0; i < (int) applicationVector.size(); i++) {
//		//FOUT << "Achieved global QoS for application " << applicationVector.at(i).getName() << ": " << applicationVector.at(i).getAchievedQoS() << " %" <<  endl;
//		//FOUT << "Minimal QoS on an hyperperiod for application " << applicationVector.at(i).getName() << ": " << applicationVector.at(i).getMinimalQoS() << " %" <<  endl;
//	}
//
//	//FOUT << "Energy consumption: " << total_energy_consumption << " mJ" << endl;
//	
//	for(int i = 0; i < (int) rz_table.size(); i++) {
//		//FOUT << endl << "---- " << rz_table.at(i)->getName() << " ----" << endl;
//		for(int j = 0; j < NB_RZ_STATES; j++) 
//			//FOUT << "Time " << RZState_string[j] << ":\t\t" << rz_table.at(i)->getOccupationTime((RZState)j) << " (" << rz_table.at(i)->getOccupationTime((RZState)j) / rs_time_stamp() * 100 << " %)" << endl;
//
//		// Task execution time
//		for(int j = 0; j < (int) rz_table.at(i)->getTaskImplementationVector().size(); j++)
//			//FOUT << "Task " << rz_table.at(i)->getTaskImplementationVector().at(j).getFullName() << endl << (rz_table.at(i)->getTaskImplementationVector().at(j).getOccupationTime(RZ_RUNNING) / rz_table.at(i)->getOccupationTime(RZ_RUNNING) * 100) << endl;
//	}
//}

void Reconfiguration_manager_base::update_occupation_rate_end_simulation(void) {
	for(int i = 0; i < (int) rz_table.size(); i++);// update_occupation_rate(i, RZ_BLANK);
}

void Reconfiguration_manager_base::check_deadline(int task_id) {
	
	//if(rs_time_stamp() > modules_table[task_id]->get_deadline_time()) {
	if(rs_time_stamp() > deadlines_table[task_id]) {

		//FOUT << rs_time_stamp() << ": Constraint not met for task " << modules_table[task_id]->getName() << endl;
		//FOUT << rs_time_stamp() << ": Deadline: " << modules_table[task_id]->get_deadline_time() << endl;
		//FOUT << rs_time_stamp() << ": Deadline: " << deadlines_table[task_id] << endl;

		applicationVector.at(getApplicationIncludingTask(task_id)).incrementMissedDeadlines(modules_table[task_id]->getName());

	} else applicationVector.at(getApplicationIncludingTask(task_id)).incrementPassedDeadlines();
}

//void Reconfiguration_manager_base::generateSimulationResultFile(void) {
//
//	string filename("result.txt");
//	ofstream file(filename.c_str());
//
//	if(file) {
//
//		file << "Simulation: " << (simulationOK ? "Passed" : "Failed") << endl << endl;
//		for(int i = 0; i < NB_HW_RECONFIGURATION_UNITS; i++) file << "HW reconfiguration unit " << i << " occupation rate: " << hw_reconfiguration_units[i].getBusyTime() / rs_time_stamp() * 100 << endl << endl;
//		for(int i = 0; i < NB_SW_RECONFIGURATION_UNITS; i++) file << "SW reconfiguration unit " << i << " occupation rate: " << sw_reconfiguration_units[i].getBusyTime() / rs_time_stamp() * 100 << " %" << endl;
//		for(int i = 0; i < (int) applicationVector.size(); i++) {
//			file << "Achieved global QoS for application " << applicationVector.at(i).getName() << ": " << applicationVector.at(i).getAchievedQoS() << " %" <<  endl;
//			file << "Minimal QoS on an hyperperiod for application " << applicationVector.at(i).getName() << ": " << applicationVector.at(i).getMinimalQoS() << " %" <<  endl;
//		}
//		file << "Energy consumption: " << total_energy_consumption << " mJ" << endl << endl;
//		
//		for(int i = 0; i < (int) rz_table.size(); i++) {
//			file << "-----------------------------------------" << endl;
//			if(rz_table[i]->get_implementation_type() == HARD) file << "RZ " << rz_table[i]->getName() << endl;
//			else if(rz_table[i]->get_implementation_type() == SOFT) file << "Processor " << rz_table[i]->getName() << endl;
//			
//			for(int j = 0; j < NB_RZ_STATES; j++) file << RZState_string[j] << ": " << rz_table.at(i)->getOccupationTime((RZState)j) / rs_time_stamp() * 100 << endl;
//			file << endl;
//			
//			// Task execution time
//			for(int j = 0; j < (int) rz_table.at(i)->getTaskImplementationVector().size(); j++)
//				file << "Task " << rz_table.at(i)->getTaskImplementationVector().at(j).getFullName() << endl << rz_table.at(i)->getTaskImplementationVector().at(j).getOccupationTime(RZ_RUNNING) / rz_table.at(i)->getOccupationTime(RZ_RUNNING) * 100 << endl;
//			file << "end" << endl << endl;
//		}
//
//		file.close();
//	} else {
//		cerr << "ERROR: Manager unable to generate simulation result file..." << endl;
//	}
//
//}

bool Reconfiguration_manager_base::is_task_preempted(int id) const {
	for(int i = 0; i < (int) preempted_tasks_list.size(); i++) {
		if(preempted_tasks_list.at(i) == id) return true;
	}

	return false;
}

bool Reconfiguration_manager_base::is_task_preempted(string taskname) const {
	for(int i = 0; i < (int) modules_table.size(); i++) {
		if(modules_table.at(i)->getName().compare(taskname) == 0) return is_task_preempted(i);
	}

	cerr << "ERROR: Unable to retrieve task name... Exiting!" << endl;
	exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
}

void Reconfiguration_manager_base::add_task_to_preemption_list(int id) {
	preempted_tasks_list.push_back(id);
}

void Reconfiguration_manager_base::remove_task_from_preemption_list(int id) {
	for(int i = 0; i < (int) preempted_tasks_list.size(); i++) {
		if(preempted_tasks_list.at(i) == id) {
			preempted_tasks_list.erase(preempted_tasks_list.begin() + i);
			return;
		}
	}
}

//REM: appel  la fin du main
void Reconfiguration_manager_base::enable_debug_signals_trace(void) {
	//trace_packet->trace_debug_signals = true;
}

void Reconfiguration_manager_base::activate_trace(void) {
	//sc_trace(tf, *trace_packet, (string) name);
}

//void Reconfiguration_manager_base::generateCSVResultFile(string filename) {
//
//	const string CSV_SEPARATOR("; ");
//
//	ofstream file(filename.c_str());
//
//	if(file) {
//
//		//
//		// Header
//		//
//		time_t rawtime;
//		time(&rawtime);
//		struct tm * timeinfo = localtime(&rawtime);
//		
//		file << "CSV file generated on " << asctime(timeinfo);
//		file << "Achieved simulation time: " << rs_time_stamp() << endl;
//		file << "Minimum simulation time: " << Application::getMinimumSimulationTime() << endl;
//		file << "Application offset: " << Application::getGlobalOffset() << endl;
//		file << "Application hyperperiod: " << Application::getGlobalHypeperiod() << endl << endl;
//
//
//		//
//		// RZs
//		//
//
//		// RZ(s) occupation rates
//		int nb_rz = 0;
//		for(int i = 0; i < (int) rz_table.size(); i++) {
//			if(rz_table.at(i)->get_implementation_type() == HARD) {
//				if(nb_rz == 0) file << "===== RZ(s) occupation rates (in % of the simulation time) =====" << endl;
//				file << CSV_SEPARATOR << rz_table.at(i)->getName();
//				nb_rz++;
//			}
//		}
//		file << endl;
//
//		if(nb_rz != 0) {
//
//			// RZ states
//			for(int i = 0; i < NB_RZ_STATES; i++) {
//				file << RZState_string[i];
//				for(int j = 0; j < (int) rz_table.size(); j++) {
//					if(rz_table.at(j)->get_implementation_type() == HARD) {
//						file << CSV_SEPARATOR << rz_table.at(j)->getOccupationTime((RZState) i) / rs_time_stamp() * 100;
//					}
//				}
//				file << endl;
//			}
//			
//			// RZs running tasks
//			for(int i = 0; i < (int) modules_table.size(); i++) {
//
//				if(GENERATE_ZERO_FILLED_LINE_IN_CSV || task_runs_on_implementation_type(i, HARD)) {
//					file << modules_table.at(i)->getName();
//					for(int j = 0; j < (int) rz_table.size(); j++) {
//						if(rz_table.at(j)->get_implementation_type() == HARD) {
//							rs_time aggregate(RS_ZERO_TIME);
//							for(int k = 0; k < (int) rz_table.at(j)->getTaskImplementationVector().size(); k++) {
//								if(rz_table.at(j)->getTaskImplementationVector().at(k).getTaskName().compare(modules_table.at(i)->getName()) == 0)
//									aggregate += rz_table.at(j)->getTaskImplementationVector().at(k).getOccupationTime(RZ_RUNNING);
//							}
//							file << CSV_SEPARATOR << aggregate / rs_time_stamp() * 100;
//						}	
//					}
//
//					file << endl;
//
//					for(int l = 0; l < (int) modules_table.at(i)->get_implementation_vector().size(); l++) {
//
//						if(modules_table.at(i)->get_implementation_vector().at(l).get_implementation_type() == HARD) {
//							file << modules_table.at(i)->get_implementation_vector().at(l).get_name();
//
//							for(int j = 0; j < (int) rz_table.size(); j++) {
//								if(rz_table.at(j)->get_implementation_type() == HARD) {
//									rs_time time(RS_ZERO_TIME);
//									for(int k = 0; k < (int) rz_table.at(j)->getTaskImplementationVector().size(); k++) {
//										if(rz_table.at(j)->getTaskImplementationVector().at(k).getTaskName().compare(modules_table.at(i)->getName()) == 0 &&
//											rz_table.at(j)->getTaskImplementationVector().at(k).getImplementationName().compare(modules_table.at(i)->get_implementation_vector().at(l).get_name()) == 0)
//											time = rz_table.at(j)->getTaskImplementationVector().at(k).getOccupationTime(RZ_RUNNING);
//									}
//									file << CSV_SEPARATOR << time / rs_time_stamp() * 100;
//								}
//							}
//
//							file << endl;
//						}
//					}
//				}
//			}
//
//			file << endl;
//
//			if(!GENERATE_ZERO_FILLED_LINE_IN_CSV) {
//				int nb_tasks_added = 0;
//				for(int i = 0; i < (int) modules_table.size(); i++) {
//					if(!task_runs_on_implementation_type(i, HARD)) {
//						if(nb_tasks_added == 0) file << "Tasks not implemented in hardware" << endl;
//						file << modules_table.at(i)->getName() << endl;
//						nb_tasks_added++;
//					}
//				}
//			}
//
//			file << endl;
//
//			// RZ(s) properties (toto)
//			file << "===== RZ(s) properties =====" << endl;
//			for(int i = 0; i < (int) rz_table.size(); i++) {
//				if(rz_table.at(i)->get_implementation_type() == HARD) {
//					file << CSV_SEPARATOR << rz_table.at(i)->getName() << endl;
//					map<string, int> rzResources(rz_table.at(i)->get_resources_ptr()->getResourceMap());
//					for(map<string, int>::const_iterator it = rzResources.begin(); it != rzResources.end(); it++) {
//						file << it->first << CSV_SEPARATOR << it->second << endl;  
//					}
//				}
//			}
//			file << endl;
//		}
//
//		//
//		// Processors
//		//
//		int nb_proc = 0;
//		for(int i = 0; i < (int) rz_table.size(); i++) {
//			if(rz_table.at(i)->get_implementation_type() == SOFT) {
//				if(nb_proc == 0) file << "===== Processor(s) occupation rates (in % of the simulation time) =====" << endl;
//				file << CSV_SEPARATOR << rz_table.at(i)->getName();
//				nb_proc++;
//			}
//		}
//		file << endl;
//
//		if(nb_proc != 0) {
//
//			// Processor states
//			for(int i = 0; i < NB_RZ_STATES; i++) {
//				file << RZState_string[i];
//				for(int j = 0; j < (int) rz_table.size(); j++) {
//					if(rz_table.at(j)->get_implementation_type() == SOFT) {
//						file << CSV_SEPARATOR << rz_table.at(j)->getOccupationTime((RZState) i) / rs_time_stamp() * 100;
//					}
//				}
//				file << endl;
//			}
//			
//			// Processors running tasks
//			for(int i = 0; i < (int) modules_table.size(); i++) {
//
//				if(GENERATE_ZERO_FILLED_LINE_IN_CSV || task_runs_on_implementation_type(i, SOFT)) {
//					file << modules_table.at(i)->getName();
//					for(int j = 0; j < (int) rz_table.size(); j++) {
//						if(rz_table.at(j)->get_implementation_type() == SOFT) {
//							rs_time aggregate(RS_ZERO_TIME);
//							for(int k = 0; k < (int) rz_table.at(j)->getTaskImplementationVector().size(); k++) {
//								if(rz_table.at(j)->getTaskImplementationVector().at(k).getTaskName().compare(modules_table.at(i)->getName()) == 0)
//									aggregate += rz_table.at(j)->getTaskImplementationVector().at(k).getOccupationTime(RZ_RUNNING);
//							}
//							file << CSV_SEPARATOR << aggregate / rs_time_stamp() * 100;
//						}	
//					}
//
//					file << endl;
//
//					for(int l = 0; l < (int) modules_table.at(i)->get_implementation_vector().size(); l++) {
//
//						if(modules_table.at(i)->get_implementation_vector().at(l).get_implementation_type() == SOFT) {
//							file << modules_table.at(i)->get_implementation_vector().at(l).get_name();
//
//							for(int j = 0; j < (int) rz_table.size(); j++) {
//								if(rz_table.at(j)->get_implementation_type() == SOFT) {
//									rs_time time(RS_ZERO_TIME);
//									for(int k = 0; k < (int) rz_table.at(j)->getTaskImplementationVector().size(); k++) {
//										if(rz_table.at(j)->getTaskImplementationVector().at(k).getTaskName().compare(modules_table.at(i)->getName()) == 0 &&
//											rz_table.at(j)->getTaskImplementationVector().at(k).getImplementationName().compare(modules_table.at(i)->get_implementation_vector().at(l).get_name()) == 0)
//											time = rz_table.at(j)->getTaskImplementationVector().at(k).getOccupationTime(RZ_RUNNING);
//									}
//									file << CSV_SEPARATOR << time / rs_time_stamp() * 100;
//								}
//							}
//
//							file << endl;
//						}
//					}
//				}
//			}
//
//			file << endl;
//
//			if(!GENERATE_ZERO_FILLED_LINE_IN_CSV) {
//				int nb_tasks_added = 0;
//				for(int i = 0; i < (int) modules_table.size(); i++) {
//					if(!task_runs_on_implementation_type(i, SOFT)) {
//						if(nb_tasks_added == 0) file << "Tasks not implemented in software" << endl;
//						file << modules_table.at(i)->getName() << endl;
//						nb_tasks_added++;
//					}
//				}
//			}
//		}
//
//		//
//		// Misc
//		//
//		file << endl;
//		file << "===== Reconfiguration units occupation rates (in % of the simulation time) =====" << endl;
//		for(int i = 0; i < NB_HW_RECONFIGURATION_UNITS; i++) file << "HW reconfiguration unit " << i << CSV_SEPARATOR << hw_reconfiguration_units[i].getBusyTime() / rs_time_stamp() * 100 << endl;
//		for(int i = 0; i < NB_SW_RECONFIGURATION_UNITS; i++) file << "SW reconfiguration unit " << i << CSV_SEPARATOR << sw_reconfiguration_units[i].getBusyTime() / rs_time_stamp() * 100 << endl;
//
//		file << endl;
//		file << "===== Scheduler =====" << endl;
//		file << "Occupation rate (in % of the simulation time)" << CSV_SEPARATOR << scheduler_busy_time / rs_time_stamp() * 100 << endl;
//		file << "Energy consumption (mJ)" << CSV_SEPARATOR << total_energy_consumption << endl;
//
//		file << endl;
//		file << "===== Achieved QoS factor (in %) =====" << endl;
//		for(int i = 0; i < (int) applicationVector.size(); i++)
//			file << applicationVector.at(i).getName() << CSV_SEPARATOR << applicationVector.at(i).getAchievedQoS() <<  endl;
//
//		file << endl;
//		file << "===== Minimal QoS factor on an hyperperiod (in %) =====" << endl;
//		for(int i = 0; i < (int) applicationVector.size(); i++)
//			file << applicationVector.at(i).getName() << CSV_SEPARATOR << applicationVector.at(i).getMinimalQoS() <<  endl;
//
//		// Check if any application has missed deadlines
//		bool applicationFailed(false);
//		for(int i = 0; i < (int) applicationVector.size(); i++) {
//			if(applicationVector.at(i).hasMissedDeadlines()) {
//				applicationFailed = true;
//				break;
//			}
//		}
//		
//		if(applicationFailed) {
//			file << endl;
//			file << "===== Missed deadlines (last hyperperiod) =====" << endl;
//			for(int i = 0; i < (int) applicationVector.size(); i++) {
//				if(applicationVector.at(i).hasMissedDeadlines()) {
//					file << applicationVector.at(i).getName() << CSV_SEPARATOR << endl;
//					for(int j = 0; j < (int) applicationVector.at(i).getMissedDeadlinesVector().size(); j++) {
//						pair<string, rs_time> missed_deadline(applicationVector.at(i).getMissedDeadlinesVector().at(j));
//						file << missed_deadline.first << CSV_SEPARATOR << missed_deadline.second << CSV_SEPARATOR << endl;
//					}
//				}
//			}
//		}
//			
//
//		//
//		// Static modules
//		//
//		int nb_static = 0;
//		for(int i = 0; i < (int) modules_table.size(); i++) {
//			if(modules_table.at(i)->is_static()) {
//				if(nb_static == 0) file << endl << "===== Static modules =====" << endl;
//				file << modules_table.at(i)->getName() << endl;
//				nb_static++;
//			}
//		}
//
//		file.close();
//	} else {
//		cerr << "ERROR: Manager unable to generate CSV result file..." << endl;
//	}
//}

//REM: sert uniquement  la gnration du CSV
//REM: permet de savoir que la tche tourne ou pas sur cette impl.
bool Reconfiguration_manager_base::task_runs_on_implementation_type(int taskID, PEImplementation impl) {
	
	rs_time aggregate_runtime(RS_ZERO_TIME);

	for(int i = 0; i < (int) rz_table.size(); i++) {
		if(rz_table.at(i)->get_implementation_type() == impl) {
			for(int j = 0; j < (int) rz_table.at(i)->getTaskImplementationVector().size(); j++) {
				if(rz_table.at(i)->getTaskImplementationVector().at(j).getTaskName().compare(modules_table.at(taskID)->getName()) == 0)
					aggregate_runtime += rz_table.at(i)->getTaskImplementationVector().at(j).getOccupationTime(RZ_RUNNING);
			}
		}
	}
	
	return (aggregate_runtime != RS_ZERO_TIME);
}

//REM: fonction statique qui permet de changer le temps du scheduler en cours de simu
void Reconfiguration_manager_base::set_scheduler_execution_time(rs_time t) {
	SCHEDULER_EXECUTION_TIME = t;
}

void Reconfiguration_manager_base::set_generate_zeros_in_csv(bool gen) {
	GENERATE_ZERO_FILLED_LINE_IN_CSV = gen;
}

//REM: retourne l'id de l'appli qui contient la tache
int Reconfiguration_manager_base::getApplicationIncludingTask(int taskID) {
	int appID = -1;
	for(int i = 0; i < (int) applicationVector.size(); i++) {
		if(applicationVector.at(i).containsTask(taskID)) appID = i;
	}

	if(appID == -1) cerr << "ERROR: Unable to find the application instanciating task " << taskID << endl;

	return appID;
}

//vector<Application> * Reconfiguration_manager_base::getApplicationVectorPtr(void) {
//	return &applicationVector;
//}

FPGA* Reconfiguration_manager_base::getFPGAptr(void) {
	return fpga;
}

rs_time Reconfiguration_manager_base::getMaximumSimulationTime(void) {
	return MAXIMUM_SIMULATION_TIME;
}

// Monitoring

void Reconfiguration_manager_base::setMonitoringEvent(int threadNumber) {

//	for (int i=0; i<threadNumber; i++) {
//		monitoring_end_of_scheduling_event_list.push_back(new sc_event());
//		monitoring_start_of_scheduling_event_list.push_back(new sc_event());
//		monitoring_update_rz_state_event_list.push_back(new sc_event());
//		monitoring_update_task_state_event_list.push_back(new sc_event());
//	}
}

void Reconfiguration_manager_base::notifyEndOfSchedulingEventMonitoringEvent() {

//	for (int i=0; i<(int)monitoring_end_of_scheduling_event_list.size(); i++) {
//		monitoring_end_of_scheduling_event_list.at(i)->notify();
//	}
}

void Reconfiguration_manager_base::notifyStartOfSchedulingEventMonitoringEvent() {

//	for (int i=0; i<(int)monitoring_start_of_scheduling_event_list.size(); i++) {
//		monitoring_start_of_scheduling_event_list.at(i)->notify();
//	}
}

void Reconfiguration_manager_base::notifyUpdateRZStateEvent() {

//	for (int i=0; i<(int)monitoring_update_rz_state_event_list.size(); i++) {
//		monitoring_update_rz_state_event_list.at(i)->notify();
//	}
}

void Reconfiguration_manager_base::notifyUpdateTaskStateEvent() {

//	for (int i=0; i<(int)monitoring_update_task_state_event_list.size(); i++) {
//		monitoring_update_task_state_event_list.at(i)->notify();
//	}
}

// Monitoring Interface

//sc_event& Reconfiguration_manager_base::endOfSchedulingEvent(int id) {
//	return *monitoring_end_of_scheduling_event_list.at(id);
//}
//
//sc_event& Reconfiguration_manager_base::startOfSchedulingEvent(int id) {
//	return *monitoring_start_of_scheduling_event_list.at(id);
//}
//
//sc_event& Reconfiguration_manager_base::updateRZStateEvent(int id) {
//	return *monitoring_update_rz_state_event_list.at(id);
//}
//
//sc_event& Reconfiguration_manager_base::updateTaskStateEvent(int id) {
//	return *monitoring_update_task_state_event_list.at(id);
//}

// Common QoS & Monitoring Interface

//sc_trace_file* Reconfiguration_manager_base::getTraceFile() const {
//	return tf;
//}

Scheduler_interface& Reconfiguration_manager_base::getShedulerInterface() {
	return *this;
}

// QoS Interface

//vector<Application>& Reconfiguration_manager_base::getApplications() {
//	return applicationVector;
//}

