/**
 * UNS - CNRS
 * Copyright 2013 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit� de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: licensing@sattse.com
 *
 * @file    recosim_main.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0-build:2014/06/04
 * @date    January 24, 2016 3:10:00 PM CET
 * @section DESCRIPTION
 *
 *          Main class for xc6vlx240t device (Virtex6)
 *          Instantiates the transcoder and launches the simulation.
 *          The simulation produces a log file and a VCD file.
*/

#define FORTRESS_CONFIGURATION Config_1


#include "user_parameters.h"
#include "simulation_controller.h"
#include "reconfigurable_zone.h"
#include "testbench_in.h"
#include "testbench_out.h"
#include "module.h"
#include "reconf_threads.h"
#include "configuration_file_parser.h"
#include "reconfiguration_manager.h"
#include "generation_helper.h"
#include "testbench_algorithms.h"
#include "module_algorithms.h"
#include "application.h"
#include "user_monitoring.h"

#include "fpga.h"
#include "task.h"

#include <sstream>
#include <fstream>
#include <string>
#include <map>

bool Utils::VERBOSE_MODE					= false;

template void user_algorithm_SW<1, 1>(User_algorithm_interface<1, 1>&);
template void user_algorithm_HW<1, 1>(User_algorithm_interface<1, 1>&);


int sc_main(int argc, char* argv[]) {

	string simulationName;

	if(argc == 2) {
		cout << "RZ file to parse: " << argv[1] << endl;
		simulationName = "";
	} else if(argc == 3) {
		cout << "RZ file to parse: " << argv[1] << endl;
		cout << "Simulation name: " << argv[2] << endl;
		simulationName = argv[2];
	} else {
		cout << "Need parameters. Usage: ./exec path_to_RZ_file simulation_name" << endl;
		return -1;
	}

	string path(argv[1]);

	/* Log Files */
	string logname("../../solutions/solution");
	if(argc == 3) logname.append("_");
	logname.append(simulationName);
	if(argc == 3) logname.append(".log");
	ofstream fout(logname.c_str());
	cout << "Log file: " << logname << endl;

	/* VCD Trace Files */
	string tracename("../../solutions/solution");
	if(argc == 3) tracename.append("_");
	tracename.append(simulationName);
	sc_trace_file *tf = sc_create_vcd_trace_file(tracename.c_str());
	Simulation_controller::setOutputStreamPtr(&fout);

	/* RZ Properties */
	map<string, RZ_config*> rz_cfg_table;
	RZ_config::initRZConfigTable(rz_cfg_table);

	/* Module Instances Declaration*/
	// Application : app1.modulesystem
	Module<1, 1>* inst_T1[1];
	Module<1, 1>* inst_T2[1];
	Module<1, 1>* inst_T3[1];
	TestbenchOut<1>* inst_TBout[1];
	TestbenchIn<1>* inst_TBin[1];

	/* Read RZ Configuration File */
	vector<RZ *> rz_table;
	vector<Task> task_table;
	FPGA* fpga = new FPGA();
	if(readRZConfigurationFile(path, rz_table, task_table, fpga) != 0) {
		cerr << "ERROR: Unable to read RZ configuration file, exiting..." << endl;
		return -1;
	}
	Reconfiguration_manager manager("Reconfiguration_manager", rz_table, task_table, WAITING_QUEUE_HANDLER_REF(AMAP_EDF), fpga, fout, tf, rz_cfg_table);

	/************************************************
	 * Application File : app1.modulesystem
	 */

	/* Config Application Init */
	string config_app_inst_name[1];
	int config_app_QoS[1];
	config_app_inst_name[0] = "app1/App_0";
	config_app_QoS[0] = 100;


	for(int i = 0; i < 1; i++) {
		/* Module Instances */
		inst_T1[i] = new Module<1, 1> ("T1", reconf_thread_standard, fout, tf, config_app_inst_name[i]);
		inst_T2[i] = new Module<1, 1> ("T2", reconf_thread_standard, fout, tf, config_app_inst_name[i]);
		inst_T3[i] = new Module<1, 1> ("T3", reconf_thread_standard, fout, tf, config_app_inst_name[i]);
		inst_TBout[i] = new TestbenchOut<1> ("TBout", testbench_out_thread<1>, fout, tf);
		inst_TBout[i]->set_period(sc_time(10000, SC_US));
		inst_TBin[i] = new TestbenchIn<1> ("TBin", testbench_in_thread<1>, fout, tf);

		/* Add TestBench in Manager */

		/* Create a sub-modules table */
		vector<Manager_interface *> modules_table;
		modules_table.push_back(inst_T1[i]);
		modules_table.push_back(inst_T2[i]);
		modules_table.push_back(inst_T3[i]);

		/* Following Node modules */
		vector<int> channel_connection_vector;

		channel_connection_vector.push_back(get_module_position_in_vector(inst_T2[i], modules_table));
		inst_T1[i]->set_following_modules_list(channel_connection_vector);
		inst_T1[i]->add_following_module_connection_name("C2");

		channel_connection_vector.clear();
		channel_connection_vector.push_back(get_module_position_in_vector(inst_T3[i], modules_table));
		inst_T2[i]->set_following_modules_list(channel_connection_vector);
		inst_T2[i]->add_following_module_connection_name("C3");

		channel_connection_vector.clear();
		channel_connection_vector.push_back(-1);
		inst_T3[i]->set_following_modules_list(channel_connection_vector);
		inst_T3[i]->add_following_module_connection_name("C4");

		/* Preceding Node modules */
		vector<int> preceding_modules_list;
		preceding_modules_list.push_back(-1); // Entry point
		inst_T1[i]->set_preceding_modules_list(preceding_modules_list);
		inst_T1[i]->add_preceding_module_connection_name("C1");

		preceding_modules_list.clear();
		preceding_modules_list.push_back(get_module_position_in_vector(inst_T1[i], modules_table));
		inst_T2[i]->set_preceding_modules_list(preceding_modules_list);
		inst_T2[i]->add_preceding_module_connection_name("C2");

		preceding_modules_list.clear();
		preceding_modules_list.push_back(get_module_position_in_vector(inst_T2[i], modules_table));
		inst_T3[i]->set_preceding_modules_list(preceding_modules_list);
		inst_T3[i]->add_preceding_module_connection_name("C3");

		inst_TBin[i]->add_preceding_module_connection_name("C4");

		/* Following Bench modules */
		channel_connection_vector.clear();
		channel_connection_vector.push_back(get_module_position_in_vector(inst_T1[i], modules_table));
		inst_TBout[i]->set_following_modules_list(channel_connection_vector);
		inst_TBout[i]->add_following_module_connection_name("C1");
		channel_connection_vector.clear();

		/* Module Hw/Sw Implementations */
		vector<ModuleImplementation> impl_T1;
		impl_T1.push_back(ModuleImplementation("Sw1", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, sc_time(3000, SC_US), sc_time(0.0, SC_US)));
		impl_T1.back().set_use_context_switch_mode(true);
		impl_T1.back().set_nb_preemption_points(0);
		impl_T1.back().set_algorithm_name("user_algorithm_SW");
		vector<ModuleImplementation> impl_T2;
		impl_T2.push_back(ModuleImplementation("Hw1", HARD, (algorithm_thread_type)user_algorithm_HW<1, 1>, sc_time(2000, SC_US), sc_time(0.0, SC_US)));
		impl_T2.back().set_use_context_switch_mode(true);
		impl_T2.back().set_nb_preemption_points(0);
		impl_T2.back().set_frequency(0);
		impl_T2.back().set_p_idle(0.0);
		impl_T2.back().set_p_run(0.0);
		impl_T2.back().set_netlist_name("");
		impl_T2.back().set_algorithm_name("");
		impl_T2.push_back(ModuleImplementation("Sw1", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, sc_time(3000, SC_US), sc_time(0.0, SC_US)));
		impl_T2.back().set_use_context_switch_mode(true);
		impl_T2.back().set_nb_preemption_points(0);
		impl_T2.back().set_algorithm_name("user_algorithm_SW");
		vector<ModuleImplementation> impl_T3;
		impl_T3.push_back(ModuleImplementation("Sw1", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, sc_time(3000, SC_US), sc_time(0.0, SC_US)));
		impl_T3.back().set_use_context_switch_mode(true);
		impl_T3.back().set_nb_preemption_points(0);
		impl_T3.back().set_algorithm_name("user_algorithm_SW");

		/* Module Properties */
		inst_T1[i]->set_deadline_time(sc_time(10000, SC_US));
		inst_T1[i]->set_task_period(sc_time(10000, SC_US));
		inst_T1[i]->set_task_offset(sc_time(0, SC_US));
		inst_T1[i]->set_priority(0);
		inst_T1[i]->set_implementation_vector(impl_T1);
		inst_T2[i]->set_deadline_time(sc_time(10000, SC_US));
		inst_T2[i]->set_task_period(sc_time(10000, SC_US));
		inst_T2[i]->set_task_offset(sc_time(0, SC_US));
		inst_T2[i]->set_priority(0);
		inst_T2[i]->set_implementation_vector(impl_T2);
		inst_T3[i]->set_deadline_time(sc_time(10000, SC_US));
		inst_T3[i]->set_task_period(sc_time(10000, SC_US));
		inst_T3[i]->set_task_offset(sc_time(0, SC_US));
		inst_T3[i]->set_priority(0);
		inst_T3[i]->set_implementation_vector(impl_T3);

		/* Connection Properties */
		inst_T1[i]->set_begin_req_time("C2", sc_time(0.0, SC_US));
		inst_T1[i]->set_end_resp_time("C2", sc_time(0.0, SC_US));
		inst_T1[i]->set_packet_size("C2", 16);
		inst_T1[i]->add_interface(ModuleInterface("C2", "DEFAULT_IF", false));
		inst_T1[i]->set_end_req_time("C1", sc_time(0.0, SC_US));
		inst_T1[i]->set_begin_resp_time("C1", sc_time(0.0, SC_US));
		inst_T1[i]->set_packet_size("C1", 16);
		inst_T1[i]->add_interface(ModuleInterface("C1", "DEFAULT_IF", false));

		inst_T2[i]->set_begin_req_time("C3", sc_time(0.0, SC_US));
		inst_T2[i]->set_end_resp_time("C3", sc_time(0.0, SC_US));
		inst_T2[i]->set_packet_size("C3", 16);
		inst_T2[i]->add_interface(ModuleInterface("C3", "DEFAULT_IF", false));
		inst_T2[i]->set_end_req_time("C2", sc_time(0.0, SC_US));
		inst_T2[i]->set_begin_resp_time("C2", sc_time(0.0, SC_US));
		inst_T2[i]->set_packet_size("C2", 16);
		inst_T2[i]->add_interface(ModuleInterface("C2", "DEFAULT_IF", false));

		inst_T3[i]->set_begin_req_time("C4", sc_time(0.0, SC_US));
		inst_T3[i]->set_end_resp_time("C4", sc_time(0.0, SC_US));
		inst_T3[i]->set_packet_size("C4", 16);
		inst_T3[i]->add_interface(ModuleInterface("C4", "DEFAULT_IF", false));
		inst_T3[i]->set_end_req_time("C3", sc_time(0.0, SC_US));
		inst_T3[i]->set_begin_resp_time("C3", sc_time(0.0, SC_US));
		inst_T3[i]->set_packet_size("C3", 16);
		inst_T3[i]->add_interface(ModuleInterface("C3", "DEFAULT_IF", false));

		inst_TBout[i]->set_begin_req_time("C1",  sc_time(0.0, SC_US));
		inst_TBout[i]->set_end_resp_time("C1",  sc_time(0.0, SC_US));
		inst_TBout[i]->set_packet_size("C1", 16);

		inst_TBin[i]->set_end_req_time("C4", sc_time(0.0, SC_US));
		inst_TBin[i]->set_begin_resp_time("C4", sc_time(0.0, SC_US));
		inst_TBin[i]->set_packet_size("C4", 16);


		/* Module Connections */
		inst_T1[i]->initiator_socket.bind(inst_T2[i]->target_socket);
		inst_T2[i]->initiator_socket.bind(inst_T3[i]->target_socket);
		inst_T3[i]->initiator_socket.bind(inst_TBin[i]->input_target_socket);
		inst_TBout[i]->output_initiator_socket.bind(inst_T1[i]->target_socket);
		inst_TBout[i]->manager_initiator_socket.bind(manager.testbench_target_socket);
		manager.testbench_initiator_socket.bind(inst_TBout[i]->manager_target_socket);

		/* Manager to modules downlink */
		manager.module_initiator_socket.bind(inst_T1[i]->manager_target_socket);
		manager.module_initiator_socket.bind(inst_T2[i]->manager_target_socket);
		manager.module_initiator_socket.bind(inst_T3[i]->manager_target_socket);

		/* Manager to modules uplink */
		inst_T1[i]->manager_initiator_socket.bind(manager.module_target_socket);
		inst_T2[i]->manager_initiator_socket.bind(manager.module_target_socket);
		inst_T3[i]->manager_initiator_socket.bind(manager.module_target_socket);

		/* Add Application Instance */
		int offset_0 = manager.addApplication(config_app_inst_name[i], "", modules_table, config_app_QoS[i]);

		/* Set TestBench (Source only) */
		inst_TBout[i]->set_application_offset(offset_0);
		inst_TBout[i]->set_application_name(config_app_inst_name[i]);
		inst_TBin[i]->set_application_name(config_app_inst_name[i]);
/*
    if (applicationName.find("/") != string::npos)
        return (applicationName.substr(applicationName.find("/") + 1)).c_str();
    else
        return applicationName;
        */


		/* Information Module */
		inst_T1[i]->info();
		inst_T2[i]->info();
		inst_T3[i]->info();
		inst_TBout[i]->info();
		inst_TBin[i]->info();

		/* Bindings verification before starting simulation */
		inst_T1[i]->verify_bindings();
		inst_T2[i]->verify_bindings();
		inst_T3[i]->verify_bindings();
		inst_TBout[i]->verify_bindings();
		inst_TBin[i]->verify_bindings();

		/* CONFIGURATION TRACE */
		inst_T1[i]->activate_trace();
		inst_T2[i]->activate_trace();
		inst_T3[i]->activate_trace();


	}


	/* Application Defined */
	manager.applicationDefined();

	/* DEBUG TRACE */
	// Disabled

	/* Activate Trace */
	manager.activate_trace();

	/* Simulator Configuration */
	manager.set_scheduler_execution_time(sc_time(0, SC_US));
	manager.set_generate_zeros_in_csv(false);





	/* Start Simulation */
	sc_start(120.0, SC_SEC);

	/* End of Simulation */
	fout << "Simulation ended at time " << sc_time_stamp() << endl;
	bool simOK(manager.isSimulationOK());
	if (simOK)
		fout << "Simulation successful" << endl;
	else
		fout << "Simulation failed" << endl;

	Reconfiguration_manager::verifySimulationTimes();

	/* Display Information */
	manager.update_occupation_rate_end_simulation();
	manager.display_occupation_rates();
	manager.generateSimulationResultFile();
	string CSVname("../../solutions/solution");
	if(argc == 3) CSVname.append("_");
	CSVname.append(simulationName);
	CSVname.append(".csv");
	manager.generateCSVResultFile(CSVname.c_str());

	/* Flush Streams */
	fout.flush();

	/* Close files */
	fout.close();
	sc_close_vcd_trace_file(tf);

	for(int i = 0; i < 1; i++) {
		delete inst_T1[i];
		delete inst_T2[i];
		delete inst_T3[i];
		delete inst_TBout[i];
		delete inst_TBin[i];
	}

	delete fpga;

	if(simOK)
		return RECOSIM_SIMULATION_SUCCESS;
	else
		return RECOSIM_SIMULATION_FAILED_ERRCODE;
}


