/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    testbench_algorithm.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-12
 * @section DESCRIPTION
 *			Testbench algorithms
 */

#ifndef TESTBENCH_ALGORITHM_H
#define TESTBENCH_ALGORITHM_H

#define NB_TRANSACTIONS_TO_SEND 100

#include "reconfiguration_manager.h"
#include "testbench_in_interface.h"
#include "testbench_out_interface.h"
#include "user_testbench_algorithms.h"
#include "utils.h"

template<int No>
void testbench_out_thread(Testbench_out_interface<No>& tb_interface) {
	
	//
	// ALGORITHM COMMANDS (Optional)
	//

	/*
		// Example : sent the "command"  to "module name" at 10 ms
		tb_interface.TB_IF_add_execution_mode_command_to_queue("module name", "command", sc_time(10, SC_MS));
	*/


	//
	// DATA SEND (Generate a sequence of random transactions)
	//
	for (int i = 0; i < NB_TRANSACTIONS_TO_SEND; i++) {

		// Check that first module in the chain is ready
#ifdef GENERATE_LOG_FILE
		Simulation_controller::get_logfile() << endl << sc_time_stamp() << ": " << tb_interface.TB_IF_name() << " checks instantiation for following modules" << endl;
#endif

		// INIT TRANSACTIONS : Initialize data to send for a default transaction for each out Interface (j port)
		// It is possible to use the connection name tb_interface.TB_IF_get_data_out_ptr("Connection1")
		// A default data of packet is [31:16] = Transaction number, [15:0] = index number (0 to packet_size-1 attribute)
		for(int j = 0; j < No; j++) {

			int max_packet = tb_interface.TB_IF_get_packet_size(j);
			int *data_out_ptr = tb_interface.TB_IF_get_data_out_ptr(j);
			for(int k = 0; k < max_packet; k++)
				data_out_ptr[k] = (i << 16) | k;
		}

		// SEND TRANSACTIONS : ALL TRANSACTIONS SENT
		// IT is possible to use TB_IF_nb_send_data(connectionName or connection id) to custom the sending
		tb_interface.TB_IF_nb_send_all_data();

		/* DO NOTHING */
		/* check_target method asks for following modules instantiation, answer from
		 * the manager will produce an event responsible for data transfers */

		// WAIT RESPONSES : Wait until all responses from following modules have been received 
		// QUESTIONFAB : et si on envoie que sur 2 sorties au lieu de trois, cela bloque ? est ce vrai ?
		wait(tb_interface.TB_IF_all_responses_received_event());

#ifdef GENERATE_LOG_FILE
		Simulation_controller::get_logfile() << sc_time_stamp() << ": " << tb_interface.TB_IF_name() << ": Transaction " << i << " has been sent" << endl;
#endif

		// WAIT NEXT PERIOD
		if(i != (NB_TRANSACTIONS_TO_SEND - 1)) {
			// Delay next transaction by waiting extra time
			sc_time delayToNextPacket((i + 1) * tb_interface.TB_IF_get_period() - sc_time_stamp());
			if(delayToNextPacket < SC_ZERO_TIME) {
				cerr << "ERROR in TB: Negative delay to next packet send! Consider revising periods and other timing information" << endl;
				exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
			}
			wait(delayToNextPacket);
		}

		// NEXT TRANSACTION ID
		tb_interface.TB_IF_increment_current_transaction_id();
	}

#ifdef GENERATE_LOG_FILE
	Simulation_controller::get_logfile() << sc_time_stamp() << ": " << tb_interface.TB_IF_name() << ": All transactions have been sent" << endl;
#endif
}

template<int Ni>
void testbench_in_thread(Testbench_in_interface<Ni> &tb_interface) {

	int nb_transactions_received = 0;
	int nb_diff = 0;
	int last_transaction_count[Ni];
	for(int i = 0; i < Ni; i++) last_transaction_count[i] = 0;

	while(nb_transactions_received != NB_TRANSACTIONS_TO_SEND * Ni) {

		sc_time timeout(Reconfiguration_manager::getMaximumSimulationTime() - sc_time_stamp() - sc_time(1, SC_US));	// Time out right before end of simulation
		wait(timeout, tb_interface.TB_IF_transaction_received_event());

		if(sc_time_stamp() == Reconfiguration_manager::getMaximumSimulationTime() - sc_time(1, SC_US)) {
			// Reached end of simulation due to timeout, something is wrong with the testbench
			cout << "CRITICAL WARNING: End of simulation reached but testbench thread is still waiting for transactions..." << endl;
			cout << "Testbench name: " << tb_interface.TB_IF_name();
			for(int i = 0; i < Ni; i++) cout << "Socket " << tb_interface.TB_IF_get_connection_name(i) << " received " << tb_interface.TB_IF_get_nb_transactions_received_per_socket(i) << " transactions" << endl;
			cout << "Consider revising:" << endl;
			cout << " - Testbench to wait for less transactions" << endl;
			cout << " - Module algorithms to send data more often" << endl;
			return;
		} else {

			// Verify data 'on-the-fly'
			//for(int i = 0; i < Ni; i++) {
			vector<int> sockets_with_new_transactions(tb_interface.TB_IF_get_sockets_with_new_transactions());
			for(int i = 0; i < (int) sockets_with_new_transactions.size(); i++) {
				int socketID = sockets_with_new_transactions.at(i);
				//if(tb_interface.TB_IF_get_nb_transactions_received_per_socket(i) != last_transaction_count[i]) {
					// Data received on this socket
					// Check data
					int *data_ptr = tb_interface.TB_IF_get_data_in_ptr(socketID);
					for(int j = 0; j < 16; j++) {
						if(data_ptr[j] != ((last_transaction_count[socketID] << 16) | j)) {
							nb_diff++;
							cout << "Socket " << socketID << ", Transaction " << last_transaction_count[socketID] << ", Data " << j << " -> " << hex << data_ptr[j] << " (expected " << ((last_transaction_count[socketID] << 16) | j) << ")" << endl;
						}
					}

					// Update transaction count
					last_transaction_count[socketID]++;
					nb_transactions_received++;
					//break;
				//}
			}

			
		}

		
	}

	if(nb_diff == 0) {
		Simulation_controller::get_logfile() << endl << "==================================================" << endl;
		Simulation_controller::get_logfile() << "  CONGRATULATIONS: Simulation ended successfully  " << endl;
		Simulation_controller::get_logfile() << "==================================================" << endl << endl;

		// Communicate with the simulation controller
		tb_interface.TB_IF_notify_simulation_controller(true);
	} else {
		Simulation_controller::get_logfile() << endl << "==================================================" << endl;
		Simulation_controller::get_logfile() << "         Simulation failed with " << dec << nb_diff << " errors            " << endl;
		Simulation_controller::get_logfile() << "==================================================" << endl << endl;

		exit(RECOSIM_TESTBENCH_CHECK_FAILED);
	}
}

#endif
