/**
 * UNS - CNRS
 * Copyright 2013 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: licensing@sattse.com
 *
 * @file    monitoring_interface.h
 * @author  Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0-build:2013/06/10
 * @date    June 25, 2014 2:10:21 PM CEST
 * @section DESCRIPTION
 *
 *          Monitoring Interface for User
*/

#ifndef MONITORING_INTERFACE_H
#define MONITORING_INTERFACE_H

#define SC_INCLUDE_DYNAMIC_PROCESSES
#include <systemc.h>

#include "scheduler_interface.h"

using namespace std;

#define MONITORING_MODULE(name) struct name##_trace_packet : public Monitoring 

#define TRACE_VAR_DOUBLE(var_nom) 	double var_nom ;

#define TRACE_VAR_INT(var_name) int var_name ;

#define TRACE_VAR_UINT(var_name) unsigned int var_name ;

#define TRACE_VAR_SC_BIGINT(var_name, size) sc_bigint<size> var_name ;

#define TRACE_VAR_SC_BIGINT_PTR(var_name, size) sc_bigint<size>* var_name ;
#define TRACE_VAR_SC_BIGINT_ALLOCATE(var_name, size, dim) var_name = new sc_bigint<size>[dim];

#define MONITORING_ADD_THREAD(module_name, thread_name) sc_spawn(sc_bind(&module_name ## _trace_packet:: thread_name, this, thread_number++), "monitoring_thread_" + thread_number); 

#define MONITORING_INIT(name)	SC_HAS_PROCESS(name ## _trace_packet); \
								name ## _trace_packet(sc_module_name nm, Monitoring_interface& services) : Monitoring(nm, services)

#define MONITORING_ACTIVATE void activate_trace()

#define MONITORING_ANALOG_TRACE(var_name, nm, min_value, max_value, height) sc_trace(services.getTraceFile(), var_name, monitoring_name + "." + nm);

#define MONITORING_TRACE(var_name, displayed_name, format) sc_trace(services.getTraceFile(), var_name, monitoring_name + "." + displayed_name);

#define MONITORING_THREAD(name) void name(int monitoringThreadID)

#define UPDATE_RZ_STATE_EVENT services.updateRZStateEvent(monitoringThreadID)
#define UPDATE_TASK_STATE_EVENT services.updateTaskStateEvent(monitoringThreadID)
#define START_OF_SCHEDULING_EVENT services.startOfSchedulingEvent(monitoringThreadID)
#define END_OF_SCHEDULING_EVENT services.endOfSchedulingEvent(monitoringThreadID)

class Monitoring_interface {

public:

	/**
	 * @brief Get the trace file
	 * @return trace file pointer
	 */
	virtual sc_trace_file* getTraceFile() const = 0;

	/**
	 * @brief Get the scheduler interface
	 * @return scheduler interface pointer
	 */
	virtual Scheduler_interface& getShedulerInterface() = 0;

	/**
	 * @brief Get the event notified at the start of the scheduling decision. The monitoring thread can wait on the event.
	 * @return event of end of scheduling
	 */
	virtual sc_event& startOfSchedulingEvent(int id) = 0;

	/**
	 * @brief Get the event notified at the end of the scheduling decision. The monitoring thread can wait on the event.
	 * @return event of end of scheduling
	 */
	virtual sc_event& endOfSchedulingEvent(int id) = 0;
	
	/**
	 * @brief Get the event notified when the RZ state changed. The monitoring thread can wait on the event.
	 * @return event of end of scheduling
	 */
	virtual sc_event& updateRZStateEvent(int id) = 0;

	/**
	 * @brief Get the event notified when the task state changed. The monitoring thread can wait on the event.
	 * @return event of end of scheduling
	 */
	virtual sc_event& updateTaskStateEvent(int id) = 0;

	/**
	 * @brief Get the RZ properties of the instance (Processor or RZ)
	 * @param inst_name Instance of the processor or default RZ ("RZDefaultGroupName")
	 * @return Pointer to RZ properties
	 */
	virtual RZ_config* getRZProperties(string inst_name) = 0;

};

#endif
