/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    simulation.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-01-19
 * @section DESCRIPTION
 *			Simulation representation (tasks, RZs)
 */

#ifndef SIMULATION_H
#define SIMULATION_H

#include <iostream>
#include <string>
#include <vector>
#include <map>
using namespace std;

#include "hardwareTask.h"
#include "softwareTask.h"
#include "virtualHardwareTask.h"
#include "RZStats.h"
#include "synthesisReport.h"
#include "processor.h"
#include "task.h"


/** Available allocation optimization strategies */
enum AllocationStrategy {NO_OPTIMIZATION, HIGHEST_FRAGMENTATION, HIGHEST_MEMORY_COST, SHORTEST_RUNTIME};
static const string allocationStrategyName[4] = {"No optimization", "Highest fragmentation", "Highest memory cost", "Shortest runtime"};
enum ProcessorAllocationStrategy {SOFT_NO_OPTIMIZATION, SOFT_SHORTEST_RUNTIME};
static const string processorAllocationStrategyName[2] = {"No optimization", "Shortest runtime"};

/** Available implementation metrics */
enum ImplementationMetric {EXTERNAL_FRAGMENTATION, POWER_CONSUMPTION};
static const string implementationMetricName[2] = {"External fragmentation", "Power consumption"};

/** Partitioning step return values */
enum PartitioningReturnValue {PARTITIONING_SUCCESS, PARTITIONING_USELESS, PARTITIONING_IMPOSSIBLE, PARTITIONING_FAILURE};

/** PlanAhead supported version enumeration */
enum PlanAheadVersion {PLANAHEAD_12_X, PLANAHEAD_13_X, PLANAHEAD_14_X};
static const string planAheadVersionName[3] = {"12.x", "13.x", "14.x"};

/** Bitstream generation for compression estimation ? */
enum BitstreamGeneration {NONE, PLANAHEAD_IMPLEMENTATION, FINAL_PLANAHEAD_VERIFICATION};
static const string bitstreamGenerationName[3] = {"None", "PlanAhead implementation", "Final PlanAhead verification"};

/** FaRM resources requirements  */
static const int FARM_NB_SLICE = 319;

static const string RZ_XML_EXTENSION = ".rrd";


class Simulation {

public:

	static bool VERBOSE_MODE;

private:

	//////////////////////////
	// FoRTReSS parameters //
	////////////////////////

	// STEP 0: Environment
	static PlanAheadVersion planAheadVersion;
	static string planAheadPath;
	

	// STEP 1: RZ determination per task
	static bool USE_RZ_SET_DESCRIPTION_FILE;
	static string RZ_SET_DESCRIPTION_FILENAME;
	static bool ALLOW_NON_RECTANGULAR_RZ;
	static double OVERSIZED_RZ_TRIGGER;
	static double RESOURCES_MARGIN;
	static bool PURELY_RECONFIGURABLE_RZ;
	static bool IGNORE_RZ_POSITION_CONSTRAINTS;

	// STEP 2: RZ sort
	static double COST_SHAPE_WEIGHT;
	static double COST_COMPLIANCE_WEIGHT;
	static double COST_INTERNAL_FRAGMENTATION_WEIGHT;
	static bool CHECK_RZ_REDUNDANCY;

	// STEP 3: RZ selection for simulation
	static ImplementationMetric IMPLEMENTATION_METRIC;
	static double TRIGGER_FACTOR_UNACCEPTABLE;
	static double TRIGGER_FACTOR_OPTIMUM;
	static int MINIMUM_DISTANCE_BETWEEN_RZS;

	// STEP 4: Allocation
	static AllocationStrategy ALLOCATION_STRATEGY;						/* Allocation optimization strategy */
	static ProcessorAllocationStrategy PROCESSOR_ALLOCATION_STRATEGY;	/* Allocation optimization strategy for processors */

	// STEP 5: PR cost model
	static BitstreamGeneration BITSTREAM_GENERATION;	/* Bitstream generation for compression estimation */
	static float MEAN_COMPRESSION_RATIO;				/* Mean compression ratio */
	static float Tbus;									/* Bus period in nanoseconds */
	static float Ticap;									/* ICAP period in nanoseconds */
	static float Nburst;								/* Burst length */
	static float tburst;								/* Time spent on a burst, in cycles of Tbus */
	static float latency;								/* Latency to initiate a transaction on the bus, in cycles of Tbus */
	static float fifoDepth;								/* FaRM FIFO depth in 32-bit words */

	// STEP 6: Simulation
	static bool RUN_SIMULATION_UNTIL_STATIC_SOLUTION;

	string simulationOkSuffix;

	//////////////
	// Members //
	////////////

	vector<HwTask> hwTasks;									/* Task collection */
	vector<SwTask> swTasks;									/* SW Task collection */
	set<string> modulesSet;
	vector<VirtualHwTask> virtualHwTasks;					/* Virtual tasks created during the partitioning step */

	vector<RZ> RZset;										/* Entire RZ set derived from task collection */
	vector<RZ> RZsubset;									/* RZsubset built for simulation */

	vector<Processor> processorSet;							/* Entire RZ set derived from task collection */
	vector<Processor> processorSubset;						/* RZsubset built for simulation */

	vector<RZStats> RZSimulationStats;						/* Statistics from last simulation for RZs */
	vector<RZStats> processorSimulationStats;				/* Statistics from last simulation for processors */
	vector<RZStats> firstRZSimulationStats;					/* Statistics from last simulation for RZs */
	vector<RZStats> firstProcessorSimulationStats;			/* Statistics from last simulation for processors */

	int nbRZInSimulation;									/* Current number of RZs in the simulation set */
	int lastRZInSimulationSet;								/* Last RZ from global set in simulation subset */
	int minimalExternalFragmentation;						/* Minimal achieved external fragmentation */
	int minimalAllocationCost;								/* Minimal allocation cost achieved */	

	vector<vector<int> > previousRZTaskAssociation;			/* Previous RZ/Task association */
	vector<vector<int> > currentRZTaskAssociation;			/* Current RZ/Task association */
	vector<vector<int> > bestRZTaskAssociation;				/* Best achieved RZ/Task association */

	vector<vector<int> > currentProcessorTaskAssociation;	/* Current Processor/Task association */
	vector<vector<int> > bestProcessorTaskAssociation;		/* Best achieved Processor/Task association */

	vector<vector<double> > compressedReconfigurationTimes;	/* Compressed reconfiguration times */
	vector<vector<double> > preloadReconfigurationTimes;	/* Preload configuration times */

	vector<vector<double> > bitstreamSizes;					/* Bitstream sizes in kB */


	// Partitioning step attributes
	int nbGreedyRZs;										/* Number of 'greedy' RZs (hosting every tasks) */
	int nbRZOptimumSet;
	int nbRZUnacceptableSet;
	vector<HwTask> optimumTaskVector;						/* Vector containing optimum tasks */
	vector<HwTask> acceptableTaskVector;					/* Vector containing acceptable tasks */
	vector<HwTask> unacceptableTaskVector;					/* Vector containing unacceptable tasks */

	vector<int> optimumTaskVectorID;						/* Vector containing optimum tasks IDs */
	vector<int> acceptableTaskVectorID;						/* Vector containing acceptable tasks IDs */
	vector<int> unacceptableTaskVectorID;					/* Vector containing unacceptable tasks IDs */

	// Simulation attributes
	int simulationNumber;
	int lastValidSimulationID;
	string currentSimulationName;

	// Directories used to store result files
	string logDirectory;
	string traceDirectory;
	string CSVDirectory;
	string UCFDirectory;
	string XMLDirectory;

	bool finalVerificationReached;
	bool backToUnpartitionedSet;

	map<string, int> physicalInterfacesLocation;

	Device* simulationDevice;

	static const int FORTRESS_NO_SOLUTION_ERRCODE;
	static const int FORTRESS_INTERNAL_ERROR_ERRCODE;


public:
	Simulation() {
		nbRZInSimulation = 0;
		minimalExternalFragmentation = 0;
		lastRZInSimulationSet = 0;
		nbGreedyRZs = 0;
		nbRZOptimumSet = 0;
		nbRZUnacceptableSet = 0;
		lastValidSimulationID = 0;
		simulationNumber = 0;

		finalVerificationReached = false;
		backToUnpartitionedSet = false;

		logDirectory = "";
		traceDirectory = "";
		CSVDirectory = "";
		UCFDirectory = "";
		XMLDirectory = "";
		simulationOkSuffix = "";

		simulationDevice = new Device();
	}

	~Simulation() {
		delete simulationDevice;
	}

private:
	// Helper functions
	bool areAllTasksHosted(vector<HwTask> &t);
	bool atLeastOneTaskHosted(vector<HwTask> &t, RZ *rz);
	int computeExternalFragmentationCost(void);
	double computeRZDistance(int rzid);
	int computeRZCost(void);
	int findEliteSet(void);
	void clearAllocation(void);
	void printSelectedRZ(void);
	int parseResultFile(void);
	void printParsedResultFile(void);
	int computeCurrentAllocationCost(void);
	double computeMemoryCost(void);
	void printPRAreaResources(ostream &stream, bool CSVFile);
	void editCurrentCSVFile(void);
	bool allocationOptimizationPossible(void);
	bool allModuleHaveSWImplementations(void);
	bool isSimulationValid(int);
	SwTask& getSWTask(string impl);
	HwTask& getHWTask(string impl);
	int getHwReconfigurableModuleNumber();

	// Flow functions
	void searchRZs(void);
	int readInputXMLFile(string filename);
	int writeOutputXMLFile(string filename);
	void sortRZSet(void);
	int defineSimulationRZSet(void);
	PartitioningReturnValue definePartitionedRZSet(void);
	void defineSimulationProcessorSet(void);
	void taskAllocation(void);
	void RZAllocation(AllocationStrategy);
	void firstTaskAllocation(void);
	void firstRZAllocation(void);
	void firstProcessorAllocation(void);
	void processorAllocation(ProcessorAllocationStrategy);
	void printAllocation(void);
	void writeTaskAllocation(void);
	void computeReconfigurationTimes(void);
	int writeRZConfigurationFile(string filename);
	int launchSimulation(void);
	void generateConstraints(bool);
	void renameValidSimulationFiles(bool partition, string newValidSimulationName, bool addNewName);
	void renameFile(string oldfile, string newfile);

public:
	// Main function
	void setHwTaskVector(vector<HwTask> &vec);
	void setSwTaskVector(vector<SwTask> &vec);
	void setModuleSet(set<string> &modset);
	void setProcessorVector(vector<Processor> &vec);
	void setInterfaceVector(vector<InterfaceLocation> &vec);
	int launchArchGenFlow(void);
	void displayParameters(void);
	void setLogDirectory(string dir);
	void setTraceDirectory(string dir);
	void setCSVDirectory(string dir);
	void setUCFDirectory(string dir);
	void setXMLDirectory(string dir);

	void setAllowRectangularRZ(bool val);
	void setResourcessMargin(double val);
	void setCostShapeWeight(double val);
	void setCostComplianceWeight(double val);
	void setCheckRZRedundancy(bool val);
	void setImplementationMetric(ImplementationMetric val);
	void setTriggerFactorUnacceptable(double val);
	void setTriggerFactorOptimum(double val);
	void setAllocationStrategy(AllocationStrategy val);
	void setProcessorAllocationStrategy(ProcessorAllocationStrategy val);
	void setBitstreamGeneration(BitstreamGeneration val);
	void setTbus(float val);
	void setTicap(float val);
	void setNburst(float val);
	void setTburst(float val);
	void setLatency(float val);
	void setFIFODepth(float val);
	
};

#endif
