/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    simulation.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-01-19
 * @section DESCRIPTION
 *			Simulation representation (tasks, RZs)
 */

#include "simulation.h"
#include "sort.h"
#include "utils.h"

#include <fstream>
#include <time.h>
#include <math.h>
#include <cstdlib>
#include <stdio.h>
#include <iomanip>

#include "RZSetDescriptionFile.h"

//int recosim_exe(int argc, char* argv[]);

#define STANDALONE_DEBUG

// FoRTReSS errcodes
const int Simulation::FORTRESS_NO_SOLUTION_ERRCODE = 0xDEADBEEF;
const int Simulation::FORTRESS_INTERNAL_ERROR_ERRCODE = 0xCAFECAFE;

// ResoSim errcodes
#define RECOSIM_SIMULATION_SUCCESS				0
#define RECOSIM_SIMULATION_FAILED_ERRCODE		1
#define RECOSIM_NOT_ALL_TASKS_HOSTED_ERRCODE	2
#define RECOSIM_INTERNAL_ERROR_ERRCODE			3
#define RECOSIM_TESTBENCH_CHECK_FAILED			4

/**
 * This method verifies if all tasks contained in the vector
 * passed as parameter may be host by the member RZsubset
 * TODO : passing RZ vector by argument
 */
bool Simulation::areAllTasksHosted(vector<HwTask> &t) {
	for(int i = 0; i < (int) t.size(); i++) {
		bool taskHosted = false;
		for(int j = 0; j < (int) RZsubset.size(); j++) {
			if(t.at(i).fitsRZ(&RZsubset.at(j))) {
				taskHosted = true;
				break;
			}
		}
		if(!taskHosted) return false;
	}

	return true;
}

/**
 * This method computes the external fragmentation in the current
 * solution (RZsubset). The value is the sum of all RZ distances
 * to each other (computed in terms of CLB/Logic element).
 */
int Simulation::computeExternalFragmentationCost(void) {

	if(RZsubset.size() == 1) return 0;
	else {

		int distance = 0;

		for(int i = 0; i < (int) RZsubset.size(); i++) {
			for(int j = 0; j < (int) RZsubset.size(); j++) {
				if(i != j) {
					distance += RZsubset.at(i).horizontalDistance(RZsubset.at(j));
					distance += RZsubset.at(i).verticalDistance(RZsubset.at(j)) * RZsubset.at(i).getDevicePtr()->getCell(CLBl).getNbCellsPerColumn();
				}
			}
		}

		//distance /= (double) RZsubset.size();

		return distance;
	}
}

/**
 * This method computes the distance for a RZ inside RZsubset
 */
double Simulation::computeRZDistance(int rzid) {

	if(RZsubset.size() == 1) return 0;
	else {

		double distance = 0;

		for(int i = 0; i < (int) RZsubset.size(); i++) {
			if(i != rzid) {
				distance += RZsubset.at(rzid).horizontalDistance(RZsubset.at(i));
				distance += RZsubset.at(rzid).verticalDistance(RZsubset.at(i)) * RZsubset.at(i).getDevicePtr()->getCell(CLBl).getNbCellsPerColumn();
			}
		}

		return distance;
	}
}

/**
 * This methods finds an elite set out of the full RZ set, containing
 * nbRZInSimulation RZs.
 */
int Simulation::findEliteSet(void) {

	int nbRZAdded = 0;
	int RZToTest = 0;

	do {
		if(RZToTest == (int)RZset.size()) {
			cerr << "Cannot find " << nbRZInSimulation << " non-overlapping RZs in RZ set..." << endl;
			return -1;
		}

		// Check if there is no overlap with other RZs already selected
		RZ currentRZ(RZset.at(RZToTest));
		if(!currentRZ.overlapsRZInSet(RZsubset)) {
			if(VERBOSE_MODE) cout << "Adding RZ " << RZToTest << " to elite set" << endl;
			RZsubset.push_back(currentRZ);
			nbRZAdded++;
			lastRZInSimulationSet = RZToTest;
		}

		RZToTest++;

	} while(nbRZAdded != nbRZInSimulation);

	return 0;
}

void Simulation::clearAllocation(void) {

	for(int i = 0; i < (int) RZsubset.size(); i++) RZsubset.at(i).clearCompatibleTasks();
	for(int i = 0; i < (int) processorSubset.size(); i++) processorSubset.at(i).clearCompatibleTasks();

	currentRZTaskAssociation.clear();
	currentProcessorTaskAssociation.clear();
	vector<int> dummyVec;
	for(int i = 0; i < (int) nbRZInSimulation; i++) currentRZTaskAssociation.push_back(dummyVec);
	for(int i = 0; i < (int) processorSet.size(); i++) currentProcessorTaskAssociation.push_back(dummyVec);
}

void Simulation::printSelectedRZ(void) {
	for(int i = 0; i < (int) RZsubset.size(); i++) {
		RZsubset.at(i).info();
		RZsubset.at(i).printRZ();
	}
}

int Simulation::parseResultFile(void) {

	const string STRING_RZ			= "RZ ";
	const string STRING_PROCESSOR	= "Processor ";
	const string STRING_END			= "end";
	const string STRING_TASK		= "Task ";
	const string STRING_BLANK		= "Blank: ";
	const string STRING_RECONFIG	= "Reconfiguring: ";
	const string STRING_MAPPED		= "Mapped: ";
	const string STRING_ACTIVE		= "Active: ";
	const string STRING_RUNNING		= "Running: ";
	const string STRING_ICAP		= "ICAP occupation rate: ";
	const string STRING_SW_LOADER	= "Software loader occupation rate: ";
	const string STRING_QOS			= "Achieved global QoS for application ";
	const string STRING_MIN_QOS		= "Minimal QoS on an hyperperiod for application ";

	string filename("result.txt");
	ifstream file(filename.c_str());

	RZSimulationStats.clear();
	processorSimulationStats.clear();

	if(file) {

		if(VERBOSE_MODE) cout << "Parsing RZ result file " << filename << endl;
		string line_read;

		while(getline(file, line_read)) {

			if(line_read.find(STRING_ICAP) != string::npos) {
				string icap_str(line_read.substr(STRING_ICAP.size()));
				int rate = Utils::atoi(icap_str);

				if(VERBOSE_MODE) cout << "ICAP occupation rate: " << rate << " %" << endl;

			} else if(line_read.find(STRING_SW_LOADER) != string::npos) {
				string sw_loader_str(line_read.substr(STRING_SW_LOADER.size()));
				int rate = Utils::atoi(sw_loader_str);

				if(VERBOSE_MODE) cout << "Software loader occupation rate: " << rate << " %" << endl;

			} else if(line_read.find(STRING_QOS) != string::npos) {
				if(VERBOSE_MODE) cout << line_read << endl;
			} else if(line_read.find(STRING_MIN_QOS) != string::npos) {
				if(VERBOSE_MODE) cout << line_read << endl;
			}else if(line_read.find(STRING_RZ) != string::npos || line_read.find(STRING_PROCESSOR) != string::npos) {
				bool isRZ;
				string rzID_str;
				if(line_read.find(STRING_RZ) != string::npos) {
					rzID_str.append(line_read.substr(STRING_RZ.size()));
					isRZ = true;
				} else {
					rzID_str.append(line_read.substr(STRING_PROCESSOR.size()));
					isRZ = false;
				}
				//int id = Utils::atoi(rzID_str);

				// Retrieve occupation rates
				getline(file, line_read);
				string blankStr = line_read.substr(STRING_BLANK.size());
				double blankTime = Utils::atod(blankStr);

				getline(file, line_read);
				string reconfigStr = line_read.substr(STRING_RECONFIG.size());
				double reconfigTime = Utils::atod(reconfigStr);

				getline(file, line_read);
				string mappedStr = line_read.substr(STRING_MAPPED.size());
				double mappedTime = Utils::atod(mappedStr);

				getline(file, line_read);
				string activeStr = line_read.substr(STRING_ACTIVE.size());
				double activeTime = Utils::atod(activeStr);

				getline(file, line_read);
				string runningStr = line_read.substr(STRING_RUNNING.size());
				double runningTime = Utils::atod(runningStr);

				map<string, double> taskMap;
				bool endFound = false;
				do {
					getline(file, line_read);
					if(line_read.find(STRING_END) != string::npos) endFound = true;
					else if(line_read.find(STRING_TASK) != string::npos) {
						string taskName = line_read.substr(STRING_TASK.size());
						getline(file, line_read);
						double taskExec = Utils::atod(line_read);
						taskMap[taskName] = taskExec;
					}
				} while(!endFound);

				if(isRZ) RZSimulationStats.push_back(RZStats(rzID_str, isRZ, blankTime, reconfigTime, mappedTime, activeTime, runningTime, taskMap));
				else processorSimulationStats.push_back(RZStats(rzID_str, isRZ, blankTime, reconfigTime, mappedTime, activeTime, runningTime, taskMap));
			} else {
				/*cerr << "ERROR: while parsing result file..." << endl;
				return -1;*/
			}
		}
		return 0;
	} else {
		cerr << "ERROR: Unable to read result file from simulation..." << endl;
		return -1;
	}
}

void Simulation::printParsedResultFile(void) {
	for(int i = 0; i < (int) RZSimulationStats.size(); i++) RZSimulationStats.at(i).printRZStats();
	for(int i = 0; i < (int) processorSimulationStats.size(); i++) processorSimulationStats.at(i).printRZStats();
}

int Simulation::computeCurrentAllocationCost(void) {
	return (int) computeMemoryCost();
}

double Simulation::computeMemoryCost(void) {

	double cost = 0;

	// Directly related to the reconfiguration time
	for(int i = 0; i < (int) compressedReconfigurationTimes.size(); i++) {
		for(int j = 0; j < (int) compressedReconfigurationTimes.at(i).size(); j++) cost += bitstreamSizes.at(i).at(j);
	}

	return cost;	// Memory needs in kB
}

bool Simulation::atLeastOneTaskHosted(vector<HwTask> &t, RZ *rz) {
	for(int i = 0; i < (int) t.size(); i++) {
		if(t.at(i).fitsRZ(rz)) return true;
	}
	return false;

	//return true;
}

/**
 * Searches the RZs for every tasks
 */
void Simulation::searchRZs(void) {

	cout << endl << "-----------------------------------------------------------" << endl;
	cout << "-----------------------------------------------------------" << endl;

	for(int i = 0; i < (int) hwTasks.size(); i++) {


		if (hwTasks.at(i).isStaticModule() == true) {
			cout << "No Searching RZs for Static task " << hwTasks[i].getName() << endl << endl;
			continue;
		}
		else
			cout << "Searching RZs for task " << hwTasks[i].getName() << " ..." << endl;

		hwTasks.at(i).findRZs();
		cout << endl;

#ifdef DEBUG
		if(!hwTasks[i].getRZSet().empty()) {
			int rz_test_id = rand() % (int) hwTasks[i].getRZSet().size();
			hwTasks[i].getRZSet().at(rz_test_id).info();
			hwTasks[i].getRZSet().at(rz_test_id).printRZ();
			hwTasks[i].getRZSet().at(rz_test_id).generateUCF(hwTasks[i].getName());
			cout << endl;
		}
#endif

		if(CHECK_RZ_REDUNDANCY) {
			// Check for double RZs
			for(int j = 0; j < (int) hwTasks[i].getRZSetPtr()->size(); j++) {
				bool redundant = false;
				for(int k = 0; k < (int) RZset.size(); k++) {
					if(hwTasks.at(i).getRZSetPtr()->at(j) == RZset.at(k)) {
						redundant = true;
						break;
					}
				}

				if(!redundant) {
					if(atLeastOneTaskHosted(hwTasks, &hwTasks.at(i).getRZSetPtr()->at(j))) RZset.push_back(hwTasks.at(i).getRZSetPtr()->at(j));
				}
			}
		} else {
			//RZset.insert(RZset.begin() + nbRZ, tasks[i].getRZSetPtr()->begin(), tasks[i].getRZSetPtr()->end());
			RZset.insert(RZset.end(), hwTasks[i].getRZSetPtr()->begin(), hwTasks[i].getRZSetPtr()->end());
			/*for(int j = 0; j < (int) tasks[i].getRZSet().size(); j++) {
				if(atLeastOneTaskHosted(tasks, &tasks.at(i).getRZSetPtr()->at(j))) {
					RZset.push_back(tasks.at(i).getRZSetPtr()->at(j));
					nbRZ++;
				}
			}*/
		}

		// Clear tables
		hwTasks[i].getRZSetPtr()->clear();
	}

	// Also search RZ space for virtual RZ hosting every tasks
	cout << "Searching RZs for greatest virtual task" << endl;

	string virtualTaskName("VirtualTask_");
	virtualTaskName.append(Utils::itoa(virtualHwTasks.size()));

	// Create vector with tasks merged into this virtual task
	vector<HwTask*> virtualTaskComponentsVector;
	for(int i = 0; i < (int) hwTasks.size(); i++) virtualTaskComponentsVector.push_back(&hwTasks.at(i));
	VirtualHwTask virtualTask(virtualTaskName, virtualTaskComponentsVector, hwTasks.front().getDevicePtr());
	virtualHwTasks.push_back(virtualTask);

	RZFinderManager::retrieveRZSet(virtualTask.getResourceRequirements());
	vector<RZ> RZSetToScan(virtualTask.getRZSet());

	if(CHECK_RZ_REDUNDANCY) {
		// Check for double RZs
		for(int j = 0; j < (int) RZSetToScan.size(); j++) {
			bool redundant = false;
			for(int k = 0; k < (int) RZset.size(); k++) {
				if(RZSetToScan.at(j) == RZset.at(k)) {
					redundant = true;
					break;
				}
			}

			if(!redundant) {
				if(atLeastOneTaskHosted(hwTasks, &RZSetToScan.at(j))) RZset.push_back(RZSetToScan.at(j));
			}
		}
	} else {
		//RZset.insert(RZset.begin() + nbRZ, RZSetToScan.begin(), RZSetToScan.end());
		RZset.insert(RZset.end(), RZSetToScan.begin(), RZSetToScan.end());
		/*for(int j = 0; j < (int) RZSetToScan.size(); j++) {
			if(atLeastOneTaskHosted(tasks, &RZSetToScan.at(j))) {
				RZset.push_back(RZSetToScan.at(j));
				nbRZ++;
			}
		}*/
	}

	//cout << "First RZ set size: " << RZset.size() << endl;

	//// Remove RZs that cannot host any task (based on resources AND interfaces)
	//cout << "Verifying RZ set" << endl;
	//int setsize = (int) RZset.size();
	//int i = 0;
	//while(i < setsize) {

	//	if(!atLeastOneTaskHosted(tasks, &RZset.at(i))) {
	//		RZset.erase(RZset.begin() + i);
	//		setsize--;
	//	} else i++;
	//}

	// Clear tables
	RZSetToScan.clear();

}

int Simulation::readInputXMLFile(string filename) {

	cout << endl << "-----------------------------------------------------------" << endl;
	cout << "-----------------------------------------------------------" << endl;
	cout << "Parsing file: " << filename << endl;

	RZSetDescriptionFile descriptionFile(filename, simulationDevice);
	if(descriptionFile.readConfigFile(simulationDevice, RZset) != 0) {
		cerr << "ERROR: Cannot parse XML file!" << endl;
		return -1;
	}

	for(int i = 0; i < (int) RZset.size(); i++) RZset.at(i).info();

	return 0;
}

int Simulation::writeOutputXMLFile(string filename) {

	if(VERBOSE_MODE) {
		cout << endl << "-----------------------------------------------------------" << endl;
		cout << "-----------------------------------------------------------" << endl;
		cout << "Writing file: " << filename << endl;
	}

	RZSetDescriptionFile descriptionFile(filename, simulationDevice);
	descriptionFile.writeConfigFile(simulationDevice, RZsubset);

	return -1;
}

/**
 * Sorts RZs according to their cost function (also calculated in this function)
 */
void Simulation::sortRZSet(void) {

	cout << endl << "Computing cost" << endl;

	// First, calculate the number of incompatible tasks for each RZ
	for(int i = 0; i < (int) RZset.size(); i++) {
		int nbTasks = 0;

		int wastage = 0;
		for(int j = 0; j < (int) hwTasks.size(); j++) {
			if(!hwTasks[j].fitsRZ(&RZset.at(i))) nbTasks++;

			// Also compute resources wastage
			wastage += hwTasks.at(j).computeInternalFragmentation(RZset.at(i));

		}

		wastage /= (int) hwTasks.size();

		RZset.at(i).setNbIncompatibleTasks(nbTasks);
		RZset.at(i).computeCost();
		//RZset.at(i).addResourceWastageCost(wastage);
	}

	cout << "Sorting " << RZset.size() << " elements" << endl;
	quickSort(RZset, 0, RZset.size() - 1);
}

// Helper function
void printVector(vector<RZ> &vec) {
	for(int i = 0; i < (int)vec.size(); i++) {
		cout << vec.at(i).getID() << " ";
	}
	cout << endl;
}

/**
 * Search a set of RZs to simulate respecting constraints like
 * overlapping and cost.
 */
int Simulation::defineSimulationRZSet(void) {
	// toto
	RZsubset.clear();

	const int nbIterationsMaxExternalFragmentation = 30;

	// Populate RZ subset with the elite RZs
	if(VERBOSE_MODE) cout << "Searching elite set..." << endl;
	findEliteSet();

	if(VERBOSE_MODE) cout << "Last RZ in simulation set: " << lastRZInSimulationSet << endl;

	// At this point, the vector is populated with the best RZs
	vector<RZ> eliteSet(RZsubset);
	vector<RZ> currentWorkingSet(RZsubset);

	bool externalFragmentationOK = false;
	double externalFragmentation = 0;
	double bestExternalFragmentation = -1.0;
	bool exitLoop = false;

	vector<RZ> bestSolution;
	vector<int> bestSolutionID;

	int nbIterations = 0;

	while(!externalFragmentationOK && !exitLoop) {

		if(VERBOSE_MODE) {
			cout << "Looking for a new solution..." << endl;
			cout << "RZsubset size: " << RZsubset.size() << endl;
			cout << "RZsubset contents: ";
			printVector(RZsubset);
		}

		int nbTaskRemoved = 0;

		for(int i = 0; i < (int) hwTasks.size(); i++) {

			if(exitLoop) break;

			bool taskHosted = false;

			/* 1) Check if the task is hosted by the RZsubset */
			for(int j = 0; j < (int) RZsubset.size(); j++) {
				if(hwTasks.at(i).fitsRZ(&RZsubset.at(j))) {
					taskHosted = true;
					//cout << "Task " << i << " hosted" << endl;
					break;
				}
			}

			/* 2) while task is not hosted */
			while(!taskHosted) {

				if(VERBOSE_MODE) {
					cout << "Task " << i << " not hosted" << endl;
					cout << "Removing one RZ from initial subset... (" << nbTaskRemoved << ")" << endl;
				}

				// Remove last RZ and search the set for a RZ hosting this task
				RZsubset.erase(RZsubset.end() - nbTaskRemoved - 1);
				nbTaskRemoved++;

				if(VERBOSE_MODE) {
					cout << "RZsubset contents: ";
					printVector(RZsubset);
				}

				if(nbTaskRemoved == nbRZInSimulation) {
					/* Removed every elite RZ in the set */

					if(VERBOSE_MODE) cout << "Removed every elite RZ in the set" << endl;

					if(nbRZInSimulation == 0) { // TO BE REMOVED
						// Skip this simulation as the best RZ cannot host every task
						cout << "Only one RZ in simulation and elite RZ does not fit..." << endl;
						return -1;
					} else {
						// Change current working set: Retry without the best RZ

						if(VERBOSE_MODE) cout << "Changing working set..." << endl;

						currentWorkingSet.erase(currentWorkingSet.begin());

						// Add next no overlap RZ
						for(int j = lastRZInSimulationSet + 1; j < (int) RZset.size(); j++) {
							RZ currentRZ(RZset.at(j));
							if(!currentRZ.overlapsRZInSet(currentWorkingSet)) {
								currentWorkingSet.push_back(currentRZ);
								lastRZInSimulationSet = j;
								break;
							}
						}

						if((int)currentWorkingSet.size() != nbRZInSimulation) {
							// Could not add RZ without overlap
							cerr << "ERROR: FoRTReSS could not find a non-overlapping RZ simulation set" << endl;
							return -1;
						}

						RZsubset = currentWorkingSet;

						if(VERBOSE_MODE) {
							cout << "New RZsubset contents: ";
							printVector(RZsubset);
						}

						i = -1;
						nbTaskRemoved = 0;
						break; // Exit not task hosted loop
					}
				}

				if(VERBOSE_MODE) {
					cout << "RZsubset contents: ";
					printVector(RZsubset);
					cout << "Searching for another RZ in the set hosting task " << i << endl;
				}


				vector<HwTask> subtasks;
				for(int j = 0; j <= i; j++) subtasks.push_back(hwTasks.at(j));
				for(int j = lastRZInSimulationSet + 1; j < (int) RZset.size(); j++) {
					if(!RZset.at(j).overlapsRZInSet(RZsubset)) {
						RZsubset.push_back(RZset.at(j));
						if(!areAllTasksHosted(subtasks)) {
							RZsubset.pop_back();
						} else {
							if(VERBOSE_MODE) cout << "Found RZ: id " << j << endl;
							lastRZInSimulationSet = j;
							taskHosted = true;
							break;
						}
					}
				}

					if(VERBOSE_MODE) {
						cout << "RZsubset contents: ";
						printVector(RZsubset);
					}

					if(!taskHosted) {
						cout << "Could not find a RZ fulfilling requirements" << endl;
						if(bestSolution.empty()) {
							// No good solution yet, exiting
							cout << "No good solution achieved, exiting..." << endl;
							return -1;
						} else {
							// Do not take this solution into account and exit loop
							cout << "RZ selection success = false" << endl;
							exitLoop = true;
							break;
						}
					}

					if(VERBOSE_MODE) {
						cout << "RZsubset contents: ";
						printVector(RZsubset);
					}

					if(!areAllTasksHosted(subtasks)) {
						// Could not find a RZ hosting this task
						cerr << "ERROR: Could not find a RZ hosting every task..." << endl;
						//return -1;
						exit(-1);
					}
				} // While not task hosted


			} // For loop through tasks

		//}// Not are all tasks hosted

		//// At this point, check if all tasks can be hosted
		//if(areAllTasksHosted(tasks)) {
		//	cerr << "ERROR 0: Not all tasks hosted" << endl;
		//	exit(-1);
		//}

		// Check that RZsubset satisfies basic conditions
		if((int) RZsubset.size() == nbRZInSimulation && areAllTasksHosted(hwTasks)) {

			externalFragmentation = computeExternalFragmentationCost();

			if(VERBOSE_MODE) cout << "External fragmentation cost: " << externalFragmentation << endl;

			if(bestExternalFragmentation == -1.0 || externalFragmentation < bestExternalFragmentation) {
				// First solution or best solution yet
				if(VERBOSE_MODE) cout << "Found new best solution" << endl;
				bestSolution = RZsubset;
				bestExternalFragmentation = externalFragmentation;
				nbIterations = 0;

				if(externalFragmentation == 0) externalFragmentationOK = true; // One RZ or ultimate solution
			} else {
				// Not a better solution
				// After a certain number of iterations without success, stop research

				if(VERBOSE_MODE) cout << "Iteration " << nbIterations << " out of " << nbIterationsMaxExternalFragmentation << " without convenient result" << endl;

				if(nbIterations == nbIterationsMaxExternalFragmentation) externalFragmentationOK = true;
			}
		} else {
			// Last iteration not good at all, terminate process
			if(VERBOSE_MODE) cout << "Reached end of RZ set, validating current best solution" << endl;
			externalFragmentationOK = true;
			RZsubset = bestSolution;
		}


		//// At this point, check if all tasks can be hosted
		//if(areAllTasksHosted(tasks)) {
		//	cerr << "ERROR 1: Not all tasks hosted" << endl;
		//	exit(-1);
		//}

		if(!externalFragmentationOK) {
			double maxDistance = computeRZDistance(0);
			if(VERBOSE_MODE) {
				cout << "RZsubset contents: ";
				printVector(RZsubset);
			}
			if(nbRZInSimulation == 1) {
				cerr << "ERROR: NOT POSSIBLE" << endl;
				exit(-1);
			} else if(nbRZInSimulation == 2) {
				// Equal distances, remove last (and worst) item
				if(VERBOSE_MODE) cout << "Removing last RZ in the set" << endl;
				RZsubset.pop_back();
			} else {
				// Remove 'furthest' RZ in the subset
				int RZWithMaxDistance = 0;
				if(VERBOSE_MODE) cout << "Distance for RZ with id " << RZWithMaxDistance << ": " << maxDistance << endl;
				for(int i = 1; i < (int) RZsubset.size(); i++) {
					double distance = computeRZDistance(i);

					if(VERBOSE_MODE) cout << "Distance for RZ with id " << i << ": " << distance << endl;

					if(distance > maxDistance) {
						maxDistance = distance;
						RZWithMaxDistance = i;
					}
				}

				// Effectively remove RZ
				if(VERBOSE_MODE) cout << "Removing furthest RZ in the set: id " << RZsubset.at(RZWithMaxDistance).getID() << endl;
				RZsubset.erase(RZsubset.begin() + RZWithMaxDistance);
			}

			if(VERBOSE_MODE) {
				cout << "RZsubset contents: ";
				printVector(RZsubset);
			}

			// Add a closer RZ
			// TODO: check if all tasks are hosted
			if(VERBOSE_MODE) cout << "Add a closer RZ" << endl;

			for(int i = lastRZInSimulationSet + 1; i < (int) RZset.size(); i++) {
				if(!RZset.at(i).overlapsRZInSet(RZsubset) && atLeastOneTaskHosted(hwTasks, &RZset.at(i))) {
					RZsubset.push_back(RZset.at(i));
					if(computeRZDistance((int) RZsubset.size() - 1) > maxDistance) {
						RZsubset.pop_back();
					} else {
						lastRZInSimulationSet = i;
						break;
					}
				}

			}
		}

		//// At this point, check if all tasks can be hosted
		//if(areAllTasksHosted(tasks)) {
		//	cerr << "ERROR 2: Not all tasks hosted" << endl;
		//	exit(-1);
		//}

		nbIterations++;


	}

	RZsubset = bestSolution;

	// At this point, check if all tasks can be hosted
	if(!areAllTasksHosted(hwTasks)) {
		if(bestSolution.empty()) {
			cout << "No solution for for " << nbRZInSimulation << " RZs" << endl;
			return -1;
		} else {

			cerr << "ERROR: Best solution cannot host all tasks" << endl;
			//taskAllocation();
			exit(-1);
		}
	}

	return 0;
}


PartitioningReturnValue Simulation::definePartitionedRZSet(void) {

	if(nbRZInSimulation == 1) return PARTITIONING_USELESS;

	if(VERBOSE_MODE) cout << "Partitioning RZ space..." << endl;

	const int nbIterationsMaxExternalFragmentation = 10;

	const double RUNTIME_MARGIN = 0.0;

	nbRZOptimumSet = 0;
	nbRZUnacceptableSet = 0;

	optimumTaskVector.clear();
	acceptableTaskVector.clear();
	unacceptableTaskVector.clear();

	optimumTaskVectorID.clear();
	acceptableTaskVectorID.clear();
	unacceptableTaskVectorID.clear();

	// First, find reference task (maximum cost)
	int maximumCost = hwTasks.front().getTaskCost();
	int maxID = 0;
	for(int i = 1; i < (int) hwTasks.size(); i++) {
		if(hwTasks.at(i).getTaskCost() > maximumCost) {
			maximumCost = hwTasks.at(i).getTaskCost();
			maxID = i;
		}
	}

	if(VERBOSE_MODE) cout << "Maximum task cost is for ID " << maxID << " (" << hwTasks.at(maxID).getName() << ")" << endl;


	// Put each task into the good list
	double unacceptableTrigger = (double) maximumCost * TRIGGER_FACTOR_UNACCEPTABLE / 100.0;
	double acceptableTrigger = (double) maximumCost * TRIGGER_FACTOR_OPTIMUM / 100.0;
	for(int i = 0; i < (int) hwTasks.size(); i++) {
		if(hwTasks.at(i).getTaskCost() < unacceptableTrigger) {
			unacceptableTaskVector.push_back(hwTasks.at(i));
			unacceptableTaskVectorID.push_back(i);
		} else if(hwTasks.at(i).getTaskCost() < acceptableTrigger) {
			acceptableTaskVector.push_back(hwTasks.at(i));
			acceptableTaskVectorID.push_back(i);
		} else {
			optimumTaskVector.push_back(hwTasks.at(i));
			optimumTaskVectorID.push_back(i);
		}
	}

	if(VERBOSE_MODE) {
		cout << "Optimum task vector contains: ";
		for(int i = 0; i < (int) optimumTaskVector.size(); i++) cout << optimumTaskVector.at(i).getName() << " ";
		cout << endl << "Acceptable task vector contains: ";
		for(int i = 0; i < (int) acceptableTaskVector.size(); i++) cout << acceptableTaskVector.at(i).getName() << " ";
		cout << endl << "Unacceptable task vector contains: ";
		for(int i = 0; i < (int) unacceptableTaskVector.size(); i++) cout << unacceptableTaskVector.at(i).getName() << " ";
		cout << endl;
	}


	if(unacceptableTaskVector.empty()) {
		cout << "No unacceptable task, thus no need for partitioning" << endl;
		return PARTITIONING_USELESS;
	} else {

		// Compute number of RZs hosting all the tasks
		nbGreedyRZs = 0;
		for(int i = 0; i < (int) RZsubset.size(); i++) {
			if(RZsubset.at(i).getCompatibleTaskVector().size() == hwTasks.size()) nbGreedyRZs++;
		}

		if(VERBOSE_MODE) cout << "Number of RZs hosting all the tasks: " << nbGreedyRZs << endl;

		// Check how many RZs are needed for the optimum partition
		double aggregateRuntime = 0;
		for(int i = 0; i < (int) RZSimulationStats.size(); i++) {
			map<string, double> executionTimes = RZSimulationStats.at(i).getTaskExecutionTimesMap();
			map<string, double>::iterator it;
			for(int j = 0; j < (int) optimumTaskVector.size(); j++) {
				for(it = executionTimes.begin(); it != executionTimes.end(); it++) {
					if(it->first.compare(optimumTaskVector.at(j).getName()) == 0) aggregateRuntime += (it->second / 100 * RZSimulationStats.at(i).getRZStateTime(RZ_RUNNING));
				}
			}
		}

		nbRZOptimumSet = (int) (aggregateRuntime * (1 + RUNTIME_MARGIN) / 100.0) + 1;
		if(VERBOSE_MODE) cout << "Aggregate runtime for optimum task subset: " << aggregateRuntime << " (" << nbRZOptimumSet << " RZs)" << endl;

		// Same for the unacceptable partition
		aggregateRuntime = 0;
		for(int i = 0; i < (int) RZSimulationStats.size(); i++) {
			map<string, double> executionTimes = RZSimulationStats.at(i).getTaskExecutionTimesMap();
			map<string, double>::iterator it;
			for(int j = 0; j < (int) unacceptableTaskVector.size(); j++) {
				for(it = executionTimes.begin(); it != executionTimes.end(); it++) {
					if(it->first.compare(unacceptableTaskVector.at(j).getName()) == 0) aggregateRuntime += (it->second / 100 * RZSimulationStats.at(i).getRZStateTime(RZ_RUNNING));
				}
			}
		}

		nbRZUnacceptableSet = (int) (aggregateRuntime * (1 + RUNTIME_MARGIN) / 100.0) + 1;
		if(VERBOSE_MODE) cout << "Aggregate runtime for unacceptable task subset: " << aggregateRuntime << " (" << nbRZUnacceptableSet << " RZs)" << endl;

		//if((nbRZOptimumSet + nbRZUnacceptableSet) > nbRZInSimulation) {
		nbGreedyRZs = nbRZUnacceptableSet * 2; // TMP
		if(nbRZUnacceptableSet > (nbGreedyRZs/2) || nbRZOptimumSet > (nbGreedyRZs/2)) {
			// Partitioning will split RZs into unacceptable and optimum RZ
			// To be optimized: if only one optimum RZ needed, just use one optimum RZ and not nbGreedyRZs/2
			cout << "Cannot partition the application because of the number of RZs" << endl;
			return PARTITIONING_IMPOSSIBLE;
		}

		if(VERBOSE_MODE) printVector(RZsubset);

		int nbRZToRemove = nbRZUnacceptableSet;
		//int nbRZToRemove = nbGreedyRZs/2;

		// Removing the nbRZUnacceptableSet last RZs and
		for(int i = 0; i < nbRZToRemove; i++) {
			/*RZsubset.pop_back();*/

			// Remove RZs with the worst cost that can host all the task
			/*int RZposition = (int) RZsubset.size() - 1;
			while(RZsubset.at(RZposition).getCompatibleTaskVector().size() != tasks.size()) RZposition--;
			RZsubset.erase(RZsubset.begin() + RZposition);*/

			// Remove RZs with that cannot host every task in priority
			int RZposition = (int) RZsubset.size() - 1;
			while(RZposition >= 0 && RZsubset.at(RZposition).getCompatibleTaskVector().size() == hwTasks.size()) RZposition--;
			if(RZposition >= 0) {
				RZsubset.erase(RZsubset.begin() + RZposition);
			} else {
				// Every RZ can host every task, removing the worst in terms of cost
				RZsubset.pop_back();
			}

			if(VERBOSE_MODE) printVector(RZsubset);
		}

		vector<RZ> RZSetToScan;

		int firstNewTask = (int)RZsubset.size();
		int lastTaskAddedInScannedSubset = 0;

		// Add new RZs only compliant with acceptable task set at most
		for(int i = 0; i < nbRZToRemove; i++) {

			if(USE_RZ_SET_DESCRIPTION_FILE) {
				RZSetToScan = RZset;
			} else {

				// Create virtual task
				string virtualTaskName("VirtualTask_");
				virtualTaskName.append(Utils::itoa(virtualHwTasks.size()));

				// Create vector with tasks merged into this virtual task
				vector<HwTask*> virtualTaskComponentsVector;
				for(int j = 0; j < (int) acceptableTaskVector.size(); j++) virtualTaskComponentsVector.push_back(&acceptableTaskVector.at(j));
				for(int j = 0; j < (int) unacceptableTaskVector.size(); j++) virtualTaskComponentsVector.push_back(&unacceptableTaskVector.at(j));
				//VirtualHwTask virtualTask(virtualTaskName, virtualTaskComponentsVector, unacceptableTaskVector.at(i).getDevicePtr());
				VirtualHwTask virtualTask(virtualTaskName, virtualTaskComponentsVector, simulationDevice);
				if(VERBOSE_MODE) virtualTask.info();

				virtualHwTasks.push_back(virtualTask);

				RZSetToScan = RZFinderManager::retrieveRZSet(virtualTask.getResourceRequirements());

			}
			if(VERBOSE_MODE) cout << "Size of RZ set to scan: " << RZSetToScan.size() << endl;

			for(int j = 0; j < (int) RZSetToScan.size(); j++) {
				RZ currentRZ(RZSetToScan.at(j));
				if(!currentRZ.overlapsRZInSet(RZsubset) && atLeastOneTaskHosted(hwTasks, &currentRZ)) { // && areAllTasksHosted(acceptableTaskVector)) {

					// Check if it can host one of the optimum tasks
					bool hostingOptimumTasks = false;
					for(int k = 0; k < (int) optimumTaskVector.size(); k++) {
						if(optimumTaskVector.at(k).fitsRZ(&currentRZ)) {
							hostingOptimumTasks = true;
							break;
						}
					}

					// TODO: check if it can host other tasks

					if(!hostingOptimumTasks) {
						// Also push RZ into current set
						if(!USE_RZ_SET_DESCRIPTION_FILE) RZset.push_back(currentRZ);
						RZsubset.push_back(currentRZ);
						lastTaskAddedInScannedSubset = j;
						//if(j > lastRZInSimulationSet) lastRZInSimulationSet = j;
						break;
					}
				}
			}

		}

		//
		// EXTERNAL FRAGMENTATION REDUCTION
		//

		bool externalFragmentationOK = false;
		double externalFragmentation = 0;
		double bestExternalFragmentation = -1.0;

		vector<RZ> bestSolution(RZsubset);

		int nbIterations = 0;

		while(!externalFragmentationOK) {

			externalFragmentation = computeExternalFragmentationCost();
			if(VERBOSE_MODE) cout << "External fragmentation cost: " << externalFragmentation << endl;

			if(bestExternalFragmentation == -1.0 || externalFragmentation < bestExternalFragmentation) {
				// First solution or best solution yet
				if(VERBOSE_MODE) cout << "Found new best solution" << endl;
				bestSolution = RZsubset;
				bestExternalFragmentation = externalFragmentation;
				nbIterations = 0;

				if(externalFragmentation == 0) externalFragmentationOK = true; // One RZ or ultimate solution
			} else {
				// Not a better solution
				// After a certain number of iterations without success, stop research
				if(VERBOSE_MODE) cout << "Iteration " << nbIterations << " out of " << nbIterationsMaxExternalFragmentation << " without convenient result" << endl;
				if(nbIterations == nbIterationsMaxExternalFragmentation) externalFragmentationOK = true;
			}

			if(!externalFragmentationOK) {
				double maxDistance = computeRZDistance(1);

				if(VERBOSE_MODE) {
					cout << "RZsubset contents: ";
					printVector(RZsubset);
				}

				if(nbRZInSimulation == 1) {
					cerr << "ERROR: NOT POSSIBLE" << endl;
					exit(-1);
				} else if(nbRZInSimulation == 2) {
					// Equal distances, remove last (and worst) item
					if(VERBOSE_MODE) cout << "Removing last RZ in the set" << endl;
					RZsubset.pop_back();
				} else {
					// Remove 'furthest' RZ in the subset from the ones just added
					int RZWithMaxDistance = firstNewTask;
					double maxDistance = computeRZDistance(RZWithMaxDistance);

					if(VERBOSE_MODE) cout << "Distance for RZ with id " << RZWithMaxDistance << ": " << maxDistance << endl;

					for(int i = (RZWithMaxDistance + 1); i < (int) RZsubset.size(); i++) {
						double distance = computeRZDistance(i);

						if(VERBOSE_MODE) cout << "Distance for RZ with id " << i << ": " << distance << endl;

						if(distance > maxDistance) {
							maxDistance = distance;
							RZWithMaxDistance = i;
						}
					}

					// Effectively remove RZ
					if(VERBOSE_MODE) cout << "Removing furthest RZ in the set: id " << RZsubset.at(RZWithMaxDistance).getID() << endl;
					RZsubset.erase(RZsubset.begin() + RZWithMaxDistance);
				}

				if(VERBOSE_MODE) {
					cout << "RZsubset contents: ";
					printVector(RZsubset);
				}

				// Add a closer RZ fulfilling requirements
				// TODO: check if all tasks are hosted
				if(VERBOSE_MODE) cout << "Adding a closer RZ" << endl;

				for(int i = lastTaskAddedInScannedSubset + 1; i < (int) RZSetToScan.size(); i++) {
					if(!RZSetToScan.at(i).overlapsRZInSet(RZsubset) && atLeastOneTaskHosted(hwTasks, &RZSetToScan.at(i))) {

						RZ currentRZ(RZSetToScan.at(i));

						// Check if it can host one of the optimum tasks
						bool hostingOptimumRZ = false;
						for(int k = 0; k < (int) optimumTaskVector.size(); k++) {
							if(optimumTaskVector.at(k).fitsRZ(&currentRZ)) {
								hostingOptimumRZ = true;
								break;
							}
						}

						// TODO: check if it can host other tasks

						/*if(!hostingOptimumRZ && (computeRZDistance((int) RZsubset.size() - 1) < maxDistance)) {
							RZsubset.push_back(RZSetToScan.at(i));
							lastRZInSimulationSet = i;
							break;
						}*/

						if(!hostingOptimumRZ) {
							RZsubset.push_back(RZSetToScan.at(i));
							lastTaskAddedInScannedSubset = i;
							if(computeRZDistance((int) RZsubset.size() - 1) > maxDistance) {
								RZsubset.pop_back();
							} else {
								lastRZInSimulationSet = i;
								break;
							}
						}
					}
				}

				if((int)RZsubset.size() != nbRZInSimulation) {
					if(VERBOSE_MODE) cout << "End of virtual RZ subset..." << endl;
					externalFragmentationOK = true;
				}
			}

			//// At this point, check if all tasks can be hosted
			//if(areAllTasksHosted(tasks)) {
			//	cerr << "ERROR 2: Not all tasks hosted" << endl;
			//	exit(-1);
			//}

			nbIterations++;
		}

		RZsubset = bestSolution;
	}

	if((int) RZsubset.size() != nbRZInSimulation) cerr << "Not enough RZ" << endl;
	if(!areAllTasksHosted(hwTasks)) cerr << "Not all tasks hosted" << endl;
	if((int) RZsubset.size() != nbRZInSimulation || !areAllTasksHosted(hwTasks)) return PARTITIONING_FAILURE;
	else return PARTITIONING_SUCCESS;
}

/**
 * This methods defines the processor set to include in
 * the simulation.
 */
void Simulation::defineSimulationProcessorSet(void) {
	// Dummy implementation
	processorSubset = processorSet;
}

void Simulation::firstTaskAllocation(void) {
	firstRZAllocation();
	firstProcessorAllocation();
	if(VERBOSE_MODE) printAllocation();
	writeTaskAllocation();
}

/**
 * This method performs a one-to-many association between
 * tasks and reconfigurable zones. Each zone compatible with
 * a task in terms of resources may host the task in simulation.
 */
void Simulation::firstRZAllocation(void) {

	if(VERBOSE_MODE) cout << "Allocation step: one-to-many type" << endl;

	if(unacceptableTaskVector.empty()) {

		// One-to-many association
		for(int i = 0; i < (int) RZsubset.size(); i++) {
			for(int j = 0; j < (int) hwTasks.size(); j++) {
				if(hwTasks[j].fitsRZ(&RZsubset.at(i))) currentRZTaskAssociation[i].push_back(j);
			}
		}

	} else {

		if(VERBOSE_MODE) cout << "After partitioning" << endl;

		int nbGreedyRZFound = 0;

		// One-to-many association
		for(int i = 0; i < (int) RZsubset.size(); i++) {

			// Check if RZ can host every task
			bool RZHostsAllTasks = true;
			for(int j = 0; j < (int) hwTasks.size(); j++) {
				if(!hwTasks.at(j).fitsRZ(&RZsubset.at(i))) RZHostsAllTasks = false;
			}

			//if(i < nbRZOptimumSet) {
			//if(i < (int) RZsubset.size()/2) {
			//if(i < nbGreedyRZs/2) {
			//if(previousRZTaskAssociation.at(i).size() == tasks.size()) {
			if(RZHostsAllTasks) {
				nbGreedyRZFound++;
				//if(nbGreedyRZFound <= nbGreedyRZs/2) {
				if(nbGreedyRZFound <= nbRZOptimumSet) {
				//if(nbGreedyRZFound <= (nbRZInSimulation - nbRZUnacceptableSet)) {
					// Top RZs should not host tasks from the unacceptable task list

					for(int j = 0; j < (int) hwTasks.size(); j++) {

						// Search for the task in the unacceptable task list
						bool unacceptable = false;
						for(int k = 0; k < (int) unacceptableTaskVectorID.size(); k++) {
							if(unacceptableTaskVectorID.at(k) == j) unacceptable = true;
						}

						if(hwTasks[j].fitsRZ(&RZsubset.at(i)) && !unacceptable) currentRZTaskAssociation[i].push_back(j);
					}
				} else {
					for(int j = 0; j < (int) hwTasks.size(); j++) {
						if(hwTasks[j].fitsRZ(&RZsubset.at(i))) currentRZTaskAssociation[i].push_back(j);
					}
				}
			} else {
				// TODO
				// Keeping classical one-to-many association type
				for(int j = 0; j < (int) hwTasks.size(); j++) {
					if(hwTasks.at(j).fitsRZ(&RZsubset.at(i))) currentRZTaskAssociation[i].push_back(j);
				}
			}
		}
	}
}

void Simulation::firstProcessorAllocation(void) {

	// One-to-many association
	for(int i = 0; i < (int) processorSubset.size(); i++) {
		for(int j = 0; j < (int) swTasks.size(); j++) {
			if(swTasks.at(j).fitsProcessor(&processorSubset.at(i))) currentProcessorTaskAssociation[i].push_back(j);
		}
	}

	//// Dummy implementation
	//for(int i = 0; i < (int) processorSubset.size(); i++) {
	//	processorSubset.at(i).clearCompatibleTasks();
	//	for(int j = 0; j < (int) tasks.size(); j++) processorSubset.at(i).addCompatibleTask(tasks.at(j).getName());
	//}
}

void Simulation::taskAllocation(void) {
	RZAllocation(ALLOCATION_STRATEGY);
	processorAllocation(PROCESSOR_ALLOCATION_STRATEGY);
	if(VERBOSE_MODE) printAllocation();
	writeTaskAllocation();
}

void Simulation::processorAllocation(ProcessorAllocationStrategy strategy) {

	if(!processorSubset.empty()) {

		vector<vector<int> > taskHosts((int) hwTasks.size());

		if(VERBOSE_MODE) {
			cout << "Processor allocation step: hybrid type" << endl;
			cout << "Processor allocation strategy: " << processorAllocationStrategyName[strategy] << endl;
		}

		// First, verify that all RZs have been running
		// If not, remove them
		for(int i = 0; i < (int) processorSimulationStats.size(); i++) {
			if(processorSimulationStats.at(i).getRZStateTime(RZ_RUNNING) == 0) {
				if(VERBOSE_MODE) cout << "Processor " << processorSubset.at(i).getName() << " (" << i << ") unused! Removing it!" << endl;
				processorSubset.erase(processorSubset.begin() + i);
				currentProcessorTaskAssociation.erase(currentProcessorTaskAssociation.begin() + i);
				processorSimulationStats.erase(processorSimulationStats.begin() + i);
				i--;
				if(VERBOSE_MODE) cout << "Updating simulation subset size! New number of processors: " << processorSubset.size() << endl;
			}
		}

		// Dummy allocation: re-using one-to-many association
		//clearAllocation();
		//firstTaskAllocation();

		if(strategy == SOFT_SHORTEST_RUNTIME) {

			// Look for the shortest runtime of all RZ-task pairs
			int procid = 0;
			map<string, double> taskExecutionTimesMap = processorSimulationStats.front().getTaskExecutionTimesMap();
			map<string, double>::iterator it = taskExecutionTimesMap.begin();
			long double shortestRuntime = (long double) it->second;
			string taskName = it->first;
			for(int i = 0; i < (int) processorSimulationStats.size(); i++) {
				taskExecutionTimesMap = processorSimulationStats.at(i).getTaskExecutionTimesMap();
				for(it = taskExecutionTimesMap.begin(); it != taskExecutionTimesMap.end(); it++) {
					long double runtime = (long double) it->second;
					if(runtime < shortestRuntime) {
						shortestRuntime = runtime;
						procid = i;
						taskName = it->first;
					}
				}
			}

			if(VERBOSE_MODE) cout << "Shortest runtime for task " << taskName << " (" << shortestRuntime << ") on Processor " << procid << endl;

			// Remove worst element from allocation list
			// Search task by its name
			bool taskFound = false;
			for(int i = 0; i < (int) currentProcessorTaskAssociation.at(procid).size(); i++) {
				//if(tasks[currentRZTaskAssociation.at(rzid).at(i)].getName().find(taskName) != string::npos) {
				//if(tasks[currentProcessorTaskAssociation.at(procid).at(i)].getName().compare(taskName) == 0) {
				if(swTasks[currentProcessorTaskAssociation.at(procid).at(i)].getName().compare(taskName) == 0) {

					//cout << "Removing task " << tasks[currentProcessorTaskAssociation.at(procid).at(i)].getName() << " (id " << i << ")" << endl;
					if(VERBOSE_MODE) cout << "Removing task " << swTasks[currentProcessorTaskAssociation.at(procid).at(i)].getName() << " (id " << i << ")" << endl;

					currentProcessorTaskAssociation.at(procid).erase(currentProcessorTaskAssociation.at(procid).begin() + i);
					taskFound = true;
					break;
				}
			}

			if(VERBOSE_MODE) if(!taskFound) cerr << "ERROR: Task not found!!!!!!!!" << endl;

			//currentRZTaskAssociation.at(rzid).erase(currentRZTaskAssociation.at(rzid).begin() + taskid);

		} else if(strategy == SOFT_NO_OPTIMIZATION) {
			cout << "No software allocation optimization required" << endl;
		} else {
			cerr << "ERROR: Unknown software allocation optimization strategy" << endl;
		}

		for(int i = 0; i < (int) processorSimulationStats.size(); i++) {
			if(processorSimulationStats.at(i).getRZStateTime(RZ_RUNNING) == 0) {
				if(VERBOSE_MODE) cout << "Processor " << processorSubset.at(i).getName() << " (" << i << ") unused! Removing it!" << endl;
				processorSubset.erase(processorSubset.begin() + i);
				currentProcessorTaskAssociation.erase(currentProcessorTaskAssociation.begin() + i);
				processorSimulationStats.erase(processorSimulationStats.begin() + i);
				i--;
				if(VERBOSE_MODE)cout << "Updating simulation subset size! New number of processors: " << processorSubset.size() << endl;
			}
		}

	}
}



/**
 * This method is called after a one-to-many association
 * is found for the current RZ set. It has to restrain
 * the final number of RZ/task pairs that lead to a
 * bitstream during the implementation phase.
 */
void Simulation::RZAllocation(AllocationStrategy strategy) {

	if(!RZsubset.empty()) {

		vector<vector<int> > taskHosts((int) hwTasks.size());

		if(VERBOSE_MODE) {
			cout << "Allocation step: hybrid type" << endl;
			cout << "Allocation strategy: " << allocationStrategyName[strategy] << endl;
		}

		// Dummy allocation: re-using one-to-many association
		//clearAllocation();
		//firstTaskAllocation();

		// First, verify that all RZs have been running
		// If not, remove them
		for(int i = 0; i < (int) RZSimulationStats.size(); i++) {
			if(RZSimulationStats.at(i).getRZStateTime(RZ_RUNNING) == 0) {
				if(VERBOSE_MODE) cout << "RZ " << RZsubset.at(i).getID() << " (" << i << ") unused! Removing it!" << endl;
				nbRZInSimulation--;
				if(VERBOSE_MODE) cout << "Updating simulation subset size! New number of RZs: " << nbRZInSimulation << endl;
				RZsubset.erase(RZsubset.begin() + i);
				currentRZTaskAssociation.erase(currentRZTaskAssociation.begin() + i);
				RZSimulationStats.erase(RZSimulationStats.begin() + i);
				i--;
			}
		}

		if(strategy == HIGHEST_FRAGMENTATION) {

			// Look for the greatest internal fragmentation of all RZ-task pairs
			int rzid = 0;
			int taskid = 0;
			int maximalFragmentation = 0;

			// In case first RZs are empty
			int firstNonEmptyRZ = 0;
			while(currentRZTaskAssociation.at(firstNonEmptyRZ).empty()) firstNonEmptyRZ++;
			rzid = firstNonEmptyRZ;
			//maximalFragmentation = tasks[currentRZTaskAssociation.front().front()].computeInternalFragmentation(RZsubset.front());
			maximalFragmentation = hwTasks[currentRZTaskAssociation.at(firstNonEmptyRZ).front()].computeInternalFragmentation(RZsubset.front());
			for(int i = 0; i < (int) currentRZTaskAssociation.size(); i++) {
				for(int j = 0; j < (int) currentRZTaskAssociation.at(i).size(); j++) {
					int fragmentation = hwTasks[currentRZTaskAssociation.at(i).at(j)].computeInternalFragmentation(RZsubset.at(i));
					if(fragmentation > maximalFragmentation) {
						maximalFragmentation = fragmentation;
						rzid = i;
						taskid = j;
					}
				}
			}

			// Remove worst element from allocation list
			currentRZTaskAssociation.at(rzid).erase(currentRZTaskAssociation.at(rzid).begin() + taskid);

		} else if(strategy == HIGHEST_MEMORY_COST) {

			// Find the task with the highest memory cost and remove it

			long double highestCost = (long double) compressedReconfigurationTimes.front().front();
			int rzid = 0;
			int taskid = 0;

			for(int i = 0; i < (int) compressedReconfigurationTimes.size(); i++) {
				for(int j = 0; j < (int) compressedReconfigurationTimes.at(i).size(); j++) {
					long double cost = (long double) compressedReconfigurationTimes.at(i).at(j);
					if(cost > highestCost) {
						highestCost = cost;
						rzid = i;
						taskid = j;
					}
				}
			}

			// Retrieve task ID in currentRZTaskAssociationc and delete it
			/*for(int i = 0; i < (int) currentRZTaskAssociation.at(rzid).size(); i++) {
				if(currentRZTaskAssociation.at(rzid).at(i) == taskid) {
					currentRZTaskAssociation.at(rzid).erase(currentRZTaskAssociation.at(rzid).begin() + i);
					break;
				}
			}*/

			currentRZTaskAssociation.at(rzid).erase(currentRZTaskAssociation.at(rzid).begin() + taskid);

		} else if(strategy == SHORTEST_RUNTIME) {

			// Look for the shortest runtime of all RZ-task pairs
			int rzid = 0;
			map<string, double> taskExecutionTimesMap = RZSimulationStats.front().getTaskExecutionTimesMap();
			map<string, double>::iterator it = taskExecutionTimesMap.begin();
			long double shortestRuntime = it->second;
			string taskName = it->first;
			for(int i = 0; i < (int) RZSimulationStats.size(); i++) {
				taskExecutionTimesMap = RZSimulationStats.at(i).getTaskExecutionTimesMap();
				for(it = taskExecutionTimesMap.begin(); it != taskExecutionTimesMap.end(); it++) {
					long double runtime = (long double) it->second * (long double)RZSimulationStats.at(i).getRZStateTime(RZ_RUNNING) / 100.0L;
					if(VERBOSE_MODE) cout << "Runtime: " << runtime << endl;
					if(runtime < shortestRuntime) {
						shortestRuntime = runtime;
						rzid = i;
						taskName = it->first;
					}
				}
			}

			if(VERBOSE_MODE) cout << "Shortest runtime for task " << taskName << " (" << shortestRuntime << ") on RZ " << rzid << endl;

			// Remove worst element from allocation list
			// Search task by its name
			bool taskFound = false;
			for(int i = 0; i < (int) currentRZTaskAssociation.at(rzid).size(); i++) {
				//if(tasks[currentRZTaskAssociation.at(rzid).at(i)].getName().find(taskName) != string::npos) {
				if(hwTasks[currentRZTaskAssociation.at(rzid).at(i)].getName().compare(taskName) == 0) {

					if(VERBOSE_MODE) cout << "Removing task " << hwTasks[currentRZTaskAssociation.at(rzid).at(i)].getName() << " (id " << i << ")" << endl;

					currentRZTaskAssociation.at(rzid).erase(currentRZTaskAssociation.at(rzid).begin() + i);
					taskFound = true;
					break;
				}
			}

			if(VERBOSE_MODE) {
				if(!taskFound) cerr << "ERROR: Task not found!!!!!!!!" << endl;
			}

			//currentRZTaskAssociation.at(rzid).erase(currentRZTaskAssociation.at(rzid).begin() + taskid);

		} else {

		}

		// Verify that no RZ has been left empty
		// If so, delete it and update simulation subset size
		for(int i = 0; i < (int) currentRZTaskAssociation.size(); i++) {
			if(currentRZTaskAssociation.at(i).empty()) {
				cout << "No task allocation for RZ " << RZsubset.at(i).getID() << " (" << i << ")! Removing it!" << endl;
				cout << "Updating simulation subset size! New number of RZs: " << --nbRZInSimulation << endl;
				RZsubset.erase(RZsubset.begin() + i);
				currentRZTaskAssociation.erase(currentRZTaskAssociation.begin() + i);
			}
		}

	}
}

void Simulation::printAllocation(void) {

	// RZs
	for(int i = 0; i < (int) currentRZTaskAssociation.size(); i++) {
		cout << "RZ with id " << RZsubset.at(i).getID() << " will host tasks: ";
		for(int j = 0; j < (int) currentRZTaskAssociation.at(i).size(); j++) cout << hwTasks[currentRZTaskAssociation.at(i).at(j)].getName() << " ";
		cout << endl;
	}

	// Processors
	for(int i = 0; i < (int) currentProcessorTaskAssociation.size(); i++) {
		cout << "Processor " << processorSubset.at(i).getName() << " will host tasks: ";
		for(int j = 0; j < (int) currentProcessorTaskAssociation.at(i).size(); j++) cout << swTasks[currentProcessorTaskAssociation.at(i).at(j)].getName() << " ";
		cout << endl;
	}
}

void Simulation::writeTaskAllocation(void) {

	// RZs
	for(int i = 0; i < (int) RZsubset.size(); i++) {
		// First of all, clear RZs previous allocation
		RZsubset.at(i).clearCompatibleTasks();
		for(int j = 0; j < (int) currentRZTaskAssociation[i].size(); j++) {
			RZsubset.at(i).addCompatibleTask(hwTasks[currentRZTaskAssociation[i][j]].getName());
		}
	}

	// Processors
	for(int i = 0; i < (int) processorSubset.size(); i++) {
		// First of all, clear RZs previous allocation
		processorSubset.at(i).clearCompatibleTasks();
		for(int j = 0; j < (int) currentProcessorTaskAssociation[i].size(); j++) {
			processorSubset.at(i).addCompatibleTask(swTasks[currentProcessorTaskAssociation[i][j]].getName());
		}
	}
}

void Simulation::computeReconfigurationTimes(void) {

	// Init vectors
	compressedReconfigurationTimes.clear();
	preloadReconfigurationTimes.clear();
	bitstreamSizes.clear();

	vector<double> vecComp;
	vector<double> vecPreload;
	vector<double> vecBitstream;

	double allocationSize = 0;

	srand(1);

	for(int i = 0; i < (int) currentRZTaskAssociation.size(); i++) {

		vecComp.clear();
		vecPreload.clear();

		// Compute uncompressed bitstream size
		float Nbitstream = (float) RZsubset.at(i).computeBitstreamSize();

		if(VERBOSE_MODE) cout << "Bitstream size: " << Nbitstream << " (32-bits words)" << endl;

		for(int j = 0; j < (int) currentRZTaskAssociation.at(i).size(); j++) {

			// Standard reconfiguration time
			float Twrite = latency * Tbus + tburst * Tbus * floor(Nbitstream/Nburst) + (tburst - fmod(Nbitstream, Nburst)) * Tbus;

			if(VERBOSE_MODE) cout << "Reconfiguration time: " << Twrite << " ns" << endl;

			// Compressed reconfiguration time
			// Size of the compressed bitstream compared to the uncompressed bitstream
			float compress_ratio_g;


			// Also check if NGC file if present in the directory. If not, always use compression estimation
			string ngcFilename(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getNetlistName());
			ifstream ngcFile(ngcFilename.c_str());

			if(((BITSTREAM_GENERATION == PLANAHEAD_IMPLEMENTATION) || finalVerificationReached) && ngcFile) {

				// Close NGC file (not used directly in the program)
				ngcFile.close();

				// Go through implementation of a PlanAhead project
				cout << "PlanAhead implementation required" << endl;

				// Search for pre-existing projects in current directory
				string compressedBitstreamName("planahead/project_fortress_");
				string netlistNameOnly(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getNetlistName());
				if(netlistNameOnly.find("/") != string::npos) netlistNameOnly.erase(netlistNameOnly.begin(), netlistNameOnly.begin() + netlistNameOnly.find_last_of("/") + 1);	// Remove hierarchy
				compressedBitstreamName.append(netlistNameOnly);
				compressedBitstreamName.erase(compressedBitstreamName.begin() + compressedBitstreamName.size() - 4, compressedBitstreamName.end());	// Remove extension
				compressedBitstreamName.append("_on_");
				compressedBitstreamName.append(RZsubset.at(i).getID());
				compressedBitstreamName.append("/bitstream.z32");

				cout << "Compressed bitstream name: " << compressedBitstreamName << endl;

				ifstream compressedBitstreamFile(compressedBitstreamName.c_str());
				if(compressedBitstreamFile) {
					cout << "PlanAhead implementation already done for the same RZ/task pair!" << endl;
					compressedBitstreamFile.close();
				} else {

					// First, create a constraint file for the RZ under test
					if(VERBOSE_MODE) cout << "Creating constraints for RZ/task pair" << endl;

					string ucfName;
					ucfName.append(RZsubset.at(i).getID());
					ucfName.append(".ucf");
					string completeName("constraints/");
					completeName.append(ucfName);
					ofstream rzucf(completeName.c_str(), ios::out | ios::trunc);
					RZsubset.at(i).generateUCF("dummy_inst", "pblock_dummy_inst", rzucf);
					rzucf.close();

					// Launch script to create PlanAhead project
					if(VERBOSE_MODE) cout << "Launching script..." << endl;


#if defined(_WIN32) || defined(__WIN32__)
					string cmd("tools\\bitstream_generation.bat ");														// Executable name
#elif defined(__linux__) || defined(__APPLE__)
					string cmd("./tools/bitstream_generation.sh ");														// Executable name
#endif
					cmd.append(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getNetlistName());						// Netlist name
					cmd.append(" ");
					cmd.append(ucfName);																				// UCF name
					//cmd.append(" xc");
					cmd.append(" ");
					cmd.append(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getDeviceID());							// Full device ID (with package and speed grade)
					cmd.append(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getDevicePtr()->getDevicePackage());
					cmd.append(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getDevicePtr()->getDeviceSpeedGrade());
					cmd.append(" ");
					cmd.append(hwTasks.at(currentRZTaskAssociation.at(i).at(j)).getDevicePtr()->getDeviceID());							// Device family (no package or speed grade)
					switch(planAheadVersion) {																			// PlanAhead version
						case PLANAHEAD_12_X: cmd.append(" v12"); break;
						case PLANAHEAD_13_X: cmd.append(" v13"); break;
						case PLANAHEAD_14_X: cmd.append(" v14"); break;
						default: cerr << "ERROR: PlanAhead version not supported!" << endl;
					}

					cmd.append(" ");
					cmd.append(planAheadPath);

					//switch(hwTasks.front().getDeviceClass()) {																			// PlanAhead version
					switch(simulationDevice->getDeviceClass()) {																			// PlanAhead version
						case Virtex5:	cmd.append(" v5"); break;
						case Virtex6:	cmd.append(" v6"); break;
						case Virtex7:	cmd.append(" v7"); break;
						case Kintex7:	cmd.append(" k7"); break;
						case Artix7:	cmd.append(" a7"); break;
						case Zynq7:		cmd.append(" z7"); break;
						default: cerr << "ERROR: Device class not recognized!" << endl;
					}

					if(VERBOSE_MODE) cout << "Command to execute: " << cmd << endl;
					//int ignoredValue = system(cmd.c_str());

					if(VERBOSE_MODE) cout << "Script terminated! Retrieving size of compressed bitstream..." << endl;
				}

				// Compute compression ratio with generated compressed file size
				ifstream compressedBitstream(compressedBitstreamName.c_str());
				if(compressedBitstream) {
					compressedBitstream.seekg(0, std::ios_base::end);	// Go to end of file
					long size = (long) compressedBitstream.tellg() / 4;		// New position ( = file size)
					compressedBitstream.close();
					compress_ratio_g = ((float) size) / Nbitstream;
					if(VERBOSE_MODE) {
						cout << "Size of compressed file: " << size << " words" << endl;
						cout << "Size of uncompressed file: " << (int) Nbitstream << " words" << endl;
						cout << "Compression ratio: " << 100 - compress_ratio_g * 100.0f << " %" << endl;
					}

				} else {
					cerr << "ERROR: Cannot open compressed bitstream... Using estimation instead" << endl;
					compress_ratio_g = 1.0f - MEAN_COMPRESSION_RATIO;	// No jitter
				}

			} else {

				// Only estimate compression using mean compression value

				if(BITSTREAM_GENERATION == PLANAHEAD_IMPLEMENTATION) {
					cout << "Cannot find netlist for this task (netlist name: " << ngcFilename << ")" << endl;
					cout << "Using compression estimation instead" << endl;
				}

				//compress_ratio_g = 0.733f + (float) (rand() % 30 - 15) / 100.0f;	// Add jitter of 15% around mean value
				compress_ratio_g = 1.0f - MEAN_COMPRESSION_RATIO;	// No jitter
				//cout << "Compression ratio: " << compress_ratio_g * 100 << " %" << endl;

//				if(VERIFICATION_COMPRESSION) {
//#ifdef STANDALONE_DEBUG
//					cout << "Youhouuuuuuuu" << endl;
//#endif
//					if((int) tasks.size() == 6) {
//						switch(i) { // RZ
//							case 0:
//								if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_decoder") != string::npos) compress_ratio_g = 1.0f - 0.327f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_encoder") != string::npos) compress_ratio_g = 1.0f - 0.351f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_decoder") != string::npos) compress_ratio_g = 1.0f - 0.492f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_encoder") != string::npos) compress_ratio_g = 1.0f - 0.453f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_decoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_encoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else cerr << "ERROR: Unknown task" << endl;
//								break;
//							case 1:
//								if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_decoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_encoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_decoder") != string::npos) compress_ratio_g = 1.0f - 0.07f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_encoder") != string::npos) compress_ratio_g = 1.0f - 0.131f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_decoder") != string::npos) compress_ratio_g = 1.0f - 0.941f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_encoder") != string::npos) compress_ratio_g = 1.0f - 0.973f;
//								else cerr << "ERROR: Unknown task" << endl;
//								break;
//						}
//					} else if((int) tasks.size() == 12) {
//						switch(i) { // RZ
//							case 0:
//							case 1:
//								if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_decoder") != string::npos) compress_ratio_g = 1.0f - 0.327f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_encoder") != string::npos) compress_ratio_g = 1.0f - 0.351f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_decoder") != string::npos) compress_ratio_g = 1.0f - 0.492f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_encoder") != string::npos) compress_ratio_g = 1.0f - 0.453f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_decoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_encoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else cerr << "ERROR: Unknown task" << endl;
//								break;
//							case 2:
//							case 3:
//								if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_decoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("MPEG2_encoder") != string::npos) cerr << "ERROR: Impossible allocation" << endl;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_decoder") != string::npos) compress_ratio_g = 1.0f - 0.07f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("AES_encoder") != string::npos) compress_ratio_g = 1.0f - 0.131f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_decoder") != string::npos) compress_ratio_g = 1.0f - 0.941f;
//								else if(tasks.at(currentRZTaskAssociation.at(i).at(j)).getName().find("RS_encoder") != string::npos) compress_ratio_g = 1.0f - 0.973f;
//								else cerr << "ERROR: Unknown task" << endl;
//								break;
//						}
//					}
//				}
			}

			allocationSize += Nbitstream * compress_ratio_g;

			vecBitstream.push_back(Nbitstream * compress_ratio_g * 4 / 1024);

			float TwriteCompressed = latency * Tbus + tburst * Tbus * floor(Nbitstream * compress_ratio_g/Nburst) + (tburst - fmod(Nbitstream * compress_ratio_g, Nburst)) * Tbus
								+ Nbitstream * (1-compress_ratio_g) * Ticap;

			//compressedReconfigurationTimes.at(i).at(j) = Twrite;

			//cout << "Compressed reconfiguration time: " << (int) TwriteCompressed << " ns" << endl;
			vecComp.push_back(TwriteCompressed);

			//preloadReconfigurationTimes.at(i).at(j) = compressedReconfigurationTimes.at(i).at(j);
			vecPreload.push_back(TwriteCompressed);
		}

		if(VERBOSE_MODE) cout << "Total bitstream size: " << allocationSize * 4 / 1024 << " kB" << endl;

		compressedReconfigurationTimes.push_back(vecComp);
		preloadReconfigurationTimes.push_back(vecPreload);
		bitstreamSizes.push_back(vecBitstream);
	}
}

SwTask& Simulation::getSWTask(string impl) {
	for(int i = 0; i < (int) swTasks.size(); i++) {
		if(swTasks.at(i).getName().compare(impl) == 0) return swTasks.at(i);
	}

	cerr << "ERROR: Unable to retrieve SW task when searching by name... Exiting!" << endl;
	exit(FORTRESS_INTERNAL_ERROR_ERRCODE);
}

HwTask& Simulation::getHWTask(string impl) {
	for(int i = 0; i < (int) hwTasks.size(); i++) {
		if(hwTasks.at(i).getName().compare(impl) == 0) return hwTasks.at(i);
	}

	cerr << "ERROR: Unable to retrieve SW task when searching by name... Exiting!" << endl;
	exit(FORTRESS_INTERNAL_ERROR_ERRCODE);
}

int Simulation::getHwReconfigurableModuleNumber() {
	int number = 0;
	bool found = false;
	for(int i = 0; i < (int) hwTasks.size(); i++) {
		string currentTaskName = hwTasks.at(i).getTaskName();

		if (hwTasks.at(i).isStaticModule() == true)
			continue;

		found = false;
		for(int j = 0; j < i; j++) {
			if (hwTasks.at(j).getTaskName().compare(currentTaskName) == 0) {
				found = true;
				break;
			}
		}
		if (found == false) number++;
	}
	return number;
}

int Simulation::writeRZConfigurationFile(string filename) {

	ofstream file(filename.c_str());
	Device * device_ptr = simulationDevice;

	if(file) {

		// FPGA
		file << "FPGA " << device_ptr->getDeviceID() << endl;

		int clbCount = 0;
		map<RBType, Cell> cellMap(device_ptr->getDeviceCellMap());
		for(map<RBType, Cell>::iterator it = cellMap.begin(); it != cellMap.end(); it++) {
			if(it->second.isReconfigurable() && device_ptr->getResourceCount(it->first) != 0) {
				// Xilinx-specific
				if(it->first == CLBm || it->first == CLBl) clbCount += device_ptr->getResourceCount(it->first);
				else file << it->second.getName() << ": " << device_ptr->getResourceCount(it->first) << endl;
			}
		}

		//if(clbCount != 0) file << "CLB: " << clbCount << endl;
		if(clbCount != 0) file << "Slice: " << clbCount * 2 << endl;
		file << "end" << endl << endl;

		// RZs
		for(int i = 0; i < (int) RZsubset.size(); i++) {
			file << "RZ " << RZsubset.at(i).getID() << endl;

			map<RBType, int> constrainedResources(RZsubset.at(i).getConstrainedResourcesCount());
			clbCount = 0;
			for(map<RBType, int>::const_iterator it = constrainedResources.begin(); it != constrainedResources.end(); it++) {
				if(it->first == SliceM || it->first == SliceL) clbCount += it->second;
				else file << device_ptr->getCellName(it->first) << ": " << it->second << endl;
			}

			//if(clbCount != 0) file << "CLB: " << clbCount/2 << endl;	// Two slices <-> 1 CLB
			if(clbCount != 0) file << "Slice: " << clbCount << endl;

			file << "end" << endl;

			// Compatible tasks
			for(int j = 0; j < (int) RZsubset.at(i).getCompatibleTaskVector().size(); j++) {
				file << "Task " << RZsubset.at(i).getCompatibleTaskVector().at(j) << endl;
				file << "CompressedTime = " << (int) compressedReconfigurationTimes.at(i).at(j) << endl;
				file << "PreloadTime = " << (int) preloadReconfigurationTimes.at(i).at(j) << endl;
				file << "ContextSwitchTime = " << getHWTask(RZsubset.at(i).getCompatibleTaskVector().at(j)).getContextSwitchTime() * 1000 << endl;	// Convert from US to NS
			}

			file << "end" << endl << endl;
		}

		// Processors
		for(int i = 0; i < (int) processorSubset.size(); i++) {
			file << "Processor " << processorSubset.at(i).getName() << endl;

            // Middleware Version (core number)
			file << "CpuId = " << i << endl;

			file << "ContextSwitchTime = " << processorSubset.at(i).getContextSwitchTime() * 1000 << endl; // Convert from US to NS

			// Compatible tasks
			for(int j = 0; j < (int) processorSubset.at(i).getCompatibleTaskVector().size(); j++) {
				file << "Task " << processorSubset.at(i).getCompatibleTaskVector().at(j) << endl;
				file << "BinaryLoadingTime =  " << getSWTask(processorSubset.at(i).getCompatibleTaskVector().at(j)).getBinaryLoadingTime() * 1000 << endl;	// Convert from US to NS
			}

			file << "end" << endl << endl;
		}

		// Tasks
		for(int i = 0; i < (int) hwTasks.size(); i++) {
			file << "Task " << hwTasks.at(i).getName() << endl;

			map<RBType, int> taskResources(hwTasks.at(i).getResourceRequirementsPtr()->getPhysicalResources());
			clbCount = 0;
			for(map<RBType, int>::const_iterator it = taskResources.begin(); it != taskResources.end(); it++) {
				if(it->first == SliceM || it->first == Slice) clbCount += it->second;
				else file << device_ptr->getCellName(it->first) << ": " << it->second << endl;
			}

			//if(clbCount != 0) file << "CLB: " << clbCount/2 << endl;	// Two slices <-> 1 CLB
			if(clbCount != 0) file << "Slice: " << clbCount << endl;

			file << "end" << endl << endl;
		}

		file.close();
		return 0;
	} else {
		cerr << "ERROR: Cannot open file " << filename << " in write mode" << endl;
		return -1;
	}
}

int Simulation::launchSimulation(void) {

	// First, write the RZ configuration file
	string configurationFile("RZset_");

	configurationFile.append(Utils::itoa(nbRZInSimulation));
	configurationFile.append("_RZ_");

	if(finalVerificationReached)
		configurationFile.append("final_verification");
	else
		configurationFile.append(Utils::itoa(++simulationNumber));

	configurationFile.append(".txt");

	writeRZConfigurationFile(configurationFile);

	//generateConstraints();

	string command("");
	string simulationName(" ");

	simulationName.append(Utils::itoa(nbRZInSimulation));
	simulationName.append("_RZ_");

	if(finalVerificationReached)
		simulationName.append("final_verification");
	else
		simulationName.append(Utils::itoa(++simulationNumber));

	currentSimulationName = simulationName;

	bool commandWritten = false;

#if defined(_WIN32) || defined(__WIN32__)
	command.append("..\\RecoSim\\exe\\RecoSim.exe ");
	command.append(configurationFile);
	command.append(simulationName);
	commandWritten = true;
#elif defined(__linux__) || defined(__APPLE__)
	command.append("../RecoSim/exe/RecoSim ");
	command.append(configurationFile);
	command.append(simulationName);
	commandWritten = true;
#endif

	if(!commandWritten) {
		cerr << "ERROR: Cannot determine target architecture for system call, exiting..." << endl;
		exit(FORTRESS_INTERNAL_ERROR_ERRCODE);
	}

	/* MODIF */
	/*
	 char * configurationFileStr = new char [configurationFile.length()+1];
	 std::strcpy (configurationFileStr, configurationFile.c_str());

	 char * simulationNameStr = new char [simulationName.length()+1];
	 std::strcpy (simulationNameStr, simulationName.c_str());

	char *argv[] = {"..\\RecoSim\\exe\\RecoSim.exe ", configurationFileStr, simulationNameStr, NULL};
	recosim_exe(3, argv);
	*/
	cout << "COMMAND : " << command.c_str() << endl;
	int rawStatus = system(command.c_str());
	int status;

#if defined(_WIN32) || defined(__WIN32__)
	status = rawStatus;
#elif defined(__linux__) || defined(__APPLE__)
	if (WIFEXITED(rawStatus)) {
		status = WEXITSTATUS(rawStatus);
	} else status = RECOSIM_INTERNAL_ERROR_ERRCODE;
#endif

	return status;
}

void Simulation::generateConstraints(bool partition) {

	if(simulationDevice->getDeviceClass() != CUSTOM) {

		// Create one master UCF file
		string ucfName(UCFDirectory);
		ucfName.append("solution_");
		ucfName.append(Utils::itoa(nbRZInSimulation));
		ucfName.append("_RZ");
		if(partition) ucfName.append("_partitioned");
		ucfName.append(".ucf");


		// UCF generation
		ofstream masterUCF(ucfName.c_str(), ios::out | ios::trunc);

		// Header
		time_t rawtime;
		struct tm *timeinfo;
		time(&rawtime);
		timeinfo = localtime(&rawtime);

		masterUCF << "##########################################################" << endl;
		masterUCF << "# Reconfigurable zone constraints" << endl;
		masterUCF << "# This file has been automatically generated by FoRTReSS" << endl;
		masterUCF << "# on " << asctime(timeinfo);;
		masterUCF << "##########################################################" << endl << endl;

		for(int i = 0; i < (int) RZsubset.size(); i++) {
			string instname("dummy_inst");
			string pblockname("pblock_");
			pblockname.append(instname);
			pblockname.append(Utils::itoa(i));
			//RZsubset.at(i).generateUCF(instname);

			// Master UCF
			RZsubset.at(i).generateUCF(instname, pblockname, masterUCF);
			masterUCF << endl << endl;
		}

		masterUCF.close();

	}

	// XML generation
	string xmlName(UCFDirectory);
	xmlName.append("solution_");
	xmlName.append(Utils::itoa(nbRZInSimulation));
	xmlName.append("_RZ");
	if(partition) xmlName.append("_partitioned");
	xmlName.append(RZ_XML_EXTENSION);
	writeOutputXMLFile(xmlName);
}

void Simulation::printPRAreaResources(ostream& stream, bool CSVFile) {

	map<RBType, int> constrainedResourcesCount;
	map<RBType, int> constrainedResourcesCountWithFaRM;
	map<RBType, int> staticResourcesCount;

	// Compute dynamic implementation resources
	for(int i = 0; i < (int) RZsubset.size(); i++) {
		map<RBType, int> constrainedResourcesCountRZ = RZsubset.at(i).getConstrainedResourcesCount();
		map<RBType, int>::iterator it;
		for(it = constrainedResourcesCountRZ.begin(); it != constrainedResourcesCountRZ.end(); it++) {

			if(simulationDevice->getCell(it->first).isReconfigurable()) {
				// Merge Slice, SliceL, SliceM together
				if(it->first == Slice || it->first == SliceL || it->first == SliceM) {
					constrainedResourcesCount[Slice] += (it->second * simulationDevice->getCell(it->first).getNbCellsPerColumn());
					constrainedResourcesCountWithFaRM[Slice] += (it->second * simulationDevice->getCell(it->first).getNbCellsPerColumn());
				} else {
					constrainedResourcesCount[it->first] += (it->second * simulationDevice->getCell(it->first).getNbCellsPerColumn());
					constrainedResourcesCountWithFaRM[it->first] = constrainedResourcesCount[it->first];
				}
			}
		}

		constrainedResourcesCountWithFaRM[Slice] += FARM_NB_SLICE;
	}

	// Compute static implementation resources
	for(int i = 0; i < (int) hwTasks.size(); i++) {
		map<RBType, int> taskStaticResourcesCount = hwTasks.at(i).getTaskResourceFile()->getStaticResources();
		for(map<RBType, int>::const_iterator it = taskStaticResourcesCount.begin(); it != taskStaticResourcesCount.end(); it++) {
			// Merge Slice, SliceL, SliceM together
			if(it->first == Slice || it->first == SliceL ||it->first == SliceM) staticResourcesCount[Slice] += it->second;
			else staticResourcesCount[it->first] += it->second;
		}
	}

	// Display
	if(CSVFile) {
		const string CSV_SEPARATOR("; ");
		stream << endl << endl << endl << "=====   Static implementation versus PR implementation   =====" << endl;
		stream << "Resource type" << CSV_SEPARATOR << "Static area" << CSV_SEPARATOR << "PR area" << CSV_SEPARATOR << "Raw improvement (%)" << CSV_SEPARATOR << "Total improvement (%)" << endl;
		for(map<RBType, int>::iterator it = constrainedResourcesCount.begin(); it != constrainedResourcesCount.end(); it++) {
			if(staticResourcesCount.find(it->first) != staticResourcesCount.end()) {
				stream << fixed << setprecision(2) << simulationDevice->getCellName(it->first) << CSV_SEPARATOR << staticResourcesCount[it->first] << CSV_SEPARATOR << it->second << CSV_SEPARATOR
					<< (double)(staticResourcesCount[it->first] - it->second)/staticResourcesCount[it->first] * 100 << CSV_SEPARATOR
					<< (double)(staticResourcesCount[it->first] - constrainedResourcesCountWithFaRM[it->first])/staticResourcesCount[it->first] * 100 << endl;
			} else stream << simulationDevice->getCellName(it->first) << CSV_SEPARATOR << "0" << CSV_SEPARATOR << it->second << CSV_SEPARATOR << "-" << CSV_SEPARATOR << "-" << endl;
		}
	} else {
		stream << endl << "=====   Static implementation versus PR implementation   =====" << endl;
		stream << "Resource type:\tStatic area\t| PR area\t| Raw improvement (%)\t| Total improvement (%)" << endl;
		for(map<RBType, int>::iterator it = constrainedResourcesCount.begin(); it != constrainedResourcesCount.end(); it++) {
			if(staticResourcesCount.find(it->first) != staticResourcesCount.end()) {
				stream << fixed << setprecision(2) << simulationDevice->getCellName(it->first) << ":\t\t" << staticResourcesCount[it->first] << "\t\t| " << it->second << "\t\t| "
					<< (double)(staticResourcesCount[it->first] - it->second)/staticResourcesCount[it->first] * 100 << "\t\t\t| "
					<< (double)(staticResourcesCount[it->first] - constrainedResourcesCountWithFaRM[it->first])/staticResourcesCount[it->first] * 100 << endl;
			} else stream << simulationDevice->getCellName(it->first) << ":\t\t0\t\t| " << it->second << "\t\t| -\t\t\t| -" << endl;
		}
	}
}

void Simulation::editCurrentCSVFile(void) {

	string filename(CSVDirectory);
	filename.append("solution_");
	filename.append(Utils::itoa(nbRZInSimulation));
	filename.append("_RZ_");
	filename.append(Utils::itoa(simulationNumber));
	filename.append(".csv");

	// First, check file existence by trying to open it read only
	ifstream input(filename.c_str());
	if(input) {

		input.close();

		ofstream file(filename.c_str(), ios::out | ios::app);

		if(file) {
			if(VERBOSE_MODE) cout << "CSV file to parse: " << filename << endl;
			printPRAreaResources(file, true);
			file.close();
		} else cerr << "ERROR: Could not append CSV result file " << filename << endl;
	}

}

bool Simulation::allModuleHaveSWImplementations(void) {
	for(set<string>::const_iterator it = modulesSet.begin(); it != modulesSet.end(); it++) {
		bool moduleWithSWImpl = false;
		for(int i = 0; i < (int) swTasks.size(); i++) {
			if(swTasks.at(i).getTaskName().compare(*it) == 0) {
				moduleWithSWImpl = true;
				break;
			}
		}

		if(!moduleWithSWImpl) return false;
	}

	// Also verify that processors are available in the design
	//return true;
	return !processorSet.empty();
}

bool Simulation::isSimulationValid(int errcode) {
	return (errcode == RECOSIM_SIMULATION_SUCCESS || errcode == RECOSIM_SIMULATION_FAILED_ERRCODE);
}


int Simulation::launchArchGenFlow(void) {

	// Init RZ class parameters
	RZ::setRZShapeCost(COST_SHAPE_WEIGHT);
	RZ::setTaskCorrelationCost(COST_COMPLIANCE_WEIGHT);
	RZ::setInternalFragmentationCost(COST_INTERNAL_FRAGMENTATION_WEIGHT);
	RZ::setMinimumDistanceBetweenRZs(MINIMUM_DISTANCE_BETWEEN_RZS);
	RZ::setIgnoreRZPositionCOnstraints(IGNORE_RZ_POSITION_CONSTRAINTS);

	// Init TaskResourceFile class parameters
	TaskResourceFile::setStaticResourcesMargin(RESOURCES_MARGIN);

	// Init ResourceRequirements class parameters
	ResourceRequirements::setEnableNonRectangularRZ(ALLOW_NON_RECTANGULAR_RZ);
	ResourceRequirements::setOversizedRZTrigger(OVERSIZED_RZ_TRIGGER);
	ResourceRequirements::setPurelyReconfigurableRZ(PURELY_RECONFIGURABLE_RZ);

	// Verify if task vector has been populated
	if(hwTasks.empty() && swTasks.empty()) {
		cerr << "Simulation tasks have not been defined! Exiting...";
		return -1;
	}

	// Read synthesis reports
	for(int i = 0; i < (int) hwTasks.size(); i++) {
		int val = hwTasks.at(i).readResourceFile();
		if(val != 0) {
			cerr << "ERROR: FoRTReSS cannot read synthesis report... Exiting!" << endl;
			return -1;
		}
		hwTasks.at(i).info();
	}

	vector<RZ> RZsubsetNoPartition;

	clock_t start = clock();
	clock_t taskStart = clock();

	/////////////////////////
	// STEP 1: Search RZs //
	///////////////////////

	if(USE_RZ_SET_DESCRIPTION_FILE) {
		// For each task, update resource requirements
		for(int i = 0; i < (int) hwTasks.size(); i++) hwTasks.at(i).defineResourceRequirements();
		int xml_parsing_value = readInputXMLFile(RZ_SET_DESCRIPTION_FILENAME);
		if(xml_parsing_value != 0) {
			cerr << "ERROR: while parsing XML RZ description file" << endl;
			return -1;
		}
	} else {
		if(!hwTasks.empty()) searchRZs();
		cout << "RZ pool found in " << (double)(clock() - taskStart)/CLOCKS_PER_SEC << " seconds" << endl;
	}


	//////////////////////////
	// STEP 2: Sorting RZs //
	////////////////////////

	taskStart = clock();
	sortRZSet();
	cout << "RZ sorted in " << (double)(clock() - taskStart)/CLOCKS_PER_SEC << " seconds" << endl;

	/////////////////////////////////////////////
	// STEP 3: Determining RZ set to simulate //
	///////////////////////////////////////////

	// Processor subset determination
	defineSimulationProcessorSet();

	taskStart = clock();

	bool simulationOK = false;
	bool oneToManyAssociationOK = false;
	bool RZSetFound = false;

	double initialMemoryCost;

	bool RZSpaceOptimised = false;
	bool RZSpacePartitioned = false;
	bool partitionRZSpace = true;
	bool partitionDone = false;

	bool constraintsGenerated = false;

	bool firstTime = true;

	vector<RZ> bestRZsubset;

	int maxHwModuleNumner = getHwReconfigurableModuleNumber();
	cout << endl << "Maximum RZ before static solution : " << maxHwModuleNumner << endl;

	do {

		if(!(firstTime && allModuleHaveSWImplementations())) {
			// If one-to-many association did not succeed, increment number of RZ in simulation
			if(!oneToManyAssociationOK) {
				nbRZInSimulation++;
				constraintsGenerated = false;
				simulationNumber = 0;

				if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION) {
					if(nbRZInSimulation == (maxHwModuleNumner+1)) {
						cout << endl << "NB RZ (" << nbRZInSimulation << ") = NB Tasks (" << (int) modulesSet.size() << ") : static solution reached...exiting" << endl;
						break;
					}
				}
				else {
					if(nbRZInSimulation == (maxHwModuleNumner+1)) {
						cout << endl << "NB RZ (" << nbRZInSimulation << ") = NB Tasks (" << (int) modulesSet.size() << ") : no dynamic implementation found...exiting" << endl;
						return -1;
					}
				}

				cout << endl << endl << "----------------------" << endl << "Searching " << nbRZInSimulation << " RZs in set" << endl;
				int RZSetFoundInt = defineSimulationRZSet();
				RZSetFound = (RZSetFoundInt == 0);

			} else if(partitionRZSpace) {

				// Save old partition
				RZsubsetNoPartition = RZsubset;

				partitionRZSpace = false;

				// Try to find a better RZ simulation set by partitionning application
				int val = definePartitionedRZSet();

				partitionDone = true;

				constraintsGenerated = false;

				switch(val) {
					case PARTITIONING_SUCCESS:
						cout << "Partitioning successfull!" << endl;
						oneToManyAssociationOK = false;
						RZSpacePartitioned = true;
						clearAllocation();
						break;
					case PARTITIONING_USELESS:
						cout << "Partitioning not needed..." << endl;
						partitionRZSpace = false;
						break;
					case PARTITIONING_IMPOSSIBLE:
						cout << "Partitioning impossible!" << endl;
						partitionRZSpace = false;
						break;
					case PARTITIONING_FAILURE:
						cerr << "Partitioning failed... Exiting" << endl;
						RZsubset = RZsubsetNoPartition;
						partitionRZSpace = false;
						break;
					default:
						cerr << "Partitioning return value unknown... Exiting" << endl;
						return -1;
				}

			}
		} else {
			RZSetFound = true;
		}

		firstTime = false;

		if(RZSetFound) {

			if(partitionDone && !RZSpacePartitioned && !allocationOptimizationPossible()) {
				cout << "No further allocation optimization possible!" << endl;
				simulationOK = true;

			} else {

				//////////////////////////////
				// STEP 4: Task allocation //
				////////////////////////////

				if(!oneToManyAssociationOK) {
					clearAllocation();
					firstTaskAllocation();
				} else {
					taskAllocation();
				}

				/////////////////////////
				// STEP 5: Cost model //
				///////////////////////

				computeReconfigurationTimes();

				double memoryCost = computeMemoryCost();
				if(VERBOSE_MODE) cout << "Memory cost for current allocation: " << memoryCost << " kB" << endl;


				/////////////////////////
				// STEP 6: Simulation //
				///////////////////////

				cout << "Launching simulation with " << nbRZInSimulation << " reconfigurable zone(s) and " << processorSubset.size() << " processor(s)" << endl;

				int returnValue = launchSimulation();

				switch(returnValue) {
					case RECOSIM_SIMULATION_SUCCESS:
						cout << endl << "Simulation successful!" << endl;
						break;
					case RECOSIM_SIMULATION_FAILED_ERRCODE:
					case RECOSIM_NOT_ALL_TASKS_HOSTED_ERRCODE:
						cout << endl << "Simulation failed!" << endl;
						break;
					case RECOSIM_INTERNAL_ERROR_ERRCODE:
						cout << endl << "Recosim terminated in an unsusal way... Exiting FoRTReSS!" << endl;
						return FORTRESS_INTERNAL_ERROR_ERRCODE;
					case RECOSIM_TESTBENCH_CHECK_FAILED:
						cout << endl << "Testbench data check failed...Exiting FoRTReSS!" << endl;
						return FORTRESS_INTERNAL_ERROR_ERRCODE;
					default:
						cerr << endl << "ERROR: Unrecognized RecoSim error code... Exiting FoRTReSS!" << endl;
						return FORTRESS_INTERNAL_ERROR_ERRCODE;
				}

				if(isSimulationValid(returnValue)) {

					// Retrieve and parse result file
					int resultFile = parseResultFile();

					// Edit current CSV file to add resources considerations (only if RZ were found)
					if(nbRZInSimulation != 0) editCurrentCSVFile();

					if(resultFile != 0) {
						cerr << "Error while reading result file, exiting..." << endl;
						return -1;
					}

					if(!constraintsGenerated) {
						constraintsGenerated = true;
						generateConstraints(RZSpacePartitioned);
					}

					if(VERBOSE_MODE) printParsedResultFile();

				}

				previousRZTaskAssociation = currentRZTaskAssociation;

				if(returnValue == RECOSIM_SIMULATION_SUCCESS) {
					// Simulation succeeded
					lastValidSimulationID = simulationNumber;

					// Rename
					if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION == true) {

						renameValidSimulationFiles(RZSpaceOptimised, "_valid", true);
					}

					if(RZSpacePartitioned) {
						RZSpaceOptimised = true;
						partitionRZSpace = false;
					}

					if(!oneToManyAssociationOK) {
						minimalAllocationCost = computeCurrentAllocationCost();

						if(VERBOSE_MODE) cout << "Found first valid solution! Allocation cost: " << minimalAllocationCost << endl;

						oneToManyAssociationOK = true;
						// Save this first good simulation set
						bestRZTaskAssociation = currentRZTaskAssociation;
						bestProcessorTaskAssociation = currentProcessorTaskAssociation;
						bestRZsubset = RZsubset;
						firstRZSimulationStats = RZSimulationStats;
						firstProcessorSimulationStats = processorSimulationStats;
						if(!RZSpaceOptimised) initialMemoryCost = memoryCost;
					} else {
						int currentAllocationCost = computeCurrentAllocationCost();

						if(VERBOSE_MODE) cout << "Current allocation cost: " << currentAllocationCost << endl;

						minimalAllocationCost = currentAllocationCost;
						bestRZTaskAssociation = currentRZTaskAssociation;
						bestProcessorTaskAssociation = currentProcessorTaskAssociation;
						bestRZsubset = RZsubset;

						if(VERBOSE_MODE) cout << "Found new best allocation! New allocation cost: " << minimalAllocationCost << endl;
					}

					if(RZSpaceOptimised || !allocationOptimizationPossible()) {
						// Verify if further allocation optimization can be made
						if(!allocationOptimizationPossible()) {
							cout << "No further allocation optimization possible!" << endl;

							if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION) {
								simulationOK = false;
								RZSpaceOptimised = false;
								partitionRZSpace = true;
								oneToManyAssociationOK = false;
								simulationNumber = 0;
							}
							else
								simulationOK = true;
						}
					}
				} else {
					// Simulation failed
					if(hwTasks.empty()) {
						// Design space cannot be improved
						simulationOK = true;
					} else {
						if(RZSpacePartitioned && !RZSpaceOptimised && !backToUnpartitionedSet) { // Partitioning ok but simulation failure
							// Go back to first set
							RZsubset = RZsubsetNoPartition;
							currentRZTaskAssociation = bestRZTaskAssociation;
							currentProcessorTaskAssociation = bestProcessorTaskAssociation;
							RZSimulationStats = firstRZSimulationStats;
							processorSimulationStats = firstProcessorSimulationStats;
							partitionRZSpace = false;
							oneToManyAssociationOK = true;
							backToUnpartitionedSet = true;

							// Verify if further allocation optimization can be made
							if(!allocationOptimizationPossible()) {
								cout << "No further allocation optimization possible!" << endl;

								if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION) {
									simulationOK = false;
									RZSpaceOptimised = false;
									partitionRZSpace = true;
									oneToManyAssociationOK = false;
									simulationNumber = 0;
								}
								else
									simulationOK = true;
							}

						} else if(oneToManyAssociationOK) {
							if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION) {
								simulationOK = false;
								RZSpaceOptimised = false;
								partitionRZSpace = true;
								oneToManyAssociationOK = false;
								simulationNumber = 0;
							}
							else
								simulationOK = true;
						}
					}
				}
			}

		} else {
			cout << "RZ selection for simulation failed for " << nbRZInSimulation << " RZ(s): Increasing number of RZs" << endl;
		}

	} while(!simulationOK);

	cout << endl << endl << "====================================" << endl;

	bool fortressFailed(bestRZTaskAssociation.empty() && bestProcessorTaskAssociation.empty());	// Both 'best' vectors empty, hence no solution found

	if(fortressFailed) {
		cout << "FoRTReSS design space exploration failed!" << endl;
		cout << "Number of reconfigurable zones: " << nbRZInSimulation << endl;
		cout << "Number of processors: " << processorSubset.size() << endl;
		cout << "Consider revising architecture:" << endl;
		cout << "    - Add HW implementations to modules" << endl;
		cout << "    - Add more processors to the simulation set" << endl;

	} else {
		cout << "FoRTReSS design space exploration succeeded!" << endl;
		cout << "Number of reconfigurable zones: " << nbRZInSimulation << endl;
		cout << "Number of processors: " << processorSubset.size() << endl;

		// Retrieve best allocation
		currentRZTaskAssociation = bestRZTaskAssociation;
		currentProcessorTaskAssociation = bestProcessorTaskAssociation;
		RZsubset = bestRZsubset;
		cout << endl << "Final allocation" << endl;
		printAllocation();

		if(nbRZInSimulation != 0) {
			computeReconfigurationTimes();
			double finalMemoryCost = computeMemoryCost();
			cout << endl << "Final memory cost: " << finalMemoryCost << " kB" << endl;
			cout << "Initial memory cost: " << initialMemoryCost << " kB" << endl;
			cout << "Improvement: " << 100 - finalMemoryCost * 100 / initialMemoryCost << " %" << endl;

			printPRAreaResources(cout, false);

			if(BITSTREAM_GENERATION == FINAL_PLANAHEAD_VERIFICATION) {

				cout << endl << endl << "Verifying solution with actual netlists and compression rates" << endl;
				finalVerificationReached = true;
				currentRZTaskAssociation = bestRZTaskAssociation;
				currentProcessorTaskAssociation = bestProcessorTaskAssociation;
				computeReconfigurationTimes();	// Call required since boolean finalVerificationReached was modified since last call -> will perform PlanAhead implementations
				writeTaskAllocation();
				printAllocation();

				cout << endl << "Launching simulation with " << nbRZInSimulation << " reconfigurable zone(s) and " << processorSubset.size() << " processor(s)" << endl;
				int returnValue = launchSimulation();

				switch(returnValue) {
					case RECOSIM_SIMULATION_SUCCESS:
						cout << endl << "Simulation successful!" << endl;
						break;
					case RECOSIM_SIMULATION_FAILED_ERRCODE:
					case RECOSIM_NOT_ALL_TASKS_HOSTED_ERRCODE:
						cout << endl << "Simulation failed!" << endl;
						break;
					case RECOSIM_INTERNAL_ERROR_ERRCODE:
						cout << endl << "Recosim terminated in an unsusal way... Exiting FoRTReSS!" << endl;
						return FORTRESS_INTERNAL_ERROR_ERRCODE;
					case RECOSIM_TESTBENCH_CHECK_FAILED:
						cout << endl << "Testbench data check failed...Exiting FoRTReSS!" << endl;
						return FORTRESS_INTERNAL_ERROR_ERRCODE;
					default:
						cerr << endl << "ERROR: Unrecognized RecoSim error code... Exiting FoRTReSS!" << endl;
						return FORTRESS_INTERNAL_ERROR_ERRCODE;
				}

				// Retrieve and parse result file
				int resultFile = parseResultFile();

				if(resultFile != 0) {
					cerr << "Error while reading result file, exiting..." << endl;
					return -1;
				}

				printParsedResultFile();

				finalMemoryCost = computeMemoryCost();
				cout << endl << "Final memory cost: " << finalMemoryCost << " kB" << endl;
				cout << "Initial memory cost: " << initialMemoryCost << " kB" << endl;
				cout << "Improvement: " << 100 - finalMemoryCost * 100 / initialMemoryCost << " %" << endl;

			}

			/////////////////////////////
			// STEP 7: UCF generation //
			///////////////////////////

			if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION == false) {
				cout << endl << "Generating associated constraints...";
				generateConstraints(RZSpaceOptimised);
				cout << "done" << endl;
			}
		}

		if (RUN_SIMULATION_UNTIL_STATIC_SOLUTION == false)
			renameValidSimulationFiles(RZSpaceOptimised, "final_solution", false);
	}

	cout << endl << "Aggregate simulation time: " << (double)(clock() - taskStart)/CLOCKS_PER_SEC << " seconds" << endl;
	cout << "Overall FoRTReSS execution time: " << (double)(clock() - start)/CLOCKS_PER_SEC << " seconds" << endl << endl;

	if(fortressFailed) return FORTRESS_NO_SOLUTION_ERRCODE;
	else return 0;
}

bool Simulation::allocationOptimizationPossible(void) {
	return !((ALLOCATION_STRATEGY == NO_OPTIMIZATION && PROCESSOR_ALLOCATION_STRATEGY == SOFT_NO_OPTIMIZATION) ||
						(ALLOCATION_STRATEGY == NO_OPTIMIZATION && processorSubset.empty()) ||
						(PROCESSOR_ALLOCATION_STRATEGY == SOFT_NO_OPTIMIZATION && RZsubset.empty()));
}

void Simulation::renameValidSimulationFiles(bool partition, string newValidSimulationName, bool addNewName) {

	string validSimulationName("");
	validSimulationName.append(Utils::itoa(nbRZInSimulation));
	validSimulationName.append("_RZ_");
	validSimulationName.append(Utils::itoa(lastValidSimulationID));

	string tmpOldName, tmpNewName;

	// Log
	tmpOldName = logDirectory;
	tmpOldName.append("solution_");
	tmpOldName.append(validSimulationName);
	tmpOldName.append(".log");
	tmpNewName = logDirectory;
	if (addNewName) tmpNewName.append("solution_").append(validSimulationName);
	tmpNewName.append(newValidSimulationName);
	tmpNewName.append(".log");
	renameFile(tmpOldName, tmpNewName);

	// Trace
	tmpOldName = traceDirectory;
	tmpOldName.append("solution_");
	tmpOldName.append(validSimulationName);
	tmpOldName.append(".vcd");
	tmpNewName = traceDirectory;
	if (addNewName) tmpNewName.append("solution_").append(validSimulationName);
	tmpNewName.append(newValidSimulationName);
	tmpNewName.append(".vcd");
	renameFile(tmpOldName, tmpNewName);

	// CSV
	tmpOldName = CSVDirectory;
	tmpOldName.append("solution_");
	tmpOldName.append(validSimulationName);
	tmpOldName.append(".csv");
	tmpNewName = CSVDirectory;
	if (addNewName) tmpNewName.append("solution_").append(validSimulationName);
	tmpNewName.append(newValidSimulationName);
	tmpNewName.append(".csv");
	renameFile(tmpOldName, tmpNewName);

	// UCF
	string validUCFID("");
	validUCFID.append(Utils::itoa(nbRZInSimulation));
	validUCFID.append("_RZ");
	if(partition) validUCFID.append("_partitioned");
	tmpOldName = UCFDirectory;
	tmpOldName.append("solution_");
	tmpOldName.append(validUCFID);
	tmpOldName.append(".ucf");
	tmpNewName = UCFDirectory;
	if (addNewName) tmpNewName.append("solution_").append(validUCFID);
	tmpNewName.append(newValidSimulationName);
	tmpNewName.append(".ucf");
	renameFile(tmpOldName, tmpNewName);

	// XML
	string validXMLID("");
	validXMLID.append(Utils::itoa(nbRZInSimulation));
	validXMLID.append("_RZ");
	if(partition) validXMLID.append("_partitioned");
	tmpOldName = XMLDirectory;
	tmpOldName.append("solution_");
	tmpOldName.append(validXMLID);
	tmpOldName.append(RZ_XML_EXTENSION);
	tmpNewName = XMLDirectory;
	if (addNewName) tmpNewName.append("solution_").append(validXMLID);
	tmpNewName.append(newValidSimulationName);
	tmpNewName.append(RZ_XML_EXTENSION);
	renameFile(tmpOldName, tmpNewName);

}

void Simulation::renameFile(string oldfile, string newfile) {

	if(VERBOSE_MODE) cout << "Removing old file: " << newfile << "...";
	int result = remove(newfile.c_str());
	if(VERBOSE_MODE) {
		if(result == 0) cout << "done" << endl;
		else cout << endl << "Remove operation failed" << endl;
	}
	if(VERBOSE_MODE) cout << "Renaming " << oldfile << " to " << newfile << "...";
	result = rename(oldfile.c_str(), newfile.c_str());
	if(VERBOSE_MODE) {
		if(result == 0) cout << "done" << endl;
		else {
			cout << endl;
			perror("Error renaming file");
		}
	}
}

void Simulation::setLogDirectory(string dir) {
	logDirectory = dir;
}

void Simulation::setTraceDirectory(string dir) {
	traceDirectory = dir;
}

void Simulation::setCSVDirectory(string dir) {
	CSVDirectory = dir;
}

void Simulation::setUCFDirectory(string dir) {
	UCFDirectory = dir;
}

void Simulation::setXMLDirectory(string dir) {
	XMLDirectory = dir;
}

void Simulation::setHwTaskVector(vector<HwTask> &vec) {
	hwTasks = vec;
}

void Simulation::setSwTaskVector(vector<SwTask> &vec) {
	swTasks = vec;
}

void Simulation::setModuleSet(set<string> &modset) {
	modulesSet = modset;
}

void Simulation::setProcessorVector(vector<Processor> &vec) {
	processorSet = vec;
}

void Simulation::setInterfaceVector(vector<InterfaceLocation> &vec) {
	//physicalInterfacesLocation = vec;

	for(int i = 0; i < (int) vec.size(); i++) {
		//physicalInterfacesLocation.insert(vec.at(i).getInterfaceExpressionVector().front());
		string if_name(vec.at(i).getInterfaceExpressionVector().front());
		physicalInterfacesLocation[if_name]++;
		//if(physicalInterfacesLocation.find(if_name) != physicalInterfacesLocation.end()) physicalInterfacesLocation[if_name] = 1;
		// TODO: change the way expression vector is handled
	}
}

void Simulation::setAllowRectangularRZ(bool val) {
	ALLOW_NON_RECTANGULAR_RZ = val;
}

void Simulation::setResourcessMargin(double val) {
	RESOURCES_MARGIN = val;
}

void Simulation::setCostShapeWeight(double val) {
	COST_SHAPE_WEIGHT = val;
}

void Simulation::setCostComplianceWeight(double val) {
	COST_COMPLIANCE_WEIGHT = val;
}

void Simulation::setCheckRZRedundancy(bool val) {
	CHECK_RZ_REDUNDANCY = val;
}

void Simulation::setImplementationMetric(ImplementationMetric val) {
	IMPLEMENTATION_METRIC = val;
}

void Simulation::setTriggerFactorUnacceptable(double val) {
	TRIGGER_FACTOR_UNACCEPTABLE = val;
}

void Simulation::setTriggerFactorOptimum(double val) {
	TRIGGER_FACTOR_OPTIMUM = val;
}

void Simulation::setAllocationStrategy(AllocationStrategy val) {
	ALLOCATION_STRATEGY = val;
}

void Simulation::setProcessorAllocationStrategy(ProcessorAllocationStrategy val) {
	PROCESSOR_ALLOCATION_STRATEGY = val;
}

void Simulation::setBitstreamGeneration(BitstreamGeneration val) {
	BITSTREAM_GENERATION = val;
}

void Simulation::setTbus(float val) {
	Tbus = val;
}

void Simulation::setTicap(float val) {
	Ticap = val;
}

void Simulation::setNburst(float val) {
	Nburst = val;
}

void Simulation::setTburst(float val) {
	tburst = val;
}

void Simulation::setLatency(float val) {
	latency = val;
}

void Simulation::setFIFODepth(float val)  {
	fifoDepth = val;
}

void Simulation::displayParameters(void) {

	cout << "STEP 0: Environment" << endl;
	cout << "\tPlanAhead version: " << planAheadVersionName[(int) planAheadVersion] << endl;
	cout << "\tPlanAhead path: " << planAheadPath << endl;
	cout << "\tVerbose mode: " << (VERBOSE_MODE ? "yes" : "no") << endl;

	cout << endl << "STEP 1: RZ determination per task" << endl;
	cout << "\tUse RZ set description file: " << (USE_RZ_SET_DESCRIPTION_FILE ? "yes" : "no") << endl;
	if(USE_RZ_SET_DESCRIPTION_FILE) cout << "\tUse RZ set description filename: " << RZ_SET_DESCRIPTION_FILENAME << endl;
	cout << "\tAllow for non-rectangular RZs: " << (ALLOW_NON_RECTANGULAR_RZ ? "yes" : "no") << endl;
	cout << "\tOversized RZ trigger: " << OVERSIZED_RZ_TRIGGER << " %" << endl;
	cout << "\tResources margin for routing: " << RESOURCES_MARGIN << " %" << endl;
	cout << "\tPurely reconfigurable RZs: " << (PURELY_RECONFIGURABLE_RZ ? "yes" : "no") << endl;
	cout << "\tIgnore RZ position constraints: " << (IGNORE_RZ_POSITION_CONSTRAINTS ? "yes" : "no") << endl;

	cout << endl << "STEP 2: RZ sort" << endl;
	cout << "\tCost component shape weight: " << COST_SHAPE_WEIGHT << endl;
	cout << "\tCost component compliance weight: " << COST_COMPLIANCE_WEIGHT << endl;
	cout << "\tCost component internal fragmentation weight: " << COST_INTERNAL_FRAGMENTATION_WEIGHT << endl;
	cout << "\tCheck RZ redundancy: " << (CHECK_RZ_REDUNDANCY ? "yes" : "no") << endl;

	cout << endl << "STEP 3: RZ selection for simulation" << endl;
	cout << "\tImplementation metric: " << implementationMetricName[(int) IMPLEMENTATION_METRIC] << endl;
	cout << "\tUnacceptable partitionning trigger: " << TRIGGER_FACTOR_UNACCEPTABLE << " %" << endl;
	cout << "\tAcceptable partitionning trigger: " << TRIGGER_FACTOR_OPTIMUM << " %" << endl;
	cout << "\tMinimum distance between RZs: " << MINIMUM_DISTANCE_BETWEEN_RZS << " column(s)" << endl;

	cout << endl << "STEP 4: Allocation" << endl;
	cout << "\tAllocation optimization strategy: " << allocationStrategyName[(int) ALLOCATION_STRATEGY] << endl;
	cout << "\tProcessor allocation optimization strategy: " << processorAllocationStrategyName[(int) PROCESSOR_ALLOCATION_STRATEGY] << endl;

	cout << endl << "STEP 5: PR cost model" << endl;
	cout << "\tBitstream generation: " << bitstreamGenerationName[(int) BITSTREAM_GENERATION] << endl;
	cout << "\tCompression ratio: " << MEAN_COMPRESSION_RATIO << " %" << endl;
	cout << "\tTbus: " << Tbus << " ns" << endl;
	cout << "\tTicap: " << Ticap << " ns" << endl;
	cout << "\tNburst: " << Nburst << " words" << endl;
	cout << "\ttburst: " << tburst << " ns" << endl;
	cout << "\tLatency: " << latency << " ns" << endl;
	cout << "\tFIFO depth: " << fifoDepth << " 32-bit words" << endl;
}
