/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    resourceRequirements.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-20
 * @section DESCRIPTION
 *			Representation of a task resource requirements. These requirements are
 *			specified for each different resource type (or RBType) and used to 
 *			determine a set of reconfigurable zones (RZs) that could possibly host
 *			tasks based on these resource requirements.
 */

#ifndef RESOURCE_REQUIREMENTS_H
#define RESOURCE_REQUIREMENTS_H

#include "RZ.h"
#include "device.h"
#include <iostream>
#include <string>
#include <vector>
#include <map>

/*!
 * @brief Enumeration of possible return values for the Tailor RZ method
 */
enum TailorRZState {Success, EndOfLine, ColumnNotUsed, LineNotUsed, HasSiblings, RestrictedColumn, BadRZShape};

/*!
 * @brief Representation of a task resource requirements. These requirements are
 * specified for each different resource type (or RBType) and used to 
 * determine a set of reconfigurable zones (RZs) that could possibly host
 * tasks based on these resource requirements.
 */
class ResourceRequirements {
	
private:

	static double OVERSIZED_RZ_TRIGGER;			// Trigger value to consider a RZ oversized
	static bool NON_RECTANGULAR_RZ;				// Allowing/Prohibiting the search for non-rectangular zones
	static int nbObjects;						// Total number of resource requirements instantiated
	static bool PURELY_RECONFIGURABLE_RZ;

	std::map<RBType, int> physicalResources;	// Resource requirements
	Device *device;								// Targeted device
	int objectID;								// This object unique identifier

	int rz_max_height;							// RZ maximum height in the whole set
	
	std::vector<RZ> RZset;						// Resulting RZ set

public:
	/*!
	 * @brief Class constructor. Infers a unique object identifier
	 * @param m The map representing the resource requirements
	 * @param dev Pointer to the device that sould be considered
	 */
	ResourceRequirements(std::map<RBType, int> m, Device* dev) : physicalResources(m), device(dev) {
		objectID = ++nbObjects;
		rz_max_height = dev->getNbLines();
	}

	ResourceRequirements() {
		device = 0;
		objectID = ++nbObjects;
		rz_max_height = 0;
	}

	~ResourceRequirements() {}

	// Getters
	Device* getDevicePtr(void);
	std::map<RBType, int> getPhysicalResources(void);
	std::vector<RZ> getRZSet(void);
	std::vector<RZ>* getRZSetPtr(void);
	int getTaskCost(void) const;

	// RZ creation
	void findRZs(void);
	TailorRZState createRectangularRZ(RZ *rz);
	TailorRZState tailorRZ(RZ *rz);
	bool fitsRZ(RZ *rz);
	void addRZToSet(RZ *rz);

	// Resource operations
	bool operator==(const ResourceRequirements &) const;
	bool operator<(const ResourceRequirements &) const;
	bool requiresResources(RBType rbtype) const;
	int computeInternalFragmentation(RZ &rz);
	bool isRZOversized(RZ *rz);
	void findStaticResources(RZ &rz) const;

	// Static member modification
	static void setEnableNonRectangularRZ(bool);
	static void setOversizedRZTrigger(double val);
	static void setPurelyReconfigurableRZ(bool);

	// Misc
	void info(void) const;

	void setRZResourceWastage(RZ *rz);

private:
	bool RZHasEnoughResources(RZ *rz, RBType rbtype);
};

#endif
