/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    resourceRequirements.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-20
 * @section DESCRIPTION
 *			Representation resource requirements for a task
 */

#include "resourceRequirements.h"

#include <ctime>
#include <cstdlib>
using namespace std;

//
// Initialize static members
//
bool ResourceRequirements::NON_RECTANGULAR_RZ = false;
double ResourceRequirements::OVERSIZED_RZ_TRIGGER = 10;
int ResourceRequirements::nbObjects = 0;
bool ResourceRequirements::PURELY_RECONFIGURABLE_RZ = false;


/*!
 * @brief Enable/disable the search for non-rectangular reconfigurable zones
 * @param val New parameter value
 */
void ResourceRequirements::setEnableNonRectangularRZ(bool val) {
	NON_RECTANGULAR_RZ = val;
}

/*!
 * @brief Set the trigger value for considering a zone as oversized
 * @param val New parameter value
 */
void ResourceRequirements::setOversizedRZTrigger(double val) {
	OVERSIZED_RZ_TRIGGER = val;
}

void ResourceRequirements::setPurelyReconfigurableRZ(bool val) {
	PURELY_RECONFIGURABLE_RZ = val;
}

/*!
 * @brief Get a pointer to the deivce linked to these resource requirements
 * @return Pointer to the device
 */
Device* ResourceRequirements::getDevicePtr(void) {
	return device;
}

/*!
 * @brief Get the physical resources needed
 * @return A map indicating for each type of resources the amount needed
 */
map<RBType, int> ResourceRequirements::getPhysicalResources(void) {
	return physicalResources;
}

/*!
 * @brief Get the set of reconfigurable zones that fit the resource requirements
 * @return A vector of reconfigurable zones
 */
vector<RZ> ResourceRequirements::getRZSet(void) {
	return RZset;
}

/*!
 * @brief Get the set of reconfigurable zones that fit the resource requirements
 * @return A pointer to vector of reconfigurable zones infered
 */
vector<RZ>* ResourceRequirements::getRZSetPtr(void) {
	return &RZset;
}


/*!
 * @brief This function finds RZs in the given device regarding tasks
 * needs in terms of resources.
 */
void ResourceRequirements::findRZs(void) {

	clock_t start = clock();
	int x0 = 0;
	int y0 = 0;
	int height = 0;
	TailorRZState state = Success;

	rz_max_height = device->getNbLines();

	// Iterate through device height for RZ upper-left corner
	for(y0 = 0; y0 < device->getNbLines(); y0++) {

		// Iterate on RZ height
		for(height = 1; (height <= rz_max_height) && (height <= (device->getNbLines() - y0)); height++) {

			// Iterate through device width for RZ upper-left corner
			for(x0 = 0; x0 < device->getNbColumns(); x0++) {

				RZ rectangular_rz(x0, y0, height, device, false);
				if(createRectangularRZ(&rectangular_rz) == EndOfLine) {
					break; // Changes current working line
				}

				if(NON_RECTANGULAR_RZ) {
					RZ current_rz(x0, y0, height, device, false);
					state = tailorRZ(&current_rz);

					if(state == EndOfLine) {
						break; // Changes current working line
					} else if(state == HasSiblings) {
						RZ sym_rz(x0 + current_rz.getResourcesArrayPtr()->size(), y0, height, device, true);
						tailorRZ(&sym_rz);
					} else {
						// Do nothing
					}
				}

			} // End for loop x0

		} // End for loop height

	} // End for loop y0

	// Clear tables (tmp)
	map<RBType, int>::iterator it = physicalResources.begin();
	while(it != physicalResources.end()) {
		if(it->second == 0) {
			physicalResources.erase(it++);
		} else {
			it++;
		}
	}

	cout << RZset.size() << " RZs found in " << (double)(clock() - start)/CLOCKS_PER_SEC << " seconds" << endl;

}


/*!
 * @brief Verify if a reconfigurable zone is oversized compared to the
 * resource requirements used to build it.
 * @param rz The reconfigurable zone to consider
 * @return true if the zone should be considered as oversized
 */
bool ResourceRequirements::isRZOversized(RZ *rz) {

	double minimumBistreamSize = 0;
	int rzBitstreamSize = 0;
	const double PERCENT = 100.0;

	for(map<RBType,int>::iterator it = physicalResources.begin(); it != physicalResources.end(); it++) {

		minimumBistreamSize += (double)it->second * (device->getCell(it->first).getBitstreamSizePerColumn() / 4);  // in words
	}

	rzBitstreamSize = rz->computeBitstreamSize();

	if(rzBitstreamSize < minimumBistreamSize) {
		cerr << "ERROR: Minimum bitstream size (" << minimumBistreamSize << ") greater than actual bitstream size (" << rzBitstreamSize << ")" << endl;
		exit(-1);
	}

	return (((double) (rzBitstreamSize - minimumBistreamSize) / (double) minimumBistreamSize) > (OVERSIZED_RZ_TRIGGER/PERCENT));

	// TEST ONLY
	//return true; // DUMMY -> No rectangular zone
	//return false;
}

/*!
 * @brief Create a rectangular reconfigurable zone accomodating the resource requirements
 * @param rz The reconfigurable zone to tailor
 * @return EndOfLine if the zone could not be extended because the end of the FPGA has been reached,
 * ColumnNotUsed if a column is not used (hence a similar zone has already been found and added to the set),\n
 * LineNotUsed if a whole line is not used (hence a similar zone has already been found and added to the set),\n
 * HasSiblings if some items in the reconfigurable zone may be re-arranged to create a new one,\n
 * RestrictedColumn if a column could not be used by the zone,\n
 * Success if the zone has been found and has no siblings.
 */
TailorRZState ResourceRequirements::createRectangularRZ(RZ *rz) {

	vector<vector<pair<RBType, ResourceConstraint> > > * array_ptr = rz->getResourcesArrayPtr();
	RZExtensionReturnValue extension = SUCCESS;
	int i = 0;

	while(!fitsRZ(rz) || !rz->isRZWellShaped()) {
	//while(!fitsRZ(rz)) {
		extension = rz->extendRZ();
		if(extension == SUCCESS) {
			for(i = 0; i < rz->getHeight(); i++) {
				if(requiresResources(array_ptr->back()[i].first)) {
					if(device->isRBReconfigurable(array_ptr->back()[i].first)) {
						array_ptr->back()[i].second = USED;
					} else {
						array_ptr->back()[i].second = NOT_RECONFIGURABLE;
					}
				} else array_ptr->back()[i].second = UNUSED;
			}
			rz->updateConstrainedResources();

#ifdef DEBUG
			rz->printRZ();
			rz->info();
#endif
		} else if(extension == END_OF_LINE) {
			return EndOfLine;
		} else if(extension == RESTRICTED_COLUMN) {
			return RestrictedColumn;
		} else {
			// Do nothing
		}
	}

	if(array_ptr->size() == 1) {
		rz_max_height = rz->getHeight();
	}

	// Verify if RZ is purely reconfigurable
	if(PURELY_RECONFIGURABLE_RZ && !rz->isPurelyReconfigurable()) return RestrictedColumn;

	// If there is too much difference between required area and constrained area, do not add RZ to set
	if(!isRZOversized(rz)) {
		setRZResourceWastage(rz);
		rz->computeCost();
		RZset.push_back(*rz);
	}

	return Success;
}

/*!
 * @brief This method tailors the RZ passed as parameter to fit current task.
 * @param rz The reconfigurable zone to tailor
 * @return EndOfLine if the zone could not be extended because the end of the FPGA has been reached,
 * ColumnNotUsed if a column is not used (hence a similar zone has already been found and added to the set),\n
 * LineNotUsed if a whole line is not used (hence a similar zone has already been found and added to the set),\n
 * HasSiblings if some items in the reconfigurable zone may be re-arranged to create a new one,\n
 * RestrictedColumn if a column could not be used by the zone,\n
 * Success if the zone has been found and has no siblings.
 */
TailorRZState ResourceRequirements::tailorRZ(RZ *rz) {

	vector<vector<pair<RBType, ResourceConstraint> > > * array_ptr = rz->getResourcesArrayPtr();

	int lastHeight = 0;
	int currentHeight = rz->getHeight();

	int nbResourcesUsed = 0;
	int nbReconfigurableResources = 0;
	int i = 0;
	int j = 0;

	RBType resourceType = Slice;

	while(!fitsRZ(rz)) {

		RZExtensionReturnValue extension = rz->extendRZ();
		if(extension == SUCCESS) {

			lastHeight = currentHeight;

			// Check resource needs for each clock domain
			nbResourcesUsed = 0;
			nbReconfigurableResources = 0;
			for(i = 0; i < rz->getHeight(); i++) {
				if(device->isRBReconfigurable(array_ptr->back()[i].first)) {
					nbReconfigurableResources++;
					if(!RZHasEnoughResources(rz, array_ptr->back()[i].first)) {
						array_ptr->back()[i].second = USED;
						nbResourcesUsed++;
						rz->updateConstrainedResources();
					} else {
						array_ptr->back()[i].second = UNUSED;
					}
				} else {
					array_ptr->back()[i].second = NOT_RECONFIGURABLE;
				}
			}

			rz->updateConstrainedResources();

			// Check if some resources have been constrained in the column
			if(nbResourcesUsed != 0) {
				int i = 0;
				while(i < rz->getHeight() && array_ptr->back().at(i).second != UNUSED) {
					i++;
				}
				currentHeight = i;
			} else {
				// In the case of different ressources, might not be an issue (i.e. BRAM not constrained)
				// If the first column is not constrained or similar resources found, then RZ is not good
				int current_column = rz->getResourcesArrayPtr()->size();
				if(current_column == 1) {
					return ColumnNotUsed;
				} else {
					RBType currentRBType, previousRBType;
					if(!rz->isReversed()) {	// Care for reversed RZs
						currentRBType = device->getColumnRBType(rz->getX0() + current_column);
						previousRBType = device->getColumnRBType(rz->getX0() + current_column - 1);
					} else {
						currentRBType = device->getColumnRBType(rz->getX0() - current_column);
						previousRBType = device->getColumnRBType(rz->getX0() - current_column + 1);
					}
					if((currentRBType == CLBl || currentRBType == CLBm) && (previousRBType == CLBl || previousRBType == CLBm)) {
						return ColumnNotUsed;
					} else if(currentRBType == previousRBType) {
						return ColumnNotUsed;
					} else {
						// Do nothing
					}
				}
			}

			rz->updateConstrainedResources();

#ifdef DEBUG
			rz->printRZ();
			rz->info();
#endif
		} else if(extension == END_OF_LINE) {
			return EndOfLine;
		} else if(extension == RESTRICTED_COLUMN) {
			return RestrictedColumn;
		} else {
			// Do nothing
		}
	}

	// Verify if RZ is purely reconfigurable
	if(PURELY_RECONFIGURABLE_RZ && !rz->isPurelyReconfigurable()) return RestrictedColumn;

	// Check if first line has some USED items
	i = 0;
	bool resUsed = false;
	while(i < (int) array_ptr->size() && !resUsed) {
		if(array_ptr->at(i).back().second == USED) resUsed = true;
		i++;
	}

	// Check if last line has some USED items
	i = 0;
	while(i < (int) array_ptr->size() && !resUsed) {
		if(array_ptr->at(i).back().second == USED) resUsed = true;
		i++;
	}
	if(!resUsed) {
		return LineNotUsed;
	} else {
		// Do nothing
	}


	// Re-arrange last column if necessary
	if(nbResourcesUsed != nbReconfigurableResources) {

		// All resources not constrained and last column -> try to re-arrange column
		for(i = 0; i < (nbReconfigurableResources - nbResourcesUsed); i++) {
			RZ rz_copy(*rz);
			// Seek first USED item and turn it into UNUSED
			j = 0;
			while(array_ptr->back()[j].second != USED) j++;

			// Check previous column, if this item is UNUSED, then stop
			if(j >= (lastHeight - 1)) {
				break;
			} else {
				// Do nothing
			}

			array_ptr->back()[j].second = UNUSED;
			resourceType = array_ptr->back()[j].first;
			// Seek first UNUSED item and turn it into USED
			j++;
			while(j < rz->getHeight() && (array_ptr->back()[j].second != UNUSED || array_ptr->back()[j].first != resourceType)) j++;
			if(j < rz->getHeight()) {
				array_ptr->back()[j].second = USED;
				tailorRZ(&rz_copy);
			} else {
				// Unable to move the additional blocks, undo changes by replacing current rz by its copy
				rz = &rz_copy;
				break;
			}
		}
	}

	// RZ is defined, re-arrange first column is necessary
	if(array_ptr->size() > 2 && rz->getHeight() > 1) {
		int nbResourcesUnusedFirstColumn = 0;
		for(i = 0; i < rz->getHeight(); i++) {
			if(array_ptr->at(0).at(i).second == UNUSED) {
				nbResourcesUnusedFirstColumn++;
			} else {
				// Do nothing
			}
		}

		int nextColumn = 0;
		int nbResourcesUnusedSecondColumn = 0;
		do {
			nextColumn++;
			nbResourcesUnusedSecondColumn = 0;
			for(int i = 0; i < rz->getHeight(); i++) {
				if(array_ptr->at(nextColumn).at(i).second == UNUSED) {
					nbResourcesUnusedSecondColumn++;
				} else {
					// Do nothing
				}
			}
		} while(nbResourcesUnusedSecondColumn == rz->getHeight());

		int maxHeightPermutation = (rz->getHeight() - nbResourcesUnusedFirstColumn > nbResourcesUnusedSecondColumn)
			? nbResourcesUnusedFirstColumn : nbResourcesUnusedFirstColumn - nbResourcesUnusedSecondColumn;

		// Try to re-arrange column
		for(i = 0; i < maxHeightPermutation; i++) {
			RZ rz_copy(*rz);
			// Seek first USED item and turn it into UNUSED
			j = 0;
			while(array_ptr->front()[j].second != USED) j++;
			array_ptr->front()[j].second = UNUSED;
			resourceType = array_ptr->front()[j].first;
			// Seek first UNUSED item and turn it into USED
			j++;
			while(j < rz->getHeight() && (array_ptr->front()[j].second != UNUSED || array_ptr->front()[j].first != resourceType)) j++;
			if(j < rz->getHeight()) {
				array_ptr->front()[j].second = USED;
				RZ rz_copy_to_add(*rz);
				addRZToSet(&rz_copy_to_add);

			} else {
				// Unable to move the additional blocks, undo changes by replacing current rz by its copy
				rz = &rz_copy;
				break;
			}
		}
	} else if(array_ptr->size() == 1) {
		rz_max_height = rz->getHeight();
	} else {
		// Do nothing
	}

	addRZToSet(rz);

	// Search for siblings if necessary (i.e. last column not full)
	if(nbResourcesUsed != nbReconfigurableResources) {
		return HasSiblings;
	} else {
		return Success;
	}
}

/*!
 * @brief Add a reconfigruable zone to the resulting set after some verification
 * @param rz The reconfigurable zone to tailor
 */
void ResourceRequirements::addRZToSet(RZ *rz) {

	vector<vector<pair<RBType, ResourceConstraint> > > * array_ptr = rz->getResourcesArrayPtr();
	bool isRectangular = true;
	int i = 0;
	int j = 0;

	// Check for UNUSED blocks: if none, then rectangular zone, already added to the set

	for(i = 0; i < (int) array_ptr->size(); i++) {
		for(j = 0; j < rz->getHeight(); j++) {
			if(array_ptr->at(i)[j].second == UNUSED) {
				isRectangular = false;
				break;
			} else {
				// Do nothing
			}
		}
	}

	if(!isRectangular && rz->isRZWellShaped()) {
		setRZResourceWastage(rz);
		rz->computeCost();
		findStaticResources(*rz);
		RZset.push_back(*rz);
	}
}

/*!
 * @brief This method checks whether the RZ passed as parameter has enough
 * rbtype resources constrained for the task. It works from both task
 * and RZ point of view:\n
 * Task : Are there enough resources for me in the RZ ?\n
 * RZ : It is useful to add the given column to the RZ to satisfy the task ?\n
 * Hence, it handles both Slice and CLBx resources.
 * @param rz The reconfigurable zone to tailor
 * @param rbtype the type of resources to consider
 * @return true if the RZ constrains enough resources to host the task
 */
bool ResourceRequirements::RZHasEnoughResources(RZ *rz, RBType rbtype) {

	// Handle both CLB and Slice (request may come from a task or a RZ)

	// From the task point of view
	if(rbtype == Slice) {
		return (physicalResources[Slice] <=
			(rz->getConstrainedResourcesCount()[SliceL] +
			rz->getConstrainedResourcesCount()[SliceM] -
			physicalResources[SliceM]));

	// From the RZ point of view
	} else if(rbtype == CLBm) {
		return ((physicalResources[SliceM] <= rz->getConstrainedResourcesCount()[SliceM]) &&
			(physicalResources[Slice] <=
			(rz->getConstrainedResourcesCount()[SliceL] +
			rz->getConstrainedResourcesCount()[SliceM] -
			physicalResources[SliceM])));
	} else if(rbtype == CLBl) {
		return (physicalResources[Slice] <=
			(rz->getConstrainedResourcesCount()[SliceL] +
			rz->getConstrainedResourcesCount()[SliceM] -
			physicalResources[SliceM]));
	} else {
		return (physicalResources[rbtype] <= rz->getConstrainedResourcesCount()[rbtype]);
	}

}

/*!
 * @brief This method checks if a resource provided by a RZ is
 * needed by the task. It takes care of particular cases
 * such as CLBs to Slice conversion.
 * Kind of like RZHasEnoughResources but with no
 * quantification of the task needs.
 * @param rbtype the type of resources to consider
 * @return true if the task requires some of rbtype resources
 */
bool ResourceRequirements::requiresResources(RBType rbtype) const {

	// Handle CLB to Slice particular case
	if(rbtype != CLBm && rbtype != CLBl) {
		map<RBType,int>::const_iterator it = physicalResources.find(rbtype);
		return (it != physicalResources.end() && it->second != 0);
	} else {
		// Check for Slice, SliceL and SliceM
		map<RBType,int>::const_iterator it1 = physicalResources.find(Slice);
		map<RBType,int>::const_iterator it2 = physicalResources.find(SliceM);

		return ((it1 != physicalResources.end() && it1->second != 0)
			|| (it2 != physicalResources.end() && it2->second != 0));
	}
}

/*!
 * @brief This method tests whether the task fits the RZ or not.
 * @param rz The reconfigurable zone to consider
 * @return true if the RZ fits the requirements
 */
bool ResourceRequirements::fitsRZ(RZ *rz) {

	// Iterate through task map
	map<RBType,int>::const_iterator it;
	for(it = physicalResources.begin(); it != physicalResources.end(); it++) {
		if(!RZHasEnoughResources(rz, it->first)) {
			return false;
		} else {
			// Do nothing
		}
	}

	return true;
}


/*!
 * @brief This function analyzes the RZ to be added to the set for
 * potential STATIC blocks that cannot be used by other reconfigurable zones.
 * @param rz The reconfigurable zone to consider
 */
void ResourceRequirements::findStaticResources(RZ &rz) const {

	//bool staticResource = false;
	int i = 0;
	int j = 0;
	bool fullyUnused = true;

	// Iterate through the RZ line after line
	for(i = 0; i < rz.getHeight(); i++) {
		for(j = 1; j < (int) rz.getResourcesArrayPtr()->size() - 1; j++) {
			if(rz.getResourcesArrayPtr()->at(j).at(i).second == UNUSED) {
				// Check if adjacent resources are USED (i.e. UNUSED resource stuck in the middle)
				if((rz.getResourcesArrayPtr()->at(j-1).at(i).second == USED || rz.getResourcesArrayPtr()->at(j-1).at(i).second == STATIC)
					&& rz.getResourcesArrayPtr()->at(j+1).at(i).second == USED) {

					// Check if column is not fully unused
					fullyUnused = true;
					for(int k = 0; k < rz.getHeight(); k++) {
						if(rz.getResourcesArrayPtr()->at(j).at(k).second == USED) {
							fullyUnused = false;
							break;
						} else {
							// Do nothing
						}
					}
					if(!fullyUnused) {
						rz.getResourcesArrayPtr()->at(j).at(i).second = STATIC;
						//staticResource = true;
					} else {
						// Do nothing
					}
				}
			}
		}
	}

	/*if(staticResource) {
		rz.info();
		rz.printRZ();
	}*/
}


/*!
 * @brief Computes the internal fragmentation (difference between the resource
 * provided by the RZ and the resource requirements).
 * @param rz The reconfigurable zone to consider
 * @return The internal fragmentation
 */
int ResourceRequirements::computeInternalFragmentation(RZ &rz) {

	map<RBType,int>::iterator it;
	int cost = 0;
	RBType rbtype = Slice;

	// Iterate through task map

	for(it = physicalResources.begin(); it != physicalResources.end(); it++) {
		rbtype = it->first;

		// Handle both CLB and Slice (request may come from a task or a RZ)
		// From the task point of view

		if(rbtype == Slice) {
			cost += (rz.getConstrainedResourcesCount()[SliceL] + rz.getConstrainedResourcesCount()[SliceM]
				- physicalResources[SliceM] - physicalResources[Slice]) * device->getResourceCost(rbtype);
		} else {
			cost += (rz.getConstrainedResourcesCount()[rbtype] - physicalResources[rbtype]) * device->getResourceCost(rbtype);
		}
	}

	// Clean tables (TODO)
	it = physicalResources.begin();
	while (it != physicalResources.end()) {
		if(it->second == 0) {
			physicalResources.erase(it++);
		} else {
			it++;
		}
	}

	return cost;
}

/*!
 * @brief Computes the cost of these resource requirements based on the
 * cost of each different type of resources (some are scarcer that the others)
 * @return The resource requirements cost
 */
int ResourceRequirements::getTaskCost(void) const {

	map<RBType,int>::const_iterator it;
	int cost = 0;

	// Iterate through task map
	for(it = physicalResources.begin(); it != physicalResources.end(); it++) {
		cost += it->second * device->getResourceCost(it->first);
	}

	return cost;
}

/*!
 * @brief Displays basic information about the requirements
 * on the standard output.
 */
void ResourceRequirements::info(void) const {

	map<RBType,int>::const_iterator it;

	cout << "Task requires:" << endl;
	for(it = physicalResources.begin(); it != physicalResources.end(); it++) {
		cout << "\t" << it->second << " " << device->getCellName(it->first) << " column(s)" << endl;
	}
}

/*!
 * @brief Definition of the == operator
 * @param res Resource requirements to be compared to the current ones
 * @return true if resource requirements are equals
 */
bool ResourceRequirements::operator==(const ResourceRequirements &res) const {
	map<RBType, int>::const_iterator it;
	map<RBType, int>::const_iterator it2;

	bool differentResources = false;
	for(it = this->physicalResources.begin(); it != this->physicalResources.end(); it++) {
		it2 = res.physicalResources.find(it->first);
		if(it->second != 0 && (it2 == res.physicalResources.end() || it->second != it2->second)) {
			differentResources = true;
		} else {
			// Do nothing
		}
	}

	for(it = res.physicalResources.begin(); it != res.physicalResources.end(); it++) {
		it2 = this->physicalResources.find(it->first);
		if(it->second != 0 && (it2 == this->physicalResources.end() || it->second != it2->second)) {
			differentResources = true;
		} else {
			// Do nothing
		}
	}

	return !differentResources;
}

bool ResourceRequirements::operator<(const ResourceRequirements &res) const {
	/*map<RBType, int>::const_iterator it;

	bool moreResources = false;
	for(it = this->physicalResources.begin(); it != this->physicalResources.end(); it++) {
		map<RBType, int>::const_iterator it2 = res.physicalResources.find(it->first);
		if(it2 == res.physicalResources.end() || it->second > it2->second) moreResources = true;
	}

	for(it = res.physicalResources.begin(); it != res.physicalResources.end(); it++) {
		map<RBType, int>::const_iterator it2 = this->physicalResources.find(it->first);
		if(it2 != this->physicalResources.end() && it->second < it2->second) moreResources = true;
	}

	return !moreResources;*/




	// DUMMY
	return (this->objectID < res.objectID);

}

/*!
 * @brief Sets the resource wastage inside the reconfigurable zone (related
 * to the internal fragmentation)
 * @param rz Pointer to the RZ to consider
 */
void ResourceRequirements::setRZResourceWastage(RZ *rz) {

	map<RBType, int> RZResources = rz->getConstrainedResourcesCount();
	map<RBType, int>::const_iterator it;
	double wastage = 0;

	// Compute wastage using resources cost for targeted device
	for(it = RZResources.begin(); it != RZResources.end(); it++) {
		bool slicesDone = false;

		if((it->first == SliceL || it->first == SliceM) && !slicesDone) {
			wastage += (RZResources[SliceL] + RZResources[SliceM] - physicalResources[Slice] - physicalResources[SliceM]) * device->getResourceCost(Slice);
		} else {
			wastage += (it->second - physicalResources[it->first]) * device->getResourceCost(it->first);
		}
	}

	rz->setResourceWastage(wastage);
}
