/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    hardwareTask.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-08-12
 * @section DESCRIPTION
 *			Representation of a synthesized task
 */

#include "hardwareTask.h"
#include <fstream>
#include <math.h>
#include <time.h>

string HwTask::getNetlistName(void) {
	return netlistName;
}

string HwTask::getDeviceID(void) {
	return resourceFile->getDeviceID();
}

DeviceClass HwTask::getDeviceClass(void) {
	return resourceFile->getDeviceClass();
}

Device * HwTask::getDevicePtr(void) {
	return resourceFile->getDevicePtr();
}

vector<RZ> HwTask::getRZSet(void) {
	return resourceRequirements.getRZSet();
}

vector<RZ> * HwTask::getRZSetPtr(void) {
	return resourceRequirements.getRZSetPtr();
}

ResourceRequirements * HwTask::getResourceRequirementsPtr(void) {
	return &resourceRequirements;
}

TaskResourceFile* HwTask::getTaskResourceFile(void) {
	return resourceFile;
}

/* Misc */
void HwTask::info(void) {
	cout << "HwTask " << name << " for device " << resourceFile->getDeviceID() << " requires:" << endl;
	map<RBType,int> physicalResources = resourceFile->getPhysicalResources();
	for(map<RBType,int>::iterator it = physicalResources.begin(); it != physicalResources.end(); it++) {
		cout << "\t" << (*it).second << " " << resourceFile->getDevicePtr()->getCellName(it->first) << " column(s)" << endl;
	}
}

void HwTask::findRZs(void) {
	defineResourceRequirements();
	RZFinderManager::retrieveRZSet(resourceRequirements);
}

void HwTask::defineResourceRequirements(void) {
	resourceRequirements = ResourceRequirements(resourceFile->getPhysicalResources(), resourceFile->getDevicePtr());
}

bool HwTask::isInterfaceDefined(string interfaceType) {
	for(int i = 0; i < (int) possibleInterfaceLocations.size(); i++) {
		if(possibleInterfaceLocations.at(i).getInterfaceExpressionVector().front().compare(interfaceType) == 0) return true;
	}

	return false;
}

bool HwTask::fitsRZ(RZ *rz) {

	// Check for LOC constraints and interfaces possible locations
	bool LOCConstraintsVerified = true;
	bool interfaceConstraintsVerified = true;

	// Every LOC constraints must be verified
	for(int i = 0; i < (int) requiredColumns.size(); i++) {
		if(!rz->constrainsColumn(requiredColumns.at(i))) LOCConstraintsVerified = false;
	}

	// Every task interface must be available on the RZ
	// If no interface location provided, do not consider interfaces
	/*if(possibleInterfaceLocations.empty()) interfaceConstraintsVerified = true;
	else {*/
		for(multiset<TaskInterface>::const_iterator it = taskInterfaces.begin(); it != taskInterfaces.end(); it++) {

			// Consider interfaces only if this type has been defined as a physical interface
			if(isInterfaceDefined(it->getName())) {
				int nbInterfacesIncluded = 0;
				for(int i = 0; i < (int) possibleInterfaceLocations.size(); i++) {
					// Retrieve interface
					vector<string> interfaceExpression = possibleInterfaceLocations.at(i).getInterfaceExpressionVector();

					// Handle up to three components in the vector (one operator)
					// DUMMY: only use first component
					string firstInterface = interfaceExpression.front();
					if((firstInterface.compare(it->getName()) == 0 || firstInterface.compare(NO_INTERFACE_DEFINED) == 0)
						&& rz->isResourcePositionOK(possibleInterfaceLocations.at(i))) nbInterfacesIncluded++;
				}

				if(nbInterfacesIncluded < (int) taskInterfaces.count(*it)) interfaceConstraintsVerified = false;
			}
		}
	//}

	if(LOCConstraintsVerified && interfaceConstraintsVerified) return resourceRequirements.fitsRZ(rz);
	else return false;
}

int HwTask::computeInternalFragmentation(RZ &rz) {
	return resourceRequirements.computeInternalFragmentation(rz);
}

int HwTask::getTaskCost(void) {
	return resourceRequirements.getTaskCost();
}

ResourceRequirements& HwTask::getResourceRequirements(void) {
	return resourceRequirements;
}

int HwTask::readResourceFile(void) {

	return resourceFile->readResourceFile();
}

bool HwTask::hasPlacementRequirements(void) {
	return !requiredColumns.empty();
}

void HwTask::setResourceMargin(double v) {
	resourceFile->setResourcesMargin(v);
}

void HwTask::setLOCConstraint(vector<Coordinates> &vec) {
	requiredColumns = vec;
}

void HwTask::setPossibleInterfaceLocations(vector<InterfaceLocation> &vec) {
	possibleInterfaceLocations = vec;
}

void HwTask::HwTaskInit(void) {

	// Search for the .syr extension in filename
	if(resourceFileName.find(".syr") != string::npos) {
		// Xilinx synthesis report
		resourceFile = new SynthesisReport(resourceFileName);
	} else if(resourceFileName.find(".tsk") != string::npos) {
		// Generic XML file
		resourceFile = new XMLTaskDescription(resourceFileName);
	} else {
		//cerr << "ERROR: unrecognized extension while opening HW task description!" << endl;
		resourceFile = 0;
	}
}

void HwTask::HwTaskInit(TaskResourceFile* res) {

	// Search for the .syr extension in filename
	if(resourceFileName.find(".syr") != string::npos) {
		// Xilinx synthesis report
		resourceFile = new SynthesisReport(*static_cast<SynthesisReport*>(res));
	} else if(resourceFileName.find(".tsk") != string::npos) {
		// Generic XML file
		resourceFile = new XMLTaskDescription(*static_cast<XMLTaskDescription*>(res));
	} else {
		//cerr << "ERROR: unrecognized extension while opening HW task description!" << endl;
		resourceFile = 0;
	}
}

double HwTask::getContextSwitchTime(void) {
	return contextSwitchTime;
}
