/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    XMLTaskDescription.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2013-01-10
 * @section DESCRIPTION
 *			XML representation of task resource needs
 */

#include "XMLTaskDescription.h"

#include <xercesc/sax/HandlerBase.hpp>
#include <xercesc/framework/LocalFileFormatTarget.hpp>

#include <string>
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdexcept>
#include <list>
#include <cmath>

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include "utils.h"

using namespace std;

RBType convertStringToRBType(Device *device, string str) {
	return device->getRBType(str);
}


int XMLTaskDescription::readResourceFile(void) {

	// Try opening the file read-only to check existence
	ifstream file(filename.c_str());
	if(!file) {
		cerr << "ERROR: XML file does not exist!" << endl;
		return -1;
	}

	cout << endl << "-----------------------------------------------------------" << endl;
	cout << "Parsing file " << filename << endl;
	if(resourceMargin == -1) cout << "Resource margin is set to " << STATIC_RESOURCES_MARGIN << "%" << endl;
	else cout << "Resource margin is set to " << resourceMargin << "%" << endl;

	// Configure DOM parser
	m_ConfigFileParser->setValidationScheme(XercesDOMParser::Val_Never);
	m_ConfigFileParser->setDoNamespaces(false);
	m_ConfigFileParser->setDoSchema(false);
	m_ConfigFileParser->setLoadExternalDTD(false);

	// Create a device
	device = new Device();

	try {
		m_ConfigFileParser->parse(filename.c_str());

		// no need to free this pointer - owned by the parent parser object
		DOMDocument* xmlDoc = m_ConfigFileParser->getDocument();

		// Get the top-level element: Name is "root". No attributes for "root"
		DOMElement* elementRoot = xmlDoc->getDocumentElement();
		if( !elementRoot ) throw(std::runtime_error( "empty XML document" ));

		DOMNodeList*      children = elementRoot->getChildNodes();
		const  XMLSize_t nodeCount = children->getLength();

		for( XMLSize_t xx = 0; xx < nodeCount; ++xx ) {
			DOMNode* currentNode = children->item(xx);
			if( currentNode->getNodeType() &&  // true is not NULL
				currentNode->getNodeType() == DOMNode::ELEMENT_NODE ) // is element 
			{
				// Found node which is an Element. Re-cast node as element
				DOMElement* currentElement = dynamic_cast< xercesc::DOMElement* >( currentNode );

				if(XMLString::equals(currentElement->getTagName(), TAG_Cell)) {

					const XMLCh* xmlch_cellname = currentElement->getAttribute(ATTR_name);
					char *cellname_cstr = XMLString::transcode(xmlch_cellname);

					const XMLCh* xmlch_cellvalue = currentElement->getAttribute(ATTR_value);
					char *cellvalue_cstr = XMLString::transcode(xmlch_cellvalue);
					double value = Utils::atod(cellvalue_cstr);

					// Consider resource margin only for sensitive RBType
					if(device->getCell(device->getRBType(cellname_cstr)).isSentitiveToResourceMargin()) {
						if(resourceMargin == -1) value += (value * STATIC_RESOURCES_MARGIN / 100.0);
						else value += (value * resourceMargin / 100.0);
					}

					physicalResources.insert(pair<RBType,int>(device->getRBType(cellname_cstr), (int) ceil(value / device->getCell(device->getRBType(cellname_cstr)).getNbCellsPerColumn())));
					staticResources.insert(pair<RBType,int>(device->getRBType(cellname_cstr), (int) ceil(value)));

					XMLString::release(&cellname_cstr);
					XMLString::release(&cellvalue_cstr);
				} else if(XMLString::equals(currentElement->getTagName(), TAG_Device_name)) {
					const XMLCh* xmlch_devicename = currentElement->getAttribute(ATTR_value);
					char *devicename_cstr = XMLString::transcode(xmlch_devicename);

					setDeviceID(devicename_cstr);

					XMLString::release(&devicename_cstr);
				}
			}
		}

		// Convert maps
		convertAllResourceMaps();
	} catch( xercesc::XMLException& e ) {
		char* message = xercesc::XMLString::transcode( e.getMessage() );
		ostringstream errBuf;
		errBuf << "Error parsing file: " << message << flush;
		XMLString::release( &message );
		return -1;
	}

	return 0;
}
