/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    testbench_in.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-07-04
 * @section DESCRIPTION
 *			Template specialization for Testbench class when there
 *			are no outputs.
 */

#ifndef TESTBENCH_IN_H
#define TESTBENCH_IN_H

// Needed for the simple_target_socket
#define SC_INCLUDE_DYNAMIC_PROCESSES

//#include <systemc.h>
//#include "tlm.h"
//#include "tlm_utils/simple_initiator_socket.h"
//#include "tlm_utils/simple_target_socket.h"
//#include "tlm_utils/multi_passthrough_initiator_socket.h"
//#include "tlm_utils/multi_passthrough_target_socket.h"
//#include "tlm_utils/peq_with_cb_and_phase.h"
#include "utils.h"
#include "memory_manager.h"
//#include "trace.h"
//#include "simulation_controller.h"
#include "user_parameters.h"
#include "testbench_in_interface.h"
#include "testbench_algorithms.h"
#include <map>
#include <set>
#include <cstring>
#include <new>

//using namespace sc_core;
//using namespace sc_dt;
using namespace std;
//using namespace tlm;
//using namespace tlm_utils;

#define FOUT if(generate_logfile()) (*fout)

DECLARE_EXTENDED_PHASE(internal_ph_tb);

template<int Ni>
class TestbenchIn : /*public sc_module,*/ public Testbench_in_interface<Ni> {

public:

	// I/O sockets
	multi_passthrough_target_socket<TestbenchIn> input_target_socket;
	
private:

	// Payload event queues
	peq_with_cb_and_phase<TestbenchIn> m_peq_targ;

	// Maps to retrieve socket ID from a transaction in PEQ callbacks
	map<rs_payload*, unsigned int> m_id_targ_map;
	
	// Events
	rs_event response_received_event[Ni];
	rs_event transaction_received;
	
	// Transactions
	rs_payload* incoming_transaction[Ni];
	rs_payload* incoming_transaction_pending[Ni];

	// Memory manager
	Memory_manager m_mm;

	// Data
	int *data_in[Ni];
	uint64_t address_in[Ni];
	int nb_data_received[Ni];

	// Packet sizes
	int input_packet_size[Ni];

	// Timing parameters
	rs_time end_req_time[Ni];
	rs_time begin_resp_time[Ni];
	rs_time computation_time;

	// Trace methods events
	rs_event set_target_phase_end_req_event[Ni];
	rs_event set_target_phase_begin_resp_event[Ni];

	// Log file stream
	ostream* fout;

	// Trace file
	//sc_trace_file *tf;
	//trace_packet_testbench_t<Ni, 0> trace_packet;

	// Misc
	string name;
	int nb_responses_received;		/*< Number of responses received so far */
	int current_transaction_id;		/*< Transaction currently processed */

	vector<string> preceding_modules_connection_name;


	// Priority channels / Communications
	set<int> priority_channels_in;
	int nb_non_priority_channels_requests_in;
	rs_event decrement_nb_non_priority_channels_requests_in_event;	/**< Event to decrement number of shared channels requests */
	rs_event shared_channel_in_ready_event;							/**< Event to notify that a shared channel is available */
	rs_event send_end_req_event[Ni];								/**< Events to send a END_REQ */

	string applicationName;

	void (*testbench_thread_ptr)(Testbench_in_interface<Ni> &);
	static void testbench_thread_ptr_helper(void* context){
		((TestbenchIn<Ni>*)context)->testbench_thread_ptr(*((TestbenchIn<Ni>*)context));
	}

	vector<int> sockets_with_new_transactions;

public:

	// Constructor
	//SC_HAS_PROCESS(TestbenchIn);
	TestbenchIn(string instname, void (*tb_thread)(Testbench_in_interface<Ni> &), ostream& logfile/*, sc_trace_file *mtf*/) :
			name(instname),
			input_target_socket(/*"input_socket"*/),
			m_peq_targ(this, &TestbenchIn::peq_targ_callback),
			fout(&logfile),
			testbench_thread_ptr(tb_thread)/*,
			tf(mtf)*/ {

		for(int i = 0; i < Ni; i++) {
			incoming_transaction[i]= 0;
			incoming_transaction_pending[i]= 0;
			nb_data_received[i] = 0;
		}

		nb_responses_received = 0;
		current_transaction_id = 0;
		nb_non_priority_channels_requests_in = 0;
		
		// Data in initialization
		for(int i = 0; i < Ni; i++) {
			input_packet_size[i] = 16;
			data_in[i] = new int[input_packet_size[i]];
			for(int j = 0; j < input_packet_size[i]; j++) data_in[i][j] = 0;
		}
	
		// Init timing parameters
		for(int i = 0; i < Ni; i++) end_req_time[Ni] = rs_time(50);
		for(int i = 0; i < Ni; i++) begin_resp_time[Ni] = rs_time(50);
		computation_time = rs_time(500);

		// Register transport methods
		input_target_socket.register_nb_transport_fw(this, &TestbenchIn::nb_transport_fw);
		
		// Spawn testbench thread as a function outside the class
		//sc_spawn(sc_bind(testbench_thread_ptr, sc_ref(*this)), "testbench_in_thread");
		thread(testbench_thread_ptr_helper, this).detach();

		//SC_METHOD(decrement_nb_non_priority_channels_requests_in_method);
		//sensitive << decrement_nb_non_priority_channels_requests_in_event;
		//dont_initialize();

		// Dynamic threads initialization for tracing purpose: set_target_phase_end_req
		//for(int i = 0; i < Ni; i++) {
			//sc_spawn_options method_options;
			//method_options.spawn_method();
			//method_options.dont_initialize();
			//method_options.set_sensitivity(&set_target_phase_end_req_event[i]);
			//string process_name("set_target_phase_end_req_process_");
			//ostringstream out;
			//out << i;
			//process_name.append(out.str());

			//sc_spawn(sc_bind(&TestbenchIn<Ni>::set_target_phase_end_req, this, i), process_name.c_str(), &method_options);
		//}
		thread(decrement_nb_non_priority_channels_requests_in_method_helper, this).detach();

		// Dynamic threads initialization for tracing purpose: set_target_phase_begin_resp
//		for(int i = 0; i < Ni; i++) {
//			sc_spawn_options method_options;
//			method_options.spawn_method();
//			method_options.dont_initialize();
//			method_options.set_sensitivity(&set_target_phase_begin_resp_event[i]);
//			string process_name("set_target_phase_begin_resp_process_");
//			ostringstream out;
//			out << i;
//			process_name.append(out.str());
//
//			sc_spawn(sc_bind(&TestbenchIn<Ni>::set_target_phase_begin_resp, this, i), process_name.c_str(), &method_options);
//		}

		// Send end req process
		for(int i = 0; i < Ni; i++) {
			//sc_spawn_options method_options;
			//method_options.spawn_method();
			//method_options.dont_initialize();
			//method_options.set_sensitivity(&send_end_req_event[i]);
			//string process_name("set_end_req_process_process_");
			//ostringstream out;
			//out << i;
			//process_name.append(out.str());

			//sc_spawn(sc_bind(&TestbenchIn<Ni>::send_end_req_method, this, i), process_name.c_str(), &method_options);
			thread(send_end_req_method_helper,this,i).detach();
		}

		// Add signals to trace
		//init_packet();
		//sc_trace(tf, trace_packet, (string) name());
	}

	~TestbenchIn() {
		for(int i = 0; i < Ni; i++) {
			delete[] data_in[i];
		}
	}

	// Payload event queue callback function
	void peq_targ_callback(rs_payload *trans, const rs_phase& phase);
	
	// Transport methods
	virtual rs_sync_enum nb_transport_fw(int id, rs_payload *trans, rs_phase& phase);

	// Helper functions
	void send_response(int id, rs_payload *trans);
	rs_sync_enum send_end_req(int socketID, rs_payload *trans);

	void decrement_nb_non_priority_channels_requests_in_method(void);
	static void decrement_nb_non_priority_channels_requests_in_method_helper(void * ctx){
		((TestbenchIn<Ni>*)ctx)->decrement_nb_non_priority_channels_requests_in_method();
	}

	// Simulation controller interface
	void notify_simulation_controller(bool);

	// Testbench interface methods
	string getName(void) const;

	// User interface
	void verify_bindings(void);
	void set_packet_size(string connection_name, int val);

	void add_preceding_module_connection_name(string name);
	void add_timing_info(rs_phase_enum phase, string connection_name, rs_time time);
	void set_end_req_time(string connection_name, rs_time t);
	void set_begin_resp_time(string connection_name, rs_time t);

	void info(void);

	// Trace packet
	//void init_packet(void);
	//void update_target_packet(int id, rs_phase phase);
	//void update_target_packet(int id);
	void set_target_phase_end_req(int id);
	//void set_target_phase_begin_resp(int id);

	// Priority channels
	void add_priority_channel(string connection_name);
	bool is_priority_channel_in(int id);
	void *send_end_req_method(int id);
	static void send_end_req_method_helper(void * ctx, int id){
		((TestbenchIn<Ni>*)ctx)->send_end_req_method(id);
	}

	void set_application_name(string n);

	int get_input_socket_id(string connection_name) const;


	// TB In Interface
	ostream& get_logfile(void);
	string TB_IF_name(void) const;
	string TB_IF_get_instance_name(void) const;
	string TB_IF_get_instance_top_application_name(void) const;
	const rs_event& TB_IF_transaction_received_event(void) const;
	void TB_IF_wait_transaction_received_event(void);
	void TB_IF_wait_transaction_received_event(rs_time);
	const int TB_IF_get_nb_transactions_received_per_socket(int socketID) const;
	const int TB_IF_get_nb_transactions_received_per_socket(string channelName) const;
	const int TB_IF_get_nb_transactions_received(void) const;
	int* TB_IF_get_data_in_ptr(int) const;
	int* TB_IF_get_data_in_ptr(string) const;
	void TB_IF_notify_simulation_controller(bool);
	int TB_IF_get_packet_size(int ID) const;
	int TB_IF_get_packet_size(string name) const;
	int TB_IF_get_input_socket_id(string name) const;
	string TB_IF_get_connection_name(int id) const;
	vector<int> TB_IF_get_sockets_with_new_transactions(void);
};


/***********************************
 ******      Target part      ******
 ***********************************/

// Forward path for the target socket
template<int Ni>
rs_sync_enum TestbenchIn<Ni>::nb_transport_fw(int id, rs_payload *trans, rs_phase& phase) {

    uint64_t         adr = trans->get_address();
    unsigned int     len = trans->get_data_length();
    //unsigned char*   byt = trans->get_byte_enable_ptr();
    //unsigned int     wid = trans->get_streaming_width();

    // Obliged to check the transaction attributes for unsupported features
    // and to generate the appropriate error response
	//if (byt != 0) {
		//trans->set_response_status(TLM_BYTE_ENABLE_ERROR_RESPONSE);
	//	return RS_COMPLETED;
	//}
	FOUT << "receve data from last modul" << endl;
	//if (len > (4 * (unsigned int) input_packet_size[id]) /*|| wid < len*/) {
		//trans->set_response_status(TLM_BURST_ERROR_RESPONSE);
	//	FOUT << "recive error " << endl;
	//	return RS_COMPLETED;
	//}

	// Now queue the transaction until the annotated time has elapsed
	m_id_targ_map[trans] = id;
	m_peq_targ.notify(trans, phase);

	return RS_ACCEPTED;
}

// Payload event queue callback (Target side)
template<int Ni>
void TestbenchIn<Ni>::peq_targ_callback(rs_payload *trans, const rs_phase& phase) {
	
	int socketID = m_id_targ_map[trans];

	int nb_total_data_received = 0;

		if(phase == BEGIN_REQ)
			FOUT << rs_time_stamp() << ": " << name<< " (T" << socketID << ") received BEGIN_REQ at address " << hex << trans->get_address() << endl;
		else if(phase == END_RESP)
			FOUT << rs_time_stamp() << ": " << name << " (T" << socketID << ") received END_RESP at address " << hex << trans->get_address() << endl;
		else if(phase == internal_ph_tb)
			FOUT << rs_time_stamp() << ": " << name << " (T" << socketID << ") received int_phase at address " << hex << trans->get_address() << endl;

	switch (phase) {
	case BEGIN_REQ:

		// Increment the transaction reference count
		//trans->acquire();
		
		send_end_req_event[socketID].notify();

		incoming_transaction[socketID] = trans;

		//nb_data_received[socketID]++;
		//status = send_end_req(socketID, trans);

		//update_target_packet(socketID, phase);

		break;

	case END_RESP:

		//fout << "TB received response for transaction " << trans->get_address() << endl;

		//incoming_transaction[socketID] = 0;

		sockets_with_new_transactions.push_back(socketID);
		transaction_received.notify();

		//update_target_packet(socketID, phase);

		//trans->release();

		break;

	case END_REQ:
	case BEGIN_RESP:
		//SC_REPORT_FATAL("TLM-2", "Illegal transaction phase received by target");
		break;

	default:

		if (phase == internal_ph_tb) {

			// Set response status (mandatory)
			//trans->set_response_status(TLM_OK_RESPONSE);
			send_response(socketID, trans);
			//update_target_packet(socketID);
			
			break;
		}
	}

	
}

template<int Ni>
void TestbenchIn<Ni>::decrement_nb_non_priority_channels_requests_in_method(void) {
	while(1) {
		decrement_nb_non_priority_channels_requests_in_event.wait();
		nb_non_priority_channels_requests_in--;
		if(nb_non_priority_channels_requests_in == 0) shared_channel_in_ready_event.notify();
	}
}

template<int Ni>
void *TestbenchIn<Ni>::send_end_req_method(int id) {
	rs_event *my_trigger = &send_end_req_event[id];
	rs_event *current_trigger = my_trigger;
	while(1)
	{
		current_trigger->wait();


		if(is_priority_channel_in(id)) {
			// Store transaction address and data into local memory
			address_in[id] = incoming_transaction[id]->get_address();
			memcpy(data_in[id], reinterpret_cast<int *>(incoming_transaction[id]->get_data_ptr()), incoming_transaction[id]->get_data_length());

			//fout << name() << " socket " << socketID << " nb_data_received: " << nb_data_received[socketID] << endl;
			FOUT << rs_time_stamp() << ": " << name << " (T" << id << ") Updating address and data. New address: " << address_in[id] << " (Transaction " << dec << nb_data_received[id] << ")" << endl;

			FOUT << "Data: ";
			for(int i = 0; i < 16; i++) FOUT << hex << data_in[id][i] << dec << " ";

			FOUT << endl << endl;


			send_end_req(id, incoming_transaction[id]);

			//next_trigger();
			current_trigger = my_trigger;
		}
		else{
			if(nb_non_priority_channels_requests_in != 0) {
				FOUT << rs_time_stamp() << ": " << name << ": (I) request already in progress in shared channel, waiting for shared_channel_in_ready_event" << endl;
				//next_trigger(shared_channel_in_ready_event);
				current_trigger = &shared_channel_in_ready_event;
				//return ;

			}
			else {
				nb_non_priority_channels_requests_in++;
				// Store transaction address and data into local memory
				address_in[id] = incoming_transaction[id]->get_address();
				memcpy(data_in[id], reinterpret_cast<int *>(incoming_transaction[id]->get_data_ptr()), incoming_transaction[id]->get_data_length());

				//fout << name() << " socket " << socketID << " nb_data_received: " << nb_data_received[socketID] << endl;
				FOUT << rs_time_stamp() << ": " << name << " (T" << id << ") Updating address and data. New address: " << address_in[id] << " (Transaction " << dec << nb_data_received[id] << ")" << endl;

				FOUT << "Data: " << hex;
				for(int i = 0; i < 16; i++) FOUT << data_in[id][i] << " ";

				FOUT << endl << endl;


				send_end_req(id, incoming_transaction[id]);

				//next_trigger();
				current_trigger = my_trigger;

			}

		}
	}
}

template<int Ni>
rs_sync_enum TestbenchIn<Ni>::send_end_req(int socketID, rs_payload *trans) {

	rs_sync_enum status;
	rs_phase bw_phase;
	rs_phase int_phase = internal_ph_tb;
    rs_time delay;

	// Queue the acceptance and the response with the appropriate latency
	bw_phase = END_REQ;
	delay = end_req_time[socketID];
	status = input_target_socket[socketID]->nb_transport_bw(trans, bw_phase);
	if (status == RS_COMPLETED) {
		// Transaction aborted by the initiator
		// (RS_UPDATED cannot occur at this point in the base protocol, so need not be checked)
		trans->release();
		return status;
	}

	set_target_phase_end_req_event[socketID].notify();

	// Queue internal event to mark beginning of response
	delay = delay + computation_time; // Latency
	m_id_targ_map[trans] = socketID;
	m_peq_targ.notify(trans, int_phase);

	if(!is_priority_channel_in(socketID)) decrement_nb_non_priority_channels_requests_in_event.notify();

	nb_data_received[socketID]++;

	return status;
}

template<int Ni>
void TestbenchIn<Ni>::send_response(int id, rs_payload *trans) {

	rs_sync_enum status;
	rs_phase bw_phase;
	//rs_time delay;

	bw_phase = BEGIN_RESP;
	//delay = begin_resp_time[id];

	status = input_target_socket[id]->nb_transport_bw(trans, bw_phase);

	set_target_phase_begin_resp_event[id].notify();

	if (status == RS_UPDATED) {
		// The timing annotation must be honored
		m_id_targ_map[trans] = id;
		m_peq_targ.notify(trans, bw_phase);
	} else if (status == RS_COMPLETED) {
		// The initiator has terminated the transaction
		//trans->release();
	}
}


template<int Ni>
void TestbenchIn<Ni>::notify_simulation_controller(bool success) {
	Reconfiguration_manager::endApplication(applicationName, success);
}

/**
 * Testbench interface methods
 */
template<int Ni>
string TestbenchIn<Ni>::getName(void) const {
	return name();
}

/******************************
 ****    User interface    ****
 ******************************/
template<int Ni>
void TestbenchIn<Ni>::set_packet_size(string connection_name, int val) {

	int socketID = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}

	FOUT << "Preceding module" << endl;
	FOUT << "Socket ID: " << socketID << endl;

	assert(socketID < Ni);
	input_packet_size[socketID] = val;

	try {
		delete[] data_in[socketID];
		data_in[socketID] = new int[input_packet_size[socketID]];
		for(int i = 0; i < input_packet_size[socketID]; i++) data_in[socketID][i] = 0;
	} catch (bad_alloc& ba) {
		cerr << "ERROR: " << name << " bad_alloc caught: " << ba.what() << endl;
		exit(-1);
		//Simulation_controller::endSimulation();
	}
}

template<int Ni>
void TestbenchIn<Ni>::verify_bindings(void) {
	if(input_target_socket.size() != Ni) {
		cerr << "ERROR: Incorrect bindings for module " << name() << "! Exiting..." << endl;
		exit(-1);
	}
}

template<int Ni>
void TestbenchIn<Ni>::set_end_req_time(string connection_name, rs_time t) {
	add_timing_info(END_REQ, connection_name, t);
}

template<int Ni>
void TestbenchIn<Ni>::set_begin_resp_time(string connection_name, rs_time t) {
	add_timing_info(BEGIN_RESP, connection_name, t);
}

template<int Ni>
void TestbenchIn<Ni>::add_preceding_module_connection_name(string name) {
	preceding_modules_connection_name.push_back(name);
}

template<int Ni>
void TestbenchIn<Ni>::add_timing_info(rs_phase_enum phase, string connection_name, rs_time time) {
	
	int socketID = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}
	
	switch(phase) {
		case END_REQ: end_req_time[socketID] = time; break;
		case BEGIN_RESP: begin_resp_time[socketID] = time; break;
		default: cerr << "ERROR: Phase not recognized!" << endl;
	}
}

template<int Ni>
void TestbenchIn<Ni>::info(void) {

	FOUT << endl << "Testbench " << name << ":" << endl;

	for(int i = 0; i < Ni; i++) {
		FOUT << "Input socket " << i << " (Connection " << preceding_modules_connection_name.at(i) << "):" << endl;
		if(priority_channels_in.find(i) != priority_channels_in.end()) FOUT << "  Priority channel" << endl;
		FOUT << "  Packet size: " << input_packet_size[i] << endl;
		FOUT << "  END_REQ time: " << end_req_time[i] << endl;
		FOUT << "  BEGIN_RESP time: " << begin_resp_time[i] << endl;
	}
}

// Priority channels
template<int Ni>
void TestbenchIn<Ni>::add_priority_channel(string connection_name) {
	int id = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			id = i;
			break;
		}
	}

	if(id == -1) cerr << "ERROR: Module " << name() << " cannot find connection name through preceding or following modules vector!" << endl;

	assert(id < Ni);
	priority_channels_in.insert(id);
}

template<int Ni>
bool TestbenchIn<Ni>::is_priority_channel_in(int id) {
	return (priority_channels_in.find(id) != priority_channels_in.end());
}

template<int Ni>
void TestbenchIn<Ni>::set_application_name(string n) {
	applicationName = n;
}



template<int Ni>
int TestbenchIn<Ni>::get_input_socket_id(string connection_name) const {
	int socketID = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}

	if(socketID == -1) {
		cerr << "Error: Testbench unable to retrieve input socket ID via connection name" << endl;
		exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
	}

	return socketID;
}


/**********************************
 *     Testbench In Interface     *
 **********************************/
template<int Ni>
ostream& TestbenchIn<Ni>::get_logfile(void){
	return (*fout);
}

template<int Ni>
string TestbenchIn<Ni>::TB_IF_name(void) const {
	return name;
}

template<int Ni>
string TestbenchIn<Ni>::TB_IF_get_instance_name(void) const {
	string fullname(name);
	if(fullname.find("/") != string::npos) {
		return (fullname.substr(fullname.find("/") + 1)).c_str();
	} else return name;
}

template<int Ni>
string TestbenchIn<Ni>::TB_IF_get_instance_top_application_name(void) const {
    if (applicationName.find("/") != string::npos)
        return (applicationName.substr(applicationName.find("/") + 1)).c_str();
    else
        return applicationName;
}

template<int Ni>
const rs_event& TestbenchIn<Ni>::TB_IF_transaction_received_event(void) const {
	return transaction_received;
}

template<int Ni>
void TestbenchIn<Ni>::TB_IF_wait_transaction_received_event(void) {
	transaction_received.wait();
}

template<int Ni>
void TestbenchIn<Ni>::TB_IF_wait_transaction_received_event(rs_time time_out) {
	transaction_received.wait_for(time_out);
}

template<int Ni>
const int TestbenchIn<Ni>::TB_IF_get_nb_transactions_received_per_socket(int socketID) const {
	return nb_data_received[socketID];
}

template<int Ni>
const int TestbenchIn<Ni>::TB_IF_get_nb_transactions_received_per_socket(string channelName) const {
	return TB_IF_get_nb_transactions_received_per_socket(get_input_socket_id(channelName));
}

template<int Ni>
const int TestbenchIn<Ni>::TB_IF_get_nb_transactions_received(void) const {
	int nb_total_data_received = 0;
	for(int i = 0; i < Ni; i++) nb_total_data_received += nb_data_received[i];
	return nb_total_data_received;
}

template<int Ni>
int* TestbenchIn<Ni>::TB_IF_get_data_in_ptr(int socketID) const {
	return data_in[socketID];
}

template<int Ni>
int* TestbenchIn<Ni>::TB_IF_get_data_in_ptr(string connectionName) const {
	return data_in[get_input_socket_id(connectionName)];
}

template<int Ni>
void TestbenchIn<Ni>::TB_IF_notify_simulation_controller(bool val) {
	notify_simulation_controller(val);
}

template<int Ni>
int TestbenchIn<Ni>::TB_IF_get_packet_size(int ID) const {
	return input_packet_size[ID];
}

template<int Ni>
int TestbenchIn<Ni>::TB_IF_get_packet_size(string connectionName) const {
	return TB_IF_get_packet_size(get_input_socket_id(connectionName));
}

template<int Ni>
int TestbenchIn<Ni>::TB_IF_get_input_socket_id(string connectionName) const {
	return get_input_socket_id(connectionName);
}

template<int Ni>
string TestbenchIn<Ni>::TB_IF_get_connection_name(int id) const {
	return preceding_modules_connection_name.at(id);
}

template<int Ni>
vector<int> TestbenchIn<Ni>::TB_IF_get_sockets_with_new_transactions(void) {
	vector<int> vectorCopy(sockets_with_new_transactions);
	sockets_with_new_transactions.clear();
	return vectorCopy;
}


/**********************************
 *      Packet trace methods      *
 **********************************/


//template<int Ni>
//void TestbenchIn<Ni>::init_packet(void) {
//	//update_initiator_packet(0);
//	//update_target_packet(0);
//}
//
//
//template<int Ni>
//void TestbenchIn<Ni>::update_target_packet(int id, rs_phase phase) {
//	assert(id < Ni);
//	update_target_packet(id);
//	trace_packet.target_packet[id].phase = rs_phase_to_ascii(phase);
//}
//
//template<int Ni>
//void TestbenchIn<Ni>::update_target_packet(int id) {
//	assert(id < Ni);
//	trace_packet.target_packet[id].transaction.update_transaction(incoming_transaction[id]);
//	trace_packet.target_packet[id].transaction_pending.update_transaction(incoming_transaction_pending[id]);
//}
//
//template<int Ni>
//void TestbenchIn<Ni>::set_target_phase_end_req(int id) {
//	assert(id < Ni);
//	while(1)
//	{
//		set_target_phase_end_req_event[i].wait();
//		trace_packet.target_packet[id].phase = END_REQ_ASCII;
//	}
//}
//
//template<int Ni>
//void TestbenchIn<Ni>::set_target_phase_begin_resp(int id) {
//	assert(id < Ni);
//	trace_packet.target_packet[id].phase = BEGIN_RESP_ASCII;
//}

#endif
