/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    module_algorithms.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2010-07-01
 * @section DESCRIPTION
 *			All user algorithms
 */

#ifndef MODULE_ALGORITHMS_H
#define MODULE_ALGORITHMS_H

#include <cstring>
#include <chrono>
#include <ctime>
#include <thread>
#include "user_algorithm_interface.h"
#include "user_module_algorithms.h"
#include <sched.h>
#include <sys/types.h>
#include <sys/syscall.h>
#include <unistd.h>


#define M 32
#define N 32
#define P 32
//#include "user_module_algorithms.h"

//int do_somthig = 1;
//void waiting(std::chrono::microseconds wait_value ){
//	std::this_thread::sleep_for(wait_value);
//	do_somthig = 0;
//}

template<int Ni, int No> void user_algorithm_SW(User_algorithm_interface<Ni, No> &user_algo_interface) {
	user_algo_interface.set_pid();

	/*******************************************************************************/
	/* This is a Default Behavior                                                  */	

	while(true) {

		/*******************************************************************************/
		/****                    MANDATORY PART (EXCEPT DISPLAY)                    ****/


		/* WAIT FOR START ALGORITHM EVENT : Only for Dynamic tasks */
		
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "\033[31;4m"<< " Beginning of loop"<<"\033[39;24m" << endl;
		user_algo_interface.b_execution_requested();
		
		/* TRACE : The algorithm is idle in order to have an accurate trace */
		user_algo_interface.set_algorithm_idle();

		/* WAIT FOR NEXT PERIOD to begin to start algorithm execution. For non-periodic modules, function returns right away */
		user_algo_interface.wait_until_next_period();
#ifdef GENERATE_LOG_FILE
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << " New period started" << endl;
#endif
		user_algo_interface.set_algorithm_waiting();

		/**** BEGINNING OF DATA RECEPTION SEGMENT ****/

		user_algo_interface.b_all_data_received();

		user_algo_interface.set_algorithm_running();

		/**** END OF DATA RECEPTION SEGMENT ****/






		/**** BEGINNING OF USER ALGORITHM ****/
		clock_t start, end;

		start = clock();


		unsigned char A[M][N];
		unsigned char B[N][P];
		unsigned char C[M][P];

		//int data1[32768];

		int i ,j ,k;
		int sum = 0;
		for(i=0; i<M; i++)
			for(j=0; j<N; j++)
				A[i][j] = (i*N + j)%M;
		for(i=0; i<N; i++)
			for(j=0; j<P; j++)
				B[i][j] = ((i*P + j) + 7)%M;
		do
		{
			for (i = 0; i < M; i++)
				for (k = 0; k < N; k++) {
					sum = 0;
					for (j = 0; j < P; j++)
						sum += A[i][j] * B[j][k];
					C[i][k] = sum;
				}
			end = clock() - start;
		} while ( ((double(end)/CLOCKS_PER_SEC)*1000000) < user_algo_interface.get_WCET().to_double()); 

		//std::this_thread::sleep_for(std::chrono::microseconds(user_algo_interface.get_WCET().value()));



		/**** END OF USER ALGORITHM ****/


		// Check that all data issued from previous algorithm execution has been processed and sent
		user_algo_interface.b_all_data_sent();

		// When all data is sent, indicate that a new sequence starts now
		// MUST BE DONE ONLY AFTER VERIFYING THAT ALL PREVIOUS DATA HAVE BEEN SENT
		user_algo_interface.start_new_transaction_sequence();
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "start new transaction" << endl;

		// Find a channel that is not transient and has updated data
		int channel_id = 0;
		while(user_algo_interface.is_channel_transient(channel_id) || !user_algo_interface.nb_data_received(channel_id)) channel_id++;

		// Copy data
		int data_in_length = 0;
		int data_out_length = 0;
		for(int i = 0; i < No; i++) {
			data_in_length = user_algo_interface.get_data_in_length(channel_id);
			//user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "data copy 1" << endl;
			data_out_length = user_algo_interface.get_data_out_length(i);

			if(data_out_length <= data_in_length) {
				memcpy(user_algo_interface.get_data_out_ptr(i), user_algo_interface.get_data_in_ptr(channel_id), data_out_length);
			} else {
				memcpy(user_algo_interface.get_data_out_ptr(i), user_algo_interface.get_data_in_ptr(channel_id), data_in_length);
				for(int j = data_in_length; j < data_out_length; j+=4) user_algo_interface.get_data_out_ptr(i)[j/4] = 0;
			}
			
			//user_algo_interface.set_address_out(i, user_algo_interface.get_address_in(channel_id));
		}
		//user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "end data copy" << endl;

		// Release input sockets (we won't need their data anymore) for they are used by preceding modules again
		user_algo_interface.release_all_input_sockets();
		//user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "release input socket" << endl;

		// Notify RZ to continue processing
		user_algo_interface.end_of_algorithm();
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "end of algorithme" << endl;

		// Send data to all sockets
		user_algo_interface.nb_send_all_data();
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "send all data" << endl;


		 //Wait for the event notifying a possible change in the user algorithm thread
		user_algo_interface.wait_for_update_user_algorithm();

		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << " Update user algorithm event received" << endl;

		if(user_algo_interface.kill_user_algorithm()) {
#ifdef GENERATE_LOG_FILE
			user_algo_interface.get_logfile() << user_algo_interface.get_name() << " User algorithm thread has to be killed!" << endl;
#endif
			user_algo_interface.rez_pid();
			return;
		}
	}
}

template<int Ni, int No> void user_algorithm_HW(User_algorithm_interface<Ni, No> &user_algo_interface) {
	pid_t pid = (pid_t)syscall(SYS_gettid);
	user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "\033[31;4m"<< " " << hex << pid << dec << "\033[39;24m" << endl;
	user_algo_interface.set_pid();

	/*******************************************************************************/
	/* This is a Default Behavior                                                  */	

	while(true) {

		/*******************************************************************************/
		/****                    MANDATORY PART (EXCEPT DISPLAY)                    ****/


		/* WAIT FOR START ALGORITHM EVENT : Only for Dynamic tasks */
		
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "\033[31;4m"<< " Beginning of loop"<<"\033[39;24m" << endl;
		user_algo_interface.b_execution_requested();
		
		/* TRACE : The algorithm is idle in order to have an accurate trace */
		user_algo_interface.set_algorithm_idle();

		/* WAIT FOR NEXT PERIOD to begin to start algorithm execution. For non-periodic modules, function returns right away */
		user_algo_interface.wait_until_next_period();
#ifdef GENERATE_LOG_FILE
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << " New period started" << endl;
#endif
		user_algo_interface.set_algorithm_waiting();

		/**** BEGINNING OF DATA RECEPTION SEGMENT ****/

		user_algo_interface.b_all_data_received();

		user_algo_interface.set_algorithm_running();

		/**** END OF DATA RECEPTION SEGMENT ****/






		/**** BEGINNING OF USER ALGORITHM ****/

		///////////////////////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////////////////////
		//
		//
		// SEND DATA
		// START HARDWARE IP
		// WAIT END PROCESSING
		// READ DATA
		//
		//
		///////////////////////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////////////////////

		std::this_thread::sleep_for(std::chrono::microseconds(user_algo_interface.get_WCET().value())); // simulate time execution



		/**** END OF USER ALGORITHM ****/


		// Check that all data issued from previous algorithm execution has been processed and sent
		user_algo_interface.b_all_data_sent();

		// When all data is sent, indicate that a new sequence starts now
		// MUST BE DONE ONLY AFTER VERIFYING THAT ALL PREVIOUS DATA HAVE BEEN SENT
		user_algo_interface.start_new_transaction_sequence();
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "start new transaction" << endl;

		// Find a channel that is not transient and has updated data
		int channel_id = 0;
		while(user_algo_interface.is_channel_transient(channel_id) || !user_algo_interface.nb_data_received(channel_id)) channel_id++;

		// Copy data
		int data_in_length = 0;
		int data_out_length = 0;
		for(int i = 0; i < No; i++) {
			data_in_length = user_algo_interface.get_data_in_length(channel_id);
			//user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "data copy 1" << endl;
			data_out_length = user_algo_interface.get_data_out_length(i);

			if(data_out_length <= data_in_length) {
				memcpy(user_algo_interface.get_data_out_ptr(i), user_algo_interface.get_data_in_ptr(channel_id), data_out_length);
			} else {
				memcpy(user_algo_interface.get_data_out_ptr(i), user_algo_interface.get_data_in_ptr(channel_id), data_in_length);
				for(int j = data_in_length; j < data_out_length; j+=4) user_algo_interface.get_data_out_ptr(i)[j/4] = 0;
			}
			
			//user_algo_interface.set_address_out(i, user_algo_interface.get_address_in(channel_id));
		}
		//user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "end data copy" << endl;

		// Release input sockets (we won't need their data anymore) for they are used by preceding modules again
		user_algo_interface.release_all_input_sockets();
		//user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "release input socket" << endl;

		// Notify RZ to continue processing
		user_algo_interface.end_of_algorithm();
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "end of algorithme" << endl;

		// Send data to all sockets
		user_algo_interface.nb_send_all_data();
		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << "send all data" << endl;


		 //Wait for the event notifying a possible change in the user algorithm thread
		user_algo_interface.wait_for_update_user_algorithm();

		user_algo_interface.get_logfile() << rs_time_stamp() << ": " << user_algo_interface.get_name() << " Update user algorithm event received" << endl;

		if(user_algo_interface.kill_user_algorithm()) {
#ifdef GENERATE_LOG_FILE
			user_algo_interface.get_logfile() << user_algo_interface.get_name() << " User algorithm thread has to be killed!" << endl;
#endif
			user_algo_interface.rez_pid();
			return;
		}
	}
}


#endif
