/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    configuration_file_parser.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-07-16
 * @section DESCRIPTION
 *			Configuration file parsing methods
 */

#include "configuration_file_parser.h"
//#include "simulation_controller.h"
#include "task_implementation.h"
#include "utils.h"

#include <sstream>
#include <fstream>

map<string, int> buildResourceMap(ifstream& stream) {

	string line_read;

	map<string, int> resourceMap;
	bool endResources = false;
	do {
		getline(stream, line_read);
		if(line_read.find("end") != string::npos) endResources = true;
		else {
			string name = line_read.substr(0, line_read.find(": "));
			string number = line_read.substr(line_read.find(": ") + 2);	// Consider separator size
			istringstream iss_if(number);
			int resource_number;
			iss_if >> resource_number;

			resourceMap.insert(pair<string, int>(name, resource_number));
		}
	} while(!endResources);

	return resourceMap;
}

int readRZConfigurationFile(string filename, vector<RZ *> &RZset, vector<Task> &taskSet, FPGA *fpga) {

	const string STRING_DEVICE = "FPGA ";
	const string STRING_CLB = "CLB: ";
	const string STRING_BRAM = "BRAM: ";
	const string STRING_DSP = "DSP: ";
	const string STRING_RZ = "RZ ";
	const string STRING_PROCESSOR = "Processor ";
	const string STRING_TASK = "Task ";
	const string STRING_COMPRESSED = "CompressedTime = ";
	const string STRING_PRELOAD = "PreloadTime = ";
	const string STRING_SWITCH_TIME = "ContextSwitchTime = ";
	const string BINARY_LOADING_TIME = "BinaryLoadingTime = ";
	const string STRING_COMMENT = "//";
	const string STRING_INTERFACES = "Interfaces";
	const string STRING_CPU_ID = "CpuId = ";

	ifstream file(filename.c_str());

	if(file) {
		//Simulation_controller::get_logfile() << "Parsing RZ configuration file " << filename << endl;

		string line_read;

		while(getline(file, line_read)) {

			if(line_read.find(STRING_RZ) != string::npos) {

				string RZname(line_read.substr(STRING_RZ.size()));

				map<string, int> resourceMap = buildResourceMap(file);

				bool endFound = false;
				vector<TaskImplementation> RZtaskVector;
				do {
					getline(file, line_read);
					if(line_read.find("end") != string::npos) endFound = true;
					else if(line_read.find(STRING_TASK) != string::npos) {
						string name = line_read.substr(STRING_TASK.size());
						
						getline(file, line_read);
						int compressedTime = Utils::atoi(line_read.substr(STRING_COMPRESSED.size()));
						
						getline(file, line_read);
						int preloadTime = Utils::atoi(line_read.substr(STRING_PRELOAD.size()));

						getline(file, line_read);
						int contextSwitchTime = Utils::atoi(line_read.substr(STRING_SWITCH_TIME.size()));

						RZtaskVector.push_back(TaskImplementation(name, compressedTime/*, preloadTime, contextSwitchTime*/));
					}
				} while(!endFound);

				RZset.push_back(new RZ(RZname, RZtaskVector, HARD,0));
				RZset.back()->setResources(resourceMap);

			} else if(line_read.find(STRING_PROCESSOR) != string::npos) {
				
				string processorName(line_read.substr(STRING_PROCESSOR.size()));
				bool endFound = false;
				vector<TaskImplementation> RZtaskVector;

				getline(file, line_read);
				int cpuId = Utils::atoi(line_read.substr(STRING_CPU_ID.size()));

				getline(file, line_read);
				int switchTime = Utils::atoi(line_read.substr(STRING_SWITCH_TIME.size()));

				do {
					getline(file, line_read);
					if(line_read.find("end") != string::npos) endFound = true;
					else if(line_read.find(STRING_TASK) != string::npos) {

						string taskName(line_read.substr(STRING_TASK.size()));
						getline(file, line_read);
						int binaryLoadingTime = Utils::atoi(line_read.substr(BINARY_LOADING_TIME.size()));
						RZtaskVector.push_back(TaskImplementation(taskName, binaryLoadingTime/*, binaryLoadingTime, switchTime*/));
					}
				} while(!endFound);

				RZset.push_back(new RZ(processorName, RZtaskVector, SOFT, cpuId));

			} else if(line_read.find(STRING_DEVICE) != string::npos) {
				string device_name(line_read.substr(STRING_DEVICE.size()));
				map<string, int> resourceMap = buildResourceMap(file);
				*fpga = FPGA(device_name, resourceMap);
			} else if(line_read.find(STRING_TASK) != string::npos) {
				string task_name(line_read.substr(STRING_TASK.size()));
				map<string, int> resourceMap = buildResourceMap(file);
				Task task(task_name, resourceMap);
				taskSet.push_back(task);
			}
		}

		file.close();

	} else {
		//cerr << "ERROR: Could not open file: " << filename << endl;
		return -1;
	}

	return 0;
}
