/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    task_implementation.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-08-22
 * @section DESCRIPTION
 *			Represents the immplementation of a task coupled with its
 *			reconfiguration times.
 */

#ifndef TASK_IMPLEMENTATION_H
#define TASK_IMPLEMENTATION_H

//#define SC_INCLUDE_DYNAMIC_PROCESSES

#include "rs_time.h"

#include <iostream>
#include <string>
#include <map>
//#include <systemc.h>

using namespace std;

/** Enumeration for the RZ state */
const int NB_RZ_STATES = 5;
enum RZState {RZ_BLANK, RZ_RECONFIG, RZ_MAPPED, RZ_ACTIVE, RZ_RUNNING};	// Running must be declared last for result display convenience
const string RZState_string[NB_RZ_STATES] = {"Blank", "Reconfiguring", "Mapped", "Active", "Running"};

/** Enumeration for the task state */
enum Task_state {INACTIVE, QUEUED, CONFIG, MAPPED, RUNNING, PREEMPTED_MAPPED, PREEMPTED_INACTIVE, WAITING, IDLE, STATIC, CONTEXT_SAVE, CONTEXT_RESTORE};
const string Task_state_string[12] = {"INACTIVE", "QUEUED", "CONFIG", "MAPPED", "RUNNING", "PREEMPTED_MAPPED", "PREEMPTED_INACTIVE", "WAITING", "IDLE", "STATIC", "CONTEXT_SAVE", "CONTEXT_RESTORE"};

//const string Task_state_string[12] = {"INACTIVE", "QUEUED", "", "", "", "", "", "", "", "", "", ""};


class TaskImplementation {

private:
	string full_name;
	string implementation_name;
	string task_name;
	string netlist_name;
	string algorithm_name;

	rs_time compressedReconfigurationTime;
	//rs_time preloadReconfigurationTime;
	//rs_time contextSwitchTime;

	map<RZState, rs_time> occupationTimes;

	bool valid;

	// Mod impl stuff
	rs_time worst_case_execution_time;
	rs_time best_case_execution_time;
	double energy_consumption;	//  enlever

	// Hw Only
	double frequency;
	double p_idle;
	double p_run;

public:
	TaskImplementation(string n, int comp/*, int preload, int switchTime*/) {

		full_name = n;
		compressedReconfigurationTime = rs_time(comp);
		//preloadReconfigurationTime = rs_time(preload, SC_NS);
		//contextSwitchTime = rs_time(switchTime, SC_NS);
		netlist_name = "";
		algorithm_name = "";

		// Retrieve implementation and task name
		size_t position(full_name.find_last_of("."));
		implementation_name = full_name.substr(position + 1, full_name.size());
		task_name = full_name.substr(0, position);

		// Init table
		occupationTimes[RZ_RUNNING] = RS_ZERO_TIME;
		occupationTimes[RZ_ACTIVE] = RS_ZERO_TIME;

		valid = true;
	}

	TaskImplementation(void) {
		valid = false;
	};

	bool isValid(void);
	string getFullName(void);
	string getTaskName(void);
	string getImplementationName(void);
	string getNetlistName(void);
	string getAlgorithmName(void);
	bool hasSameNetlist(TaskImplementation &) const;
	rs_time getCompressedReconfigurationTime(void);
	rs_time getPreloadReconfigurationTime(void);
	rs_time getContextSwitchTime(void);
	void addOccupationTime(RZState state, rs_time t);
	rs_time getOccupationTime(RZState state);

	rs_time getWorstCaseExecutionTime(void) const;
	rs_time getBestCaseExecutionTime(void) const;
	double getFrequency(void) const;
	double getEnergyConsumption(void) const;
	double getPidle(void) const;
	double getPrun(void) const;

	void setWorstCaseExecutionTime(rs_time);
	void setBestCaseExecutionTime(rs_time);
	void setFrequency(double);
	void setEnergyConsumption(double);
	void setPidle(double);
	void setPrun(double);
	void setNetlistName(string);
	void setAlgorithmName(string);
	
};

#endif
