/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    scheduling.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-15
 * @section DESCRIPTION
 *			Some defines needed by the scheduler
 */

#include "scheduling.h"

/* AMAP - EDF */

DEFAULT_MAPPING_ALGORITHM(AMAP_EDF)
DEFAULT_SCHEDULING_ALGORITHM(AMAP_EDF)
WAITING_QUEUE_HANDLER(AMAP_EDF) {
	 CALL_WAITING_QUEUE_HANDLER(default_algo);
}


/* USER : Scheduling Algorithm : Update Waiting Queue */
void waiting_queue_handler_scheduling_default_algo(Scheduler_interface &scheduler) {

    // Example of EDF
	int queueSize = scheduler.Scheduler_get_waiting_queue_size();
	
	double max_deadline = 0;
	for (int pos=0; pos<queueSize; pos++) {
		SchedulerRequest schedulerReq = scheduler.Scheduler_get_element(pos);

		if (schedulerReq.has_current_implementation() == true) {
			// Use of Pidle, Prun, Frequency ... 
			ModuleImplementation modImpl = schedulerReq.get_current_implementation();
		}

		rs_time deadline = schedulerReq.get_deadline_time();
		double new_deadline = deadline / rs_time(10);
		if (new_deadline > max_deadline)
			max_deadline = new_deadline;
	}


/* 
    // Example of EDF
	int queueSize = scheduler.Scheduler_get_waiting_queue_size();
	
	double max_deadline = 0;
	for (int pos=0; pos<queueSize; pos++) {
		SchedulerRequest schedulerReq = scheduler.Scheduler_get_element(pos);

		// Use of Pidle, Prun, Frequency ... 
		ModuleImplementation modImpl = schedulerReq.get_current_implementation();

		rs_time deadline = schedulerReq.get_deadline_time();
		double new_deadline = deadline / rs_time(10, SC_US);
		if (new_deadline > max_deadline)
			max_deadline = new_deadline;
	}
	//cout << " _________"  << endl;
	
	for (int pos=0; pos<queueSize; pos++) {
		SchedulerRequest schedulerReq = scheduler.Scheduler_get_element(pos);
		rs_time deadline = schedulerReq.get_deadline_time();
		double new_priority = max_deadline - (deadline / rs_time(10, SC_US));
		//cout <<dec << "sort task " << schedulerReq.getName() << " old : " << schedulerReq.get_task_priority() << ", new : " << new_priority << endl;
		schedulerReq.set_task_priority((int)new_priority);
	}
	
	// Update priority queue(e.g. priority change)
	scheduler.Scheduler_update_queue();
*/

}

/*
  This function is called for the tasks on INACTIVE or PREEMPTED_INACTIVE state. 
  It search the RZ and the best implementation according to the task.
  Return Return_mapping_information(implementation_RZ,implementation_ID) :
      - RZ* implementation_RZ : RZ to perform the task (Proc or Reconf Zone)
	  - int implementation_ID : ID number of the best implementation (Hw or Sw)
*/

Return_mapping_information waiting_queue_handler_mapping_default_algo(Scheduler_interface &scheduler) {

	Return_mapping_information mapping;

	/********************************************************************************************/
	/* AMAP Algo : To satisfy AMAP mapping, search for an empty RZ, Use the most RZ as possible */

	/* Get request with the highest priority inside waiting queue */
	//SchedulerRequest current_request = scheduler.Scheduler_get_top_request();
	SchedulerRequest current_request = scheduler.Scheduler_current_task_waiting();

	/* Retrieve task ID associated with scheduler request */
	int waiting_task_id = scheduler.Scheduler_get_module_ID(current_request);
	pid_t waiting_task_pid = scheduler.Scheduler_get_module_tid(waiting_task_id);

	/* List of reconfigurable zones that fit waiting task id */
	vector<RZ *> compatible_RZs = scheduler.Scheduler_get_compatible_rz_vector(waiting_task_id);

	/* Search free RZs (Blank) and compatible with the waiting task id */
	RZ* implementation_RZ = 0;
	int i = 0;
	while(i < (int) compatible_RZs.size() && implementation_RZ == 0) {
		if(scheduler.Scheduler_is_RZ_blank(compatible_RZs.at(i))) implementation_RZ = compatible_RZs.at(i);
		i++;
	}

	if(implementation_RZ == 0) {
		/* RZ not found !
		* AMAP mapping cannot be satisfied,
		* searching for a MAPPED/ALGO_PREEMTPTED_MAPPED task to be replaced
		*/
		i = 0;
		while(i < (int) compatible_RZs.size() && implementation_RZ == 0) {
			if(scheduler.Scheduler_get_task_state(compatible_RZs.at(i)) == MAPPED || scheduler.Scheduler_get_task_state(compatible_RZs.at(i)) == PREEMPTED_MAPPED)
				implementation_RZ = compatible_RZs.at(i);
			i++;
		}
	}

	if(implementation_RZ != 0) {
		/* RZ found ! AMAP mapping can be satisfied, Task can be placed on RZ 'implementation_RZ'
		 * Now, Searching Implementation according to reconfiguration time
		 */

		/* DISPLAY : LOG File */
		#ifdef GENERATE_LOG_FILE
			// Task can be placed on RZ 'implementation_RZ'
			scheduler.Scheduler_get_output_stream() << rs_time_stamp() << ": " << scheduler.Scheduler_get_name() << " allocating " << current_request.getName() << " to " << implementation_RZ->getName() << endl;
		#endif

		/* Search between possible different implementations of the task on the PE 
		 *  and pick up the one with the shortest time of (reconfiguration overhead + WCET)
		 */
		vector<TaskImplementation> taskVector(implementation_RZ->getTaskImplementationVector());
		int bestImplementationID = -1;
		rs_time best_aggregate_time(RS_ZERO_TIME);
		for(int j = 0; j < (int) taskVector.size(); j++) {
			/* Find implementation for which reconfiguration_time+execution_time is the lowest */
			rs_time aggregate_time(taskVector.at(j).getCompressedReconfigurationTime() + taskVector.at(j).getWorstCaseExecutionTime());
			if(current_request.has_implementation(taskVector.at(j)) && (bestImplementationID == -1 || aggregate_time < best_aggregate_time)) {
				bestImplementationID = j;
				best_aggregate_time = aggregate_time;
			}
		}

		mapping = Return_mapping_information(implementation_RZ, bestImplementationID, waiting_task_pid);
	} 

	return mapping;
}


/**
 * This function handles scheduling. It looks into the waiting queue for
 * tasks to implement on the FPGA. Currently, the scheduling scheme combines
 * a As Much As Possible (AMAP) mapping with an Earliest-Deadline First (EDF)
 * algorithm. Note that the AMAP mapping may only use as much RZs as there are
 * tasks (i.e. a static implementation). If there are more RZs, then they remain
 * blank.
 */

//REM: Appel de fonction  partir de Reconfiguration_manager::scheduler_thread()

WAITING_QUEUE_HANDLER(default_algo) {

	/* USER : Scheduling Algorithm : Update Waiting Queue */
	CALL_SCHEDULING_ALGORITHM(default_algo);

	// SCHED : exit if no solution found at this moment (wait for the next call)
	bool exitCondition = false;
	Return_mapping_information mapping;

	scheduler.Scheduler_reset_current_position_in_waiting_queue();

	//while(scheduler.Scheduler_are_tasks_waiting() && !exitCondition) {
	cout << "curent position : " << scheduler.Scheduler_position_of_current_element() << endl ;
	cout << "queue size      : " << scheduler.Scheduler_get_waiting_queue_size() << endl ;
	while(scheduler.Scheduler_last_task_waiting() == false && !exitCondition) {

		//!\ NECESSARY SYNC
		//wait(RS_ZERO_TIME);

		/* DISPLAY : Console, Waiting Task Queue */
		scheduler.Scheduler_display_waiting_queue();

		/* SCHED : Get request with the highest priority inside waiting queue */
		//SchedulerRequest current_request = scheduler.Scheduler_get_top_request();
		SchedulerRequest current_request = scheduler.Scheduler_next_task_waiting();

		/* SCHED : Retrieve task ID associated with scheduler request */
		int waiting_task_id = scheduler.Scheduler_get_module_ID(current_request);

		/* TRACE : Set scheduler active for this task */
		scheduler.Scheduler_set_scheduler_active(waiting_task_id);

		switch(scheduler.Scheduler_get_task_state(waiting_task_id)) {

			case STATIC:
				// Should not happen here, should be considered directly in b_transport
				//SC_REPORT_FATAL("TLM-2.0", "Reconfiguration manager: Static state should not happen at this point!");
			
			case IDLE:				/* place sur le FPGA et en attente de la nouvelle priode */
			case WAITING:			/* place sur le FPGA et en attente des communications */
			case CONTEXT_SAVE:		/* ok */
			case CONTEXT_RESTORE:	/* ok */
			case QUEUED:			/* elle est en file d'attente pour etre configure (Not Reconfigurable Task) */
			case CONFIG:			/* En cours de configuration (Not Reconfigurable Task) */
			case RUNNING:			/* ACTIVE (instantie) (Not Reconfigurable Task) */

				/* DISPLAY : Console, Precedence list vector */
				scheduler.Scheduler_print_finished_precedence_vector(waiting_task_id);

				/* SCHED: Check if the owner of the request is the same as the one that initiated reconfiguration. 
				 * If so, wait till end of computation (new execution request)
				 */
				if(scheduler.Scheduler_has_task_already_requested_mapping(waiting_task_id, current_request.get_request_owner())) {
					/* DISPLAY : LOG File */
					#ifdef GENERATE_LOG_FILE
						scheduler.Scheduler_get_output_stream() << rs_time_stamp() << ": " << scheduler.Scheduler_get_name() << ": Packet n+1 -> Task is still in RUNNING state, return" << endl;
					#endif

					/* SCHED : Exit Scheduler */
					//REM: la tache refait une demande alors que le paquet d'avant n'est pas encore trait par la tache suivante
					exitCondition = true;

					/* SCHED : Simulate scheduler execution time and energy consumption */
					scheduler.Scheduler_emulate_scheduler_behavior();
				} 
				else {
					/*
					 * Vrifie que la requete vient d'une autre tache
					 * On ne peut pas etre RUNNING ici
					 */
					if(waiting_task_id != current_request.get_request_owner()) {
						/* DISPLAY : LOG File */
						#ifdef GENERATE_LOG_FILE
							scheduler.Scheduler_get_output_stream() << rs_time_stamp() << ": " << scheduler.Scheduler_get_name() << ": Packet n -> Task is already in RUNNING state, removing task from list" << endl;
						#endif

						/* SCHED : Remove highest priority request from the queue */
						//REM: enlve la requete de la file
						//scheduler.Scheduler_pop_request();
						scheduler.Scheduler_erase_current_task_waiting();

						/* SCHED : Simulate scheduler execution time and energy consumption */
						scheduler.Scheduler_emulate_scheduler_behavior();

						/* SCHED : notification (Notify precedence) */
						//REM: si tache running, on prvient le module prcdent (request owner) qu'il peut envoyer ces donnes
						if(scheduler.Scheduler_get_task_state(waiting_task_id) == IDLE ||
							scheduler.Scheduler_get_task_state(waiting_task_id) == WAITING ||
							scheduler.Scheduler_get_task_state(waiting_task_id) == CONTEXT_RESTORE) {

							/* SCHED : Direct Notification  */
							scheduler.Scheduler_notify_request_owner_module_ready(current_request.get_request_owner(), waiting_task_id);
						
						} 
						else {
							
							/* SCHED : Notification after CONFIG */
							//REM: ajoute le request owner  la liste des prdcesseurs qui sont prets  envoyer les donnes
							scheduler.Scheduler_add_finished_precedence(waiting_task_id, current_request.get_request_owner());
						}
					} 
					else {
						/* SCHED : Preemption only, should happen here 
						 * during transitory states CONTEXT_SAVE & CONTEXT_RESTORE --> Do nothing 
						 */						
						exitCondition = true;
					}
				}

				break;

			case PREEMPTED_MAPPED:	/* Atteint un pt de premption et en attente d'une dcision du scheduler */
			case MAPPED:			/* Not Reconfigurable Task */

				/* DISPLAY : LOG File */
				#ifdef GENERATE_LOG_FILE
					scheduler.Scheduler_get_output_stream() << rs_time_stamp() << ": " << scheduler.Scheduler_get_name() << ": Task is in MAPPED state, no reconfiguration needed" << endl;
				#endif

				/* SCHED : Remove highest priority request from the queue */
				//REM: la tche est dj sur la RZ donc, on peut directement la relancer
				//scheduler.Scheduler_pop_request();
				scheduler.Scheduler_erase_current_task_waiting();

				/* SCHED : Simulate scheduler execution time and energy consumption */
				scheduler.Scheduler_emulate_scheduler_behavior();

				/* TRACE : Changes a task from MAPPED to RUNNING state. */
				scheduler.Scheduler_rerun_task(current_request);

				break;

			case PREEMPTED_INACTIVE:	/* premt et enlever de la RZ/Proc */
			case INACTIVE:				/* fin d'excution ou au dpart */
				
				//REM: au dpart, toutes les taches sont PREEMPTED_INACTIVE ou INACTIVE.
				//REM: on prend les dcisions concernant le mapping

				/* DISPLAY : LOG File */
				#ifdef GENERATE_LOG_FILE
					scheduler.Scheduler_get_output_stream() << rs_time_stamp() << ": " << scheduler.Scheduler_get_name() << ": Task is in INACTIVE/PREEMPTED_INACTIVE state" << endl;
				#endif

				//
				// MAPPING DECISIONS TAKEN HERE
				//
				mapping = CALL_MAPPING_ALGORITHM(default_algo);

				if (mapping.RZ_found() == true) {

					if (mapping.hasCfgTcfPt() == true) {

						/* Update Fct Point */ 
						RZ* rz = mapping.get_implementation_RZ();
						
						if (scheduler.Scheduler_exist_rz_properties(rz) == true) {
							RZ_config rz_config = scheduler.Scheduler_get_rz_properties(rz);
							Config_param_group* cfgParamGrp = rz_config.getConfigDomain();
							if (cfgParamGrp != NULL) {
								if (cfgParamGrp->isCommonDomain() == true) {
									// CommonDomain true
									string groupName = rz->getGroupDomainName();
									Config_fct_point fctPt = mapping.getConfigFctPoint();
									if (fctPt.equals(rz->getConfigFctPoint()) == false) {
										// Fct Pt change
										vector<RZ *> rz_table = scheduler.Scheduler_get_all_rz_vector();
										// Update Fct Pt for all RZs (for same group)
										for (unsigned int i=0; i<rz_table.size(); i++) {
											RZ *c_rz = rz_table.at(i);
											if (groupName.compare(c_rz->getGroupDomainName()) == 0) {

												// Same Group
												c_rz->setConfigFctPoint(mapping.getConfigFctPoint());

												// Notify modification of parameters to the waiting module
												scheduler.Scheduler_send_update_parameters_to_module(rz);
											}
										}
									}
								}
								else {
									// CommonDomain false
									rz->setConfigFctPoint(mapping.getConfigFctPoint());

									// Notify modification of parameters to the waiting module
									scheduler.Scheduler_send_update_parameters_to_module(rz);
								}
							}
						}
					}

					/* SCHED : Remove highest priority request from the queue */
					//scheduler.Scheduler_pop_request();
					scheduler.Scheduler_erase_current_task_waiting();

					/* SCHED : Simulate scheduler behavior (execution time of the scheduler) BEFORE configuring task for timing accuracy */
					scheduler.Scheduler_emulate_scheduler_behavior();

					/* SCHED : Launch task configuration through either HW of SW reconfiguration unit. */  
					scheduler.Scheduler_configure_task(mapping.get_implementation_RZ(), waiting_task_id, mapping.get_implementation_TID() ,  mapping.get_implementation_ID(), current_request.get_request_owner());
				}
				else {

					/* RZ not found !
					* AMAP mapping cannot be satisfied,
					* No solutions at this time...
					*/

					/* DISPLAY : LOG File */
					#ifdef GENERATE_LOG_FILE
						scheduler.Scheduler_get_output_stream() << rs_time_stamp() << ": " << scheduler.Scheduler_get_name() << " cannot allocate " << current_request.getName() << "... Exiting" << endl;
					#endif

					if (mapping.exit_mapping_algorithm() == true)
						exitCondition = true;

					/* SCHED : Simulate scheduler execution time and energy consumption */
					scheduler.Scheduler_emulate_scheduler_behavior();
				}
				
				break;

			default:
				//SC_REPORT_FATAL("TLM-2", "Manager cannot recognize task state");
				cerr << "Manager cannot recognize tast state" << endl;
		}

		/* DISPLAY : Console */
		scheduler.Scheduler_display_task_state_table();
		scheduler.Scheduler_display_rz_current_module_table();
	}
}
