/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    module_implementation.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-06-01
 * @section DESCRIPTION
 *			Defines a the algorithm type of implementation, either
 *			hard or soft. There might be more than one hard/soft implementation
 *			for a single module
 */

#include "module_implementation.h"

string ModuleImplementation::get_name(void) {
	return name;
}

PEImplementation ModuleImplementation::get_implementation_type(void) {
	return implementation_type;
}

rs_time ModuleImplementation::get_worst_case_execution_time(void) const {
	return worst_case_execution_time;
}

rs_time ModuleImplementation::get_best_case_execution_time(void) const {
	return best_case_execution_time;
}

double ModuleImplementation::get_frequency(void) const {
	return frequency;
}

/*
double ModuleImplementation::get_energy_consumption(void) const {
	return energy_consumption;
}
*/

double ModuleImplementation::get_p_idle(void) const {
	return p_idle;
}

double ModuleImplementation::get_p_run(void) const {
	return p_run;
}

string ModuleImplementation::get_netlist_name(void) const {
	return netlist_name;
}

string ModuleImplementation::get_algorithm_name(void) const {
	return algorithm_name;
}

void ModuleImplementation::activate_implementation(void) {
	enable_implementation = true;
}

void ModuleImplementation::deactivate_implementation(void) {
	enable_implementation = false;
}

bool ModuleImplementation::is_implementation_enabled(void) const {
	return enable_implementation;
}

void ModuleImplementation::info(ostream& st) const {
	st << "Implementation " << name << ": ";
	st << ((implementation_type == HARD) ? "Hardware" : "Software");
	st << (enable_implementation ? ", ENABLED" : ", DISABLED") << endl;
	st << "BCET: " << best_case_execution_time << endl;
	st << "WCET: " << worst_case_execution_time << endl;
	st << "Frequency: " << frequency << " MHz" << endl;
	st << "Energy consumption: " << energy_consumption << "uJ" << endl;
	st << "Preemption points: " << nb_preemption_points << endl;
	st << endl;
}

void ModuleImplementation::add_physical_channels(string connection, int n) {
	for(int i = 0; i < n; i++) {
		physical_channels[connection]++;
	}
}

map<string, int> ModuleImplementation::get_physical_channel_map(void) {
	return physical_channels;
}

// Setters
void ModuleImplementation::set_worst_case_execution_time(rs_time t) {
	worst_case_execution_time = t;
}

void ModuleImplementation::set_best_case_execution_time(rs_time t) {
	best_case_execution_time = t;
}

void ModuleImplementation::set_frequency(double f) {
	frequency = f;
}

void ModuleImplementation::set_netlist_name(string n) {
	netlist_name = n;
}

void ModuleImplementation::set_algorithm_name(string n) {
	algorithm_name = n;
}

void ModuleImplementation::set_p_idle(double e) {
	p_idle = e;
}

void ModuleImplementation::set_p_run(double e) {
	p_run = e;
}

algorithm_thread_type ModuleImplementation::get_algorithm_thread_ptr(void) {
	return algorithm_thread_ptr;
}

void ModuleImplementation::set_nb_preemption_points(int n) {
	nb_preemption_points = n;
}

int ModuleImplementation::get_nb_preemption_points(void) {
	return nb_preemption_points;
}

void ModuleImplementation::set_use_context_switch_mode(bool val) {
	use_context_switch = val;
}

bool ModuleImplementation::use_context_switch_mode(void) {
	return use_context_switch;
}
