/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    testbench_out.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-07-02
 * @section DESCRIPTION
 *			Template specialization for Testbench class when there
 *			are no inputs.
 */

#ifndef TESTBENCH_OUT_H
#define TESTBENCH_OUT_H

// Needed for the simple_target_socket
#define SC_INCLUDE_DYNAMIC_PROCESSES

//#include <systemc.h>
//#include "tlm.h"
//#include "tlm_utils/simple_initiator_socket.h"
//#include "tlm_utils/simple_target_socket.h"
//#include "tlm_utils/multi_passthrough_initiator_socket.h"
//#include "tlm_utils/peq_with_cb_and_phase.h"
#include "rs_event.h"
#include "rs_time.h"
#include "rs_socket.h"
#include "utils.h"
#include "memory_manager.h"
//#include "trace.h"
//#include "simulation_controller.h"
#include "user_parameters.h"
#include "testbench_out_interface.h"
#include "testbench_algorithms.h"
#include <iostream>
#include <cassert>
#include <map>
#include <queue>
#include "algorithm_mode_command.h"

//using namespace sc_core;
//using namespace sc_dt;
using namespace std;
//using namespace tlm;
//using namespace tlm_utils;

#define FOUT if(generate_logfile()) (*fout)
//#define FOUT if(generate_logfile()) ((*fout) << hex << this_thread::get_id())<< dec << " | "

template<int No>
class TestbenchOut : /*public sc_module,*/ public Testbench_out_interface<No> {
	
public:

	// I/O sockets
	multi_passthrough_initiator_socket<TestbenchOut> output_initiator_socket;
	//simple_initiator_socket<TestbenchOut> output_initiator_socket;
	
	// Configuration socket
	simple_initiator_socket<TestbenchOut> manager_initiator_socket;
	simple_target_socket<TestbenchOut> manager_target_socket;
	
private:

	// Payload event queues
	peq_with_cb_and_phase<TestbenchOut> m_peq;

	// Maps to retrieve socket ID from a transaction in PEQ callbacks
	map<rs_payload*, unsigned int> m_id_init_map;
	
	// Events
	rs_event lock_tb_sinc;
	rs_event transaction_transmitted_event[No];
	rs_event configuration_done_event;
	rs_event all_responses_received_event;
	rs_event start_initiator_channel_event[No];
	rs_event end_of_response_channel_event[No];
	
	// Transactions
	rs_payload* request_in_progress[No];
	rs_payload* configuration_in_progress;

	// Memory manager
	Memory_manager m_mm;

	// Data
	int *data_out[No];
	uint64_t address_out[No];

	// Packet sizes
	int output_packet_size[No];		/* Linked to attribute */

	// Timing parameters
	rs_time begin_req_time[No];		/* Linked to attribute */
	rs_time end_resp_time[No];		/* Linked to attribute */

	// Trace methods events
	rs_event set_initiator_phase_begin_req_event[No];
	rs_event set_initiator_phase_end_resp_event[No];

	// Log file stream
	ostream* fout;

	// Trace file
	//sc_trace_file *tf;
	//trace_packet_testbench_t<0, No> trace_packet;

	// Connections lists (only following modules)
	vector<int> channel_connection_vector;	// w/ redundancies
	vector<int> following_modules_list;		// w/o redundancies

	vector<string> following_modules_connection_name;
	vector<string> preceding_modules_connection_name;

	// Misc
	string name;
	int nb_responses_received;		/*< Number of responses received so far */
	int current_transaction_id;		/*< Transaction currently processed */

	// Priority channels
	set<int> priority_channels_out;				/**< Set containing all priority channels */
	int nb_non_priority_channels_requests_out;	/**< Number of requests for a channel */
	rs_event shared_channel_out_ready_event;	/**< Event to notify that the physical shared channel is ready */

	string applicationName;

	void (*testbench_thread_ptr)(Testbench_out_interface<No>&); 
	static void testbench_thread_ptr_helper(void* context){
		((TestbenchOut<No>*)context)->testbench_thread_ptr(*((TestbenchOut<No>*)context));
	}

	rs_time period;

	map<int, vector<int> > channels_required;	/* <module_id, list of socket ID> */

	// Algorithms execution modes control
	//rs_event_queue algorithm_mode_command_event_queue;
	priority_queue<AlgorithmModeCommand, vector<AlgorithmModeCommand>, AlgorithmModeCommandComparison> algorithm_mode_command_priority_queue;

public:

	// Constructor
	//SC_HAS_PROCESS(TestbenchOut);
	TestbenchOut(string instname, void (*tb_thread)(Testbench_out_interface<No> &), ostream& logfile/*, sc_trace_file *mtf*/) :
			output_initiator_socket(/*"output_initiator_socket"*/),
			name(instname),
			manager_initiator_socket(/*"manager_initiator_socket"*/),
			manager_target_socket(/*"manager_target_socket"*/),
			m_peq(this, &TestbenchOut::peq_callback),
			fout(&logfile),
			configuration_in_progress(0),
			testbench_thread_ptr(tb_thread){

		for(int i = 0; i < No; i++) request_in_progress[i]= 0;

		nb_responses_received = 0;
		current_transaction_id = 0;
		nb_non_priority_channels_requests_out = 0;

		period = RS_ZERO_TIME;

		// Data out initialization
		for(int i = 0; i < No; i++) {
			output_packet_size[i] = 16;
			data_out[i] = new int[output_packet_size[i]];
			for(int j = 0; j < output_packet_size[i]; j++) data_out[i][j] = 0;
		}
		
		// Init timing parameters
		for(int i = 0; i < No; i++) begin_req_time[i] = rs_time(1);
		for(int i = 0; i < No; i++) end_resp_time[i] = rs_time(1);

		// Register transport methods
		output_initiator_socket.register_nb_transport_bw(this, &TestbenchOut<No>::nb_transport_bw);
		//output_initiator_socket.register_nb_transport_bw(this, &TestbenchOut::nb_transport_bw);
		manager_target_socket.register_b_transport(this, &TestbenchOut<No>::b_transport_manager);

		// Spawn testbench thread as a function outside the class
		//this_thread::sleep_for(chrono::seconds(1));
		thread(testbench_thread_ptr_helper, this).detach();

		//SC_METHOD(launch_algorithm_command);
			//sensitive << algorithm_mode_command_event_queue;
			//dont_initialize();

		// Dynamic threads initialization for channel communications
		for(int i = 0; i < No; i++) {
			//sc_spawn_options method_options;
			//method_options.spawn_method();
			//method_options.dont_initialize();
			//method_options.set_sensitivity(&start_initiator_channel_event[i]);
			//string process_name("Dynamic_process_");
			//ostringstream out;
			//out << i;
			//process_name.append(out.str());

			//sc_spawn(sc_bind(&TestbenchOut<No>::dynamic_transfer_process_method_helper, this, i), process_name.c_str(), &method_options);
			thread(dynamic_transfer_process_method_helper, this, i).detach();
		}

		// Dynamic threads initialization for tracing purpose: set_initiator_phase_begin_req
//		for(int i = 0; i < No; i++) {
//			sc_spawn_options method_options;
//			method_options.spawn_method();
//			method_options.dont_initialize();
//			method_options.set_sensitivity(&set_initiator_phase_begin_req_event[i]);
//			string process_name("set_initiator_phase_begin_req_process_");
//			ostringstream out;
//			out << i;
//			process_name.append(out.str());
//
//			sc_spawn(sc_bind(&TestbenchOut<No>::set_initiator_phase_begin_req, this, i), process_name.c_str(), &method_options);
//		}

		// Dynamic threads initialization for tracing purpose: set_initiator_phase_end_resp
//		for(int i = 0; i < No; i++) {
//			sc_spawn_options method_options;
//			method_options.spawn_method();
//			method_options.dont_initialize();
//			method_options.set_sensitivity(&set_initiator_phase_end_resp_event[i]);
//			string process_name("set_initiator_phase_end_resp_process_");
//			ostringstream out;
//			out << i;
//			process_name.append(out.str());
//
//			sc_spawn(sc_bind(&TestbenchOut<No>::set_initiator_phase_end_resp, this, i), process_name.c_str(), &method_options);
//		}

		// Add signals to trace
		init_packet();
		//sc_trace(tf, trace_packet, (string) name());
	}

	~TestbenchOut() {
		for(int i = 0; i < No; i++) {
			delete[] data_out[i];
		}
	}

	// Transfer methods
	static void dynamic_transfer_process_method_helper(void* context,int channel_id){
		((TestbenchOut<No>*)context)->dynamic_transfer_process_method(channel_id);
	}
	void *dynamic_transfer_process_method(int channel_id);
	void initiator_process(int id);	

	void wait_on_sinc();
	void start_tb();
	// Payload event queue callback function
	void peq_callback(rs_payload& trans, const rs_phase& phase);
	
	// Transport methods
	virtual rs_sync_enum nb_transport_bw(int id, rs_payload& trans, rs_phase& phase);
	void b_transport_manager(rs_payload& trans/*, rs_time& delay*/);

	// Simulation controller interface
	void notify_simulation_controller(bool);

	// User interface
	void verify_bindings(void);
	void set_packet_size(string connection_name, int val);
	void set_following_modules_list(vector<int> &v);
	void set_application_offset(int offset);

	void add_following_module_connection_name(string name);
	//void add_timing_info(rs_phase_enum phase, string connection_name, rs_time time);
	void set_begin_req_time(string connection_name, rs_time t);
	void set_end_resp_time(string connection_name, rs_time t);

	void info(void);

	// Priority channels
	void add_priority_channel(string connection_name);
	bool is_priority_channel_out(int id);

	// Trace packet
	void init_packet(void);
	//void update_initiator_packet(int id, rs_payload* trans, rs_phase phase);
	void update_initiator_packet(int id, rs_payload* trans);
	void set_initiator_phase_begin_req(int id);
	void set_initiator_phase_end_resp(int id);

	void set_application_name(string s);
	void set_period(rs_time p);

	int get_output_socket_id(string connection_name) const;

	// TB OUT Interface
	ostream& get_logfile(void);
	string TB_IF_name(void) const;
	string TB_IF_get_instance_name(void) const;
	string TB_IF_get_instance_top_application_name(void) const;
	int* TB_IF_get_data_out_ptr(int) const;
	int* TB_IF_get_data_out_ptr(string) const;
	void TB_IF_nb_send_data(int socketID);
	void TB_IF_nb_send_data(string connectionName);
	void TB_IF_nb_send_all_data(void);
	//const rs_event& TB_IF_all_responses_received_event(void) const;
	void TB_IF_all_responses_received_wait(void);
	void TB_IF_increment_current_transaction_id(void);
	string TB_IF_getApplicationName(void);
	rs_time TB_IF_get_period(void) const;
	//void TB_IF_add_execution_mode_command_to_queue(string module_name, string command, rs_time delay);
	int TB_IF_get_packet_size(int ID) const;
	int TB_IF_get_packet_size(string connectionName) const;
	int TB_IF_get_output_socket_id(string name) const;

	void launch_algorithm_command(void);
	void send_algorithm_command_to_manager(AlgorithmModeCommand*);

	void manager_b_transport(int module_id);
};


template<int No>
void TestbenchOut<No>::wait_on_sinc(){
	lock_tb_sinc.wait();
}
template<int No>
void TestbenchOut<No>::start_tb(){
	lock_tb_sinc.notify();
}

/*******************************************
 ******  Manager communication part   ******
 *******************************************/

// REM: Communication du module vers reconf manager
// QUESTIONFAB : mais elle sert exactement  quoi cette fonction ? elle prpare un paquet pour quoi ? On dirait que le paquer contient que le module id comme info ?

template<int No>
void TestbenchOut<No>::manager_b_transport(int module_id) {

	rs_payload* trans;
	rs_time delay(RS_ZERO_TIME);
	int local_data = module_id;

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	// Set all attributes except byte_enable_length and extensions (unused)
	trans->set_command(COM_READ_COMMAND);
	trans->set_address(TB_TO_MANAGER_CHECK_SOCKET);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(&local_data));
	trans->set_data_length(4);
	//trans->set_streaming_width(4);							// = data_length to indicate no streaming
	//trans->set_byte_enable_ptr(0);							// 0 indicates unused
	//trans->set_dmi_allowed(false);							// Mandatory initial value
	//trans->set_response_status(TLM_INCOMPLETE_RESPONSE);	// Mandatory initial value

	manager_initiator_socket.b_transport(*trans/*, delay*/);

	//if(trans->is_response_error()) SC_REPORT_ERROR("TLM-2", "Response error from TB b_transport");

	//Utils::check_transaction(*trans);
	trans->release();
}

template<int No>
void TestbenchOut<No>::send_algorithm_command_to_manager(AlgorithmModeCommand* cmd) {
	FOUT << ": " << "\033[37;4m"<< "name" << "\033[39;24m"<< " send command to scheduler " << endl;

	rs_payload* trans;
	rs_time delay(RS_ZERO_TIME);

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	// Set all attributes except byte_enable_length and extensions (unused)
	trans->set_command(COM_WRITE_COMMAND);
	trans->set_address(TB_TO_MANAGER_ALGORITHM_COMMAND);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(cmd));
	trans->set_data_length(4);
	//trans->set_streaming_width(4);							// = data_length to indicate no streaming
	//trans->set_byte_enable_ptr(0);							// 0 indicates unused
	//trans->set_dmi_allowed(false);							// Mandatory initial value
	//trans->set_response_status(TLM_INCOMPLETE_RESPONSE);	// Mandatory initial value

	manager_initiator_socket->b_transport(*trans);

	//if(trans->is_response_error()) SC_REPORT_ERROR("TLM-2", "Response error from TB b_transport");

	//Utils::check_transaction(*trans);
	trans->release();
}

template<int No>
void TestbenchOut<No>::b_transport_manager(rs_payload& trans/*, rs_time& delay*/) {

	//FOUT << ": " << "\033[37;4m"<< "name" << "\033[39;24m"<< "called by scheduler " << endl;
	if(trans.get_address() == MANAGER_COMMAND_SCHEDULING) {

		int data = *(reinterpret_cast<int *>(trans.get_data_ptr()));
		User_algorithm_action action = (User_algorithm_action) (data & 0xffff);
		int module_ready = data >> 16;

		// If next module ready, launch initiator
		if(action == NEXT_MODULE_READY) {

			FOUT << rs_time_stamp() << ": " << name << ": Module " << module_ready << " ready" << endl;

			// Multiple channels may be connected to the same module. Check for this particular case
			for(int i = 0; i < No; i++) {

				for(int i = 0; i < (int) channels_required[module_ready].size(); i++) {
					//FOUT << rs_time_stamp() << ": " << name << ": Channel " << channels_required[module_ready].at(i) << " ready" << endl;
					start_initiator_channel_event[channels_required[module_ready].at(i)].notify();
				}

				channels_required[module_ready].clear();
			}
		}

		// Wait, but delay should be 0
		//wait(delay);

		//trans.set_response_status(TLM_OK_RESPONSE);

	} else {
		cerr << rs_time_stamp() << ": " << name << ": ERROR: Address contained a wrong command" << endl;
		exit(-1);
		//Simulation_controller::endSimulation();
	}
}


/****************************
 ****   Initiator part   ****
 ****************************/

template<int No>
rs_sync_enum TestbenchOut<No>::nb_transport_bw(int id, rs_payload& trans, rs_phase& phase) {

	// Put the transaction into the queue
	m_id_init_map[&trans] = id;
	m_peq.notify(trans, phase);
    return RS_ACCEPTED;
}

//Payload event queue callback function
template<int No>
void TestbenchOut<No>::peq_callback(rs_payload& trans, const rs_phase& phase) {
	
	int socketID = m_id_init_map[&trans];

		//if (phase == END_REQ)
			//FOUT << rs_time_stamp() << ": " << name << " (I" << socketID << ") received END_REQ at address " << hex << trans.get_address() << endl;
		//else if (phase == BEGIN_RESP)
			//FOUT << rs_time_stamp() << ": " << name << " (I" << socketID << ") received BEGIN_RESP at address " << hex << trans.get_address() << endl;

	if (phase == END_REQ || (&trans == request_in_progress[socketID] && phase == BEGIN_RESP)) {
		//update_initiator_packet(socketID, &trans, phase);

		// The end of the BEGIN_REQ phase
		request_in_progress[socketID] = 0;

		if(!is_priority_channel_out(socketID)) nb_non_priority_channels_requests_out--;
		if(nb_non_priority_channels_requests_out == 0) shared_channel_out_ready_event.notify();
		
	} else if (phase == BEGIN_REQ || phase == END_RESP){
		cerr << "Illegal transaction phase received by initiator (testbench)" <<endl;
		exit(-1);
		//SC_REPORT_FATAL("TLM-2", "Illegal transaction phase received by initiator (testbench)");
	}
	if (phase == BEGIN_RESP) {

		// The target has finished processing
		//Utils::check_transaction(trans);
		trans.release();

		// Send final phase transition to target
		rs_phase fw_phase = END_RESP;
		rs_time delay = end_resp_time[socketID];
		output_initiator_socket[socketID]->nb_transport_fw(trans, fw_phase);
		// Ignore return value

		set_initiator_phase_end_resp_event[socketID].notify();

		// End of transaction
		request_in_progress[socketID] = 0;
		//response_received_event.notify();
		
		
		if(++nb_responses_received == No) {
			nb_responses_received = 0;
			all_responses_received_event.notify();
		}
		
		//update_initiator_packet(socketID, &trans, phase);
	}
}


template<int No>
void *TestbenchOut<No>::dynamic_transfer_process_method(int channel_id) {
	//FOUT << "dynamic_transfert creat" << endl;
	rs_event *my_trigger = &start_initiator_channel_event[channel_id];
	rs_event *current_trigger = my_trigger;
	while(1){
		//FOUT << rs_time_stamp() << ": " << name << ": wait on start initiator envent " << channel_id << endl;
		current_trigger->wait();
		//FOUT << rs_time_stamp() << ": Module " << name << ": Channel " << channel_id << " ready! Launching initiator" << endl;
		if(is_priority_channel_out(channel_id)) FOUT << rs_time_stamp() << ": Module " << name << ": Priority channel!" << endl;
		else 					FOUT << rs_time_stamp() << ": Module " << name << ": Non priority channel!" << endl;

		if(request_in_progress[channel_id]) {
			//FOUT << rs_time_stamp() << ": " << name << ": request already in progress in channel " << channel_id << ", waiting for end_of_response_channel_event" << endl;
			//next_trigger(end_of_response_channel_event[channel_id]);
			current_trigger = &end_of_response_channel_event[channel_id];
		}

		else if(!is_priority_channel_out(channel_id) && nb_non_priority_channels_requests_out != 0) {
			//FOUT << rs_time_stamp() << ": " << name << ": request already in progress in shared channel, waiting for shared_channel_ready" << endl;
			//next_trigger(shared_channel_out_ready_event);
			current_trigger = &shared_channel_out_ready_event;
		}
		else{
			initiator_process(channel_id);
			//next_trigger();
			current_trigger = my_trigger;
		}
	}
}

template<int No>
void TestbenchOut<No>::initiator_process(int id) {

	rs_payload* trans;
	rs_phase phase = BEGIN_REQ;

	if(!is_priority_channel_out(id)) nb_non_priority_channels_requests_out++;

	trans = m_mm.allocate();
	trans->acquire();

	FOUT << endl << rs_time_stamp() << ": " << name << " creates transaction " << current_transaction_id << " for socket " << id << endl;
		
	int adr = current_transaction_id;
	FOUT << "Address: " << adr << endl;

	// Set all attributes except byte_enable_length and extensions (unused)
	trans->set_command(COM_WRITE_COMMAND);
	trans->set_address(adr);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(data_out[id]));
	trans->set_data_length(4 * output_packet_size[id]);
	
	request_in_progress[id] = trans;


	rs_sync_enum status = output_initiator_socket[id]->nb_transport_fw(*trans, phase);


	FOUT << endl << rs_time_stamp() << ": " << name << " end initiator " << current_transaction_id << " for socket " << id << endl;
}

template<int No>
void TestbenchOut<No>::notify_simulation_controller(bool success) {
	Reconfiguration_manager::endApplication(applicationName, success);
}

/******************************
 ****    User interface    ****
 ******************************/
template<int No>
void TestbenchOut<No>::set_packet_size(string connection_name, int val) {

	int socketID = -1;
	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
		if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}

#ifdef CONSOLE_DEBUG
	//cout << "Following module" << endl;
	//cout << "Socket ID: " << socketID << endl;
#endif

	try {
		assert(socketID < No);
		output_packet_size[socketID] = val;
		delete[] data_out[socketID];
		data_out[socketID] = new int[output_packet_size[socketID]];
		for(int i = 0; i < output_packet_size[socketID]; i++) data_out[socketID][i] = 0;
	} catch (bad_alloc& ba) {
		cerr << "ERROR: " << name << " bad_alloc caught: " << ba.what() << endl;
		exit(-1);
		//Simulation_controller::endSimulation();
	}
}

template<int No>
void TestbenchOut<No>::set_following_modules_list(vector<int> &v) {

	channel_connection_vector = v;
	following_modules_list.clear();

	// Check for redundancies for following modules list
	for(int i = 0; i < (int) channel_connection_vector.size(); i++) {
		bool item_found = false;
		for(int j = 0; j < (int) following_modules_list.size(); j++) {
			if(following_modules_list.at(j) == channel_connection_vector.at(i)) item_found = true;
		}
		if(!item_found) following_modules_list.push_back(channel_connection_vector.at(i));
	}
}

template<int No>
void TestbenchOut<No>::set_application_offset(int offset) {
	for(int j = 0; j < (int) following_modules_list.size(); j++) {
		if(following_modules_list.at(j) != -1) following_modules_list[j] += offset;
	}
	for(int j = 0; j < (int) channel_connection_vector.size(); j++) {
		if(channel_connection_vector.at(j) != -1) channel_connection_vector[j] += offset;
	}
}

template<int No>
void TestbenchOut<No>::verify_bindings(void) {
	//if(output_initiator_socket.size() != No) {
	//	cerr << "ERROR: Incorrect bindings for module " << name() << "! Exiting..." << endl;
	//	exit(-1);
	//}
}


template<int No>
void TestbenchOut<No>::set_begin_req_time(string connection_name, rs_time t) {
	//add_timing_info(BEGIN_REQ, connection_name, t);
}

template<int No>
void TestbenchOut<No>::set_end_resp_time(string connection_name, rs_time t) {
	//add_timing_info(END_RESP, connection_name, t);
}

template<int No>
void TestbenchOut<No>::add_following_module_connection_name(string name) {
	following_modules_connection_name.push_back(name);
}

//template<int No>
//void TestbenchOut<No>::add_timing_info(rs_phase_enum phase, string connection_name, rs_time time) {
//	
//	int socketID = -1;
//	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
//		if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
//			socketID = i;
//			break;
//		}
//	}
//	
//	switch(phase) {
//		case BEGIN_REQ: begin_req_time[socketID] = time; break;
//		case END_RESP: end_resp_time[socketID] = time; break;
//		default: cerr << "ERROR: Phase not recognized!" << endl;
//	}
//}

template<int No>
void TestbenchOut<No>::info(void) {

	FOUT << endl << "Testbench " << name << ":" << endl;
	FOUT << "Period: " << period << endl;

	for(int i = 0; i < No; i++) {
		FOUT << "Output socket " << i << " (Connection " << following_modules_connection_name.at(i) << "):" << endl;
		if(priority_channels_out.find(i) != priority_channels_out.end()) FOUT << "  Priority channel" << endl;
		FOUT << "  Packet size: " << output_packet_size[i] << endl;
		FOUT << "  BEGIN_REQ time: " << begin_req_time[i] << endl;
		FOUT << "  END_RESP time: " << end_resp_time[i] << endl;
	}

	FOUT << "Following list: ";
	for(int i = 0; i < (int) following_modules_list.size(); i++) FOUT << dec << following_modules_list.at(i) << " ";
	FOUT << endl;
}

// Priority channels
template<int No>
void TestbenchOut<No>::add_priority_channel(string connection_name) {
	int id = -1;

	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
		if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
			id = i;
			break;
		}
	}

	if(id == -1) cerr << "ERROR: Module " << name << " cannot find connection name through preceding or following modules vector!" << endl;

	assert(id < No);
	priority_channels_out.insert(id);
}

template<int No>
bool TestbenchOut<No>::is_priority_channel_out(int id) {
	return (priority_channels_out.find(id) != priority_channels_out.end());
}

template<int No>
void TestbenchOut<No>::set_application_name(string s) {
	applicationName = s;
}

template<int No>
void TestbenchOut<No>::set_period(rs_time p) {
	period = p;
}


template<int No>
int TestbenchOut<No>::get_output_socket_id(string connection_name) const {
	int socketID = -1;
	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
		if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}

	if(socketID == -1) {
		cerr << "Error: Testbench unable to retrieve output socket ID via connection name" << endl;
		exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
	}

	return socketID;
}


/*********************************
 *    Testbench out interface    *
 *********************************/
template<int No>
ostream& TestbenchOut<No>::get_logfile(void){
	return (*fout);
}

template<int No>
string TestbenchOut<No>::TB_IF_name(void) const {
	return name;
}

template<int No>
string TestbenchOut<No>::TB_IF_get_instance_name(void) const {
	string fullname(name);
	if(fullname.find("/") != string::npos) {
		return (fullname.substr(fullname.find("/") + 1)).c_str();
	} else return name;
}

template<int No>
string TestbenchOut<No>::TB_IF_get_instance_top_application_name(void) const {
    if (applicationName.find("/") != string::npos)
        return (applicationName.substr(applicationName.find("/") + 1)).c_str();
    else
        return applicationName;
}

template<int No>
int* TestbenchOut<No>::TB_IF_get_data_out_ptr(int socketID) const {
	return data_out[socketID];
}

template<int No>
int* TestbenchOut<No>::TB_IF_get_data_out_ptr(string connectionName) const {
	return data_out[get_output_socket_id(connectionName)];
}

template<int No>
void TestbenchOut<No>::TB_IF_nb_send_data(int socketID) {
	// Retrieve module ID
	//FOUT << "\033[37;4m" << "socket ID: " << channels_required.empty() << "\033[39;24m"<< endl;
	int module_id = channel_connection_vector.at(socketID);
	//FOUT << "Module ID: " << module_id << endl;

	// QUESTIONFAB : a quoi cela sert exactement "channels_required" ?
	bool first_module_request = (channels_required[module_id].empty());
	channels_required[module_id].push_back(socketID);

	if(first_module_request) manager_b_transport(module_id);
	
	FOUT << " End of send" << endl;

}

template<int No>
void TestbenchOut<No>::TB_IF_nb_send_data(string connectionName) {
	TB_IF_nb_send_data(get_output_socket_id(connectionName));
}

template<int No>
void TestbenchOut<No>::TB_IF_nb_send_all_data(void) {
	int i = 0;
	for(i = 0; i < No; i++) TB_IF_nb_send_data(i);
}

//template<int No>
//const rs_event& TestbenchOut<No>::TB_IF_all_responses_received_event(void) const {
	//return all_responses_received_event;
//}
template<int No>
void TestbenchOut<No>::TB_IF_all_responses_received_wait(void){
	all_responses_received_event.wait();
}

template<int No>
void TestbenchOut<No>::TB_IF_increment_current_transaction_id(void) {
	current_transaction_id++;
}

template<int No>
string TestbenchOut<No>::TB_IF_getApplicationName(void) {
	return applicationName;
}

template<int No>
rs_time TestbenchOut<No>::TB_IF_get_period(void) const {
	return period;
}

//template<int No>
//void TestbenchOut<No>::TB_IF_add_execution_mode_command_to_queue(string module_name, string command, rs_time delay) {
//	algorithm_mode_command_priority_queue.push(AlgorithmModeCommand(module_name, command, delay));
//	algorithm_mode_command_event_queue.notify(delay);
//}

template<int No>
void TestbenchOut<No>::launch_algorithm_command(void) {

	AlgorithmModeCommand top_command(algorithm_mode_command_priority_queue.top());
	//cout << name << ": Popping command " << top_command.get_command() << " for module " << top_command.get_target_module() << " at t=" << rs_time_stamp() << endl;
	send_algorithm_command_to_manager(&top_command);
	algorithm_mode_command_priority_queue.pop();

}

template<int No>
int TestbenchOut<No>::TB_IF_get_packet_size(int ID) const {
	return output_packet_size[ID];
}

template<int No>
int TestbenchOut<No>::TB_IF_get_packet_size(string connectionName) const {
	return TB_IF_get_packet_size(get_output_socket_id(connectionName));
}

template<int No>
int TestbenchOut<No>::TB_IF_get_output_socket_id(string connectionName) const {
	return get_output_socket_id(connectionName);
}


/**********************************
 *      Packet trace methods      *
 **********************************/
template<int No>
void TestbenchOut<No>::init_packet(void) {
	//update_initiator_packet(0);
	//update_target_packet(0);
}

//template<int No>
//void TestbenchOut<No>::update_initiator_packet(int id, rs_payload* trans, rs_phase phase) {
//	assert(id < No);
	//trace_packet.initiator_packet[id].phase = rs_phase_to_ascii(phase);
//	update_initiator_packet(id, trans);
//}

template<int No>
void TestbenchOut<No>::update_initiator_packet(int id, rs_payload* trans) {
	assert(id < No);
	//trace_packet.initiator_packet[id].transaction.update_transaction(trans);
}

template<int No>
void TestbenchOut<No>::set_initiator_phase_begin_req(int id) {
	assert(id < No);
	//trace_packet.initiator_packet[id].phase = BEGIN_REQ_ASCII;
}

template<int No>
void TestbenchOut<No>::set_initiator_phase_end_resp(int id) {
	assert(id < No);
	//trace_packet.initiator_packet[id].phase = END_RESP_ASCII;
}

#endif
