/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    application.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-08-30
 * @section DESCRIPTION
 *			Application representation
 */

#include <stdint.h>
#include "application.h"
#include "reconfiguration_manager.h"

rs_time Application::minimumSimulationTime = RS_ZERO_TIME;
rs_time Application::globalOffset = RS_ZERO_TIME;
rs_time Application::globalHyperperiod = RS_ZERO_TIME;
rs_time Application::userOffset = RS_ZERO_TIME;
rs_time Application::userHyperperiod = RS_ZERO_TIME;
bool Application::overrideOffset = false;
bool Application::overrideHyperperiod = false;

/** A set of helper functions */
uint64_t computeGCD(uint64_t a, uint64_t b) {
	if(a == 0 && b != 0) return b;
	if(b == 0 && a != 0) return a;
	if(a == b) return a;
	if(a < b) {
		uint64_t tmp = b;
		b = a;
		a = tmp;
	}
	return computeGCD(b, a % b);
}

uint64_t computeLCM(uint64_t a, uint64_t b) {
	return a * b / computeGCD(a, b);
}

uint64_t convertTimeToMicroseconds(rs_time t) {
	return  t.value();
}

/** Application class */
string Application::getName(void) {
	return name;
}

int Application::getFirstTaskID(void) {
	return firstTaskID;
}

int Application::getTaskNumber() {
	return (lastTaskID - firstTaskID) + 1;
}

/*
double Application::getQoSFactor(void) {
	return qosFactor;
}
*/

double Application::getAchievedQoS(void) {
	// In percent
	if((nbPassedDeadlines + nbMissedDeadlines) != 0) return (double) ((nbPassedDeadlines * 100.0)/(nbPassedDeadlines + nbMissedDeadlines));
	else return 100.0;
}

double Application::getMinimalQoS(void) {
	return minimalInstantQoS;
}

bool Application::containsTask(int taskID) {
	return (taskID >= firstTaskID && taskID <= lastTaskID);
}

void Application::incrementPassedDeadlines(void) {
	nbPassedDeadlines++;
	nbPassedDeadlinesOnHyperperiod++;
}

void Application::incrementMissedDeadlines(string taskname) {
	nbMissedDeadlines++;
	nbMissedDeadlinesOnHyperperiod++;

	missed_deadlines.push_back(pair<string, rs_time>(taskname, rs_time_stamp()));
}

void Application::computeMinimumSimulationTime(vector<Manager_interface *>& table) {

	if(firstTaskID == lastTaskID) {
		offset = RS_ZERO_TIME;
		if(table.at(firstTaskID)->get_task_period() == RS_ZERO_TIME) hyperperiod = table.at(firstTaskID)->get_task_deadline();
		else hyperperiod = table.at(firstTaskID)->get_task_deadline();
	} else {
	
		// Compute offset
		for(int i = firstTaskID; i <= lastTaskID; i++) {
			// Search biggest WCET through task implementation vector
			rs_time maximumWCET = table.at(i)->get_implementation_vector().front().get_worst_case_execution_time();
			for(int j = 1; j < (int) table.at(i)->get_implementation_vector().size(); j++) {
				if(table.at(i)->get_implementation_vector().at(j).get_worst_case_execution_time() > maximumWCET) maximumWCET = table.at(i)->get_implementation_vector().at(j).get_worst_case_execution_time();
			}

			offset += maximumWCET;

			// Also add task offset
			offset += table.at(i)->get_task_offset();
		}

		// Compute hyperperiod (Least Common Multiple of task deadlines or task periods)
		rs_time val1 = (table.at(firstTaskID)->get_task_period() == RS_ZERO_TIME) ? table.at(firstTaskID)->get_task_deadline() : table.at(firstTaskID)->get_task_period();
		rs_time val2 = (table.at(firstTaskID + 1)->get_task_period() == RS_ZERO_TIME) ? table.at(firstTaskID + 1)->get_task_deadline() : table.at(firstTaskID + 1)->get_task_period();
		uint64_t lcm = computeLCM(convertTimeToMicroseconds(val1), convertTimeToMicroseconds(val2));
		for(int i = (firstTaskID + 2); i <= lastTaskID; i++) {
			rs_time val = (table.at(i)->get_task_period() == RS_ZERO_TIME) ? table.at(i)->get_task_deadline() : table.at(i)->get_task_period();
			lcm = computeLCM(lcm, convertTimeToMicroseconds(val));
		}
		hyperperiod = rs_time((double)lcm);
	}
}

void Application::setAchievedSimulationTime(rs_time t) {
	achievedSimulationTime = t;
}

rs_time Application::getAchievedSimulationTime(void) {
	return achievedSimulationTime;
}

bool Application::isSimulationTimeEnough(void) {
	return (achievedSimulationTime >= minimumSimulationTime);
}

bool Application::getStatus(void) {
	return status;
}

void Application::setStatus(bool s) {
	status = s;
}

void Application::checkQoSOnHyperperiod(void) {

	if(nbMissedDeadlinesOnHyperperiod + nbPassedDeadlinesOnHyperperiod != 0) {
		double instantQoS = (double)nbPassedDeadlinesOnHyperperiod * 100.0 / (double)(nbPassedDeadlinesOnHyperperiod + nbMissedDeadlinesOnHyperperiod);
		if(instantQoS < qosFactor) {
			//Simulation_controller::get_logfile() << rs_time_stamp() << ": QoS too low on last hyperperiod for task " << name  << endl;
			//Simulation_controller::get_logfile() << "Achieved " << instantQoS << "%, required " << qosFactor << "%" << endl;
			//Simulation_controller::get_logfile() << rs_time_stamp() << ": Stopping simulation..." <<  endl;
			qosFailure = true;
			Reconfiguration_manager::endApplication(name, false);
		} else {
			//Simulation_controller::get_logfile() << rs_time_stamp() << ": Achieved " << instantQoS << "%, required " << qosFactor << "%" << endl;
		}

		if(instantQoS < minimalInstantQoS) minimalInstantQoS = instantQoS;
	}

	nbMissedDeadlinesOnHyperperiod = 0;
	nbPassedDeadlinesOnHyperperiod = 0;
	missed_deadlines_last_hyperperiod = missed_deadlines;
	missed_deadlines.clear();
}

rs_time Application::getOffset(void) {
	return offset;
}

rs_time Application::getHyperperiod(void) {
	return hyperperiod;
}

vector<pair<string, rs_time> >& Application::getMissedDeadlinesVector(void) {
	return missed_deadlines_last_hyperperiod;
}

bool Application::hasMissedDeadlines(void) {
	if(qosFailure) return !missed_deadlines_last_hyperperiod.empty();
	else return false;
}

string Application::getFullModuleName(string moduleName) {
	string fullname;
	fullname.append(prefix);
	fullname.append(moduleName);
	if(instanceID != -1) {
		fullname.append("_");
		fullname.append(Utils::itoa(instanceID));
	}

	return fullname;
}

void Application::computeGlobalHypeperiod(vector<Application> &applications) {
	
	rs_time global_hyperperiod;

	if(applications.size() == 1) global_hyperperiod = applications.front().getHyperperiod();
	else {
		uint64_t global_hyperperiod_int = computeLCM(convertTimeToMicroseconds(applications.at(0).getHyperperiod()), convertTimeToMicroseconds(applications.at(1).getHyperperiod()));
		
		for(int i = 2; i < (int) applications.size(); i++) {
			global_hyperperiod_int = computeLCM(global_hyperperiod_int, convertTimeToMicroseconds(applications.at(i).getHyperperiod()));
		}

		global_hyperperiod = rs_time((double)global_hyperperiod_int);
	}
	
	globalHyperperiod = global_hyperperiod;
}

void Application::computeGlobalOffset(vector<Application> &applications) {
	rs_time maximumOffset = RS_ZERO_TIME;
	
	for(int i = 0; i < (int) applications.size(); i++) {
		if(applications.at(i).getOffset() > maximumOffset) maximumOffset = applications.at(i).getOffset();
	}

	globalOffset = maximumOffset;
}

rs_time Application::getGlobalHypeperiod(void) {
	if(overrideHyperperiod) return userHyperperiod;
	else return globalHyperperiod;
}

rs_time Application::getGlobalOffset(void) {
	if(overrideOffset) return userOffset;
	else return globalOffset;
}

void Application::computeMinimumSimulationTime(vector<Application> &applications) {
	computeGlobalOffset(applications);
	computeGlobalHypeperiod(applications);
	minimumSimulationTime = globalOffset + globalHyperperiod;
}

rs_time Application::getMinimumSimulationTime(void) {
	return minimumSimulationTime;
}

bool Application::isQoSSatisfied(void) {
	return !qosFailure;
}

void Application::setOffset(rs_time t) {
	userOffset = t;
	overrideOffset = true;
}

void Application::setHyperperiod(rs_time t) {
	userHyperperiod = t;
	overrideHyperperiod = true;
}
