/**
 * UNS - CNRS
 * Copyright 2013 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: licensing@sattse.com
 *
 * @file    qos_management.h
 * @author  Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0-build:2013/06/10
 * @date    July 22, 2014 2:10:21 PM CEST
 * @section DESCRIPTION
 *
 *          Monitoring and Trace Metrics
*/

#ifndef QOS_MANAGEMENT_H
#define QOS_MANAGEMENT_H

#include <systemc.h>
#include "utils.h"
#include "monitoring.h"
#include "qos_interface.h"

#define FOUT if(generate_logfile()) (*fout)

class QoS_management : public sc_module {

protected:

	// Log file stream
	ofstream* fout;

public:

	QoS_interface& services;

	// Trace Signal
	TRACE_VAR_SC_BIGINT_PTR(QoS_var, TRACE_LV_SIZE);

	QoS_management(sc_module_name nm, QoS_interface& _qos_if, ofstream* _fout) : sc_module(nm), services(_qos_if), fout(_fout) {

		// Create Common Thread
		sc_spawn_options method_options;
		sc_spawn(sc_bind(&QoS_management::QoS_common_thread, this, services.getApplications().size(), Monitoring::thread_number++), "Dynamic_process_QoS_common", &method_options);

		// Create Thread for each Application
		for(int i = 0; i < (int) services.getApplications().size(); i++) {
			sc_spawn_options method_options;
			string process_name("Dynamic_process_QoS_");
			ostringstream out;
			out << i;
			process_name.append(out.str());

			sc_spawn(sc_bind(&QoS_management::QoS_application_thread, this, i, services.getApplications().at(i), Monitoring::thread_number++), process_name.c_str(), &method_options);
		}

		// Activate Trace
		activate_common_trace(services.getApplications().size());
		for(int i = 0; i < (int) services.getApplications().size(); i++)
			activate_application_trace(i, services.getApplications().at(i));
	}

	QOS_COMMON_ACTIVATE {

		TRACE_VAR_SC_BIGINT_ALLOCATE(QoS_var, TRACE_LV_SIZE, ApplicationNumber);

	}

	QOS_APPLICATION_ACTIVATE {

		// Init
		QoS_var[applicationID] = 0;

		// Trace
		QOS_TRACE(QoS_var[applicationID], application.getName() + ".QoS_var", unsigned);

	}


	// !! Ajouter une application interface


	QOS_COMMON_THREAD {

		cout << "Start common application "  << endl;

		while(true) {

			wait();

			cout << "NEVER common application "  << endl;

		}

	}

	// Resultat true, false
	// Top/Bottom
	// Typage vnement

	// Fermer le flux log solution aprs ...

	QOS_APPLICATION_THREAD {

		vector<Application> appList = services.getApplications();

		// TEST
		int current_hyperperiod = 0;

		cout << "Start application " << application.getName() << endl;

		wait(application.getGlobalOffset());
		while(true) {

			// New hyperperiod -> update VCD trace
			//set_hyperperiod_status(++current_hyperperiod);

			string hpid_str(Utils::itoa(++current_hyperperiod));
			QoS_var[applicationID] = ((sc_bigint<TRACE_LV_SIZE>) Utils::string_to_ascii("Hyperperiod ") << (hpid_str.size() * 8))
								+ Utils::string_to_ascii(hpid_str);

			wait(application.getGlobalHypeperiod());

			cout << "QoS check ME for application " << application.getName() << endl;

			FOUT << sc_time_stamp() << ": " << name() << ": QoS check ME for application " << application.getName() << endl;
			application.checkQoSOnHyperperiod();
		}

	}

};


#endif
