/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    module.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-05-10
 * @section DESCRIPTION
 *			Representation of a sub-module. Number of IO can be tuned
 *			using templates.
 */

#ifndef MODULE_H
#define MODULE_H

#define SC_INCLUDE_DYNAMIC_PROCESSES

#include <systemc.h>
#include "tlm.h"
#include "tlm_utils/peq_with_cb_and_phase.h"
#include "tlm_utils/multi_passthrough_initiator_socket.h"
#include "tlm_utils/multi_passthrough_target_socket.h"
#include "tlm_utils/simple_initiator_socket.h"
#include "tlm_utils/simple_target_socket.h"

#include "memory_manager.h"
#include "reconf_thread_interface.h"
#include "user_algorithm_interface.h"
#include "trace.h"
#include "utils.h"
#include "manager_interface.h"
#include "module_interface.h"

#include <vector>
#include <map>
#include <set>
#include <sstream>

using namespace sc_core;
using namespace sc_dt;
using namespace std;
using namespace tlm;
using namespace tlm_utils;

#define FOUT if(generate_logfile()) (*fout)


//#define CONSOLE_DEBUG

template<int Ni = 1, int No = 1>
class Module : public sc_module, public Reconf_thread_interface, public User_algorithm_interface<Ni, No>, public Manager_interface {

public:

	// I/O Sockets
	multi_passthrough_target_socket<Module> target_socket;
	multi_passthrough_initiator_socket<Module> initiator_socket;

	// Configuration socket
	simple_target_socket<Module> manager_target_socket;
	simple_initiator_socket<Module> manager_initiator_socket;

private:

	// Memory manager
	Memory_manager m_mm;

	// Payload event queues
	peq_with_cb_and_phase<Module> m_peq;
	peq_with_cb_and_phase<Module> m_peq_targ;

	// Maps to retrieve socket ID from a transaction in PEQ callbacks
	map<tlm_generic_payload*, unsigned int> m_id_targ_map;
	map<tlm_generic_payload*, unsigned int> m_id_init_map;

	// Implementation vector
	vector<ModuleImplementation> implementation_execution_times_vector;

	// Data
	sc_dt::uint64 address_in[Ni];
	sc_dt::uint64 address_out[No];
	int *data_in[Ni];
	int *data_out[No];
	int transactions_processed;
	bool request_in_progress[No];

	// Transactions
	tlm_generic_payload* incoming_transaction[Ni];
	tlm_generic_payload* incoming_transaction_pending[Ni];
	tlm_generic_payload* outgoing_transaction[No];

	// Events
	sc_event end_of_response_channel_event[No];				/**< Event to notify end of response for a specific channel */
	sc_event configuration_updated_event;					/**< Event to notify a configuration */
	sc_event start_initiator_channel_event[No];				/**< Event to start a specific channel */
	sc_event question_manager_task_end_computation_event;	/**< Event to question the manager after executing a task */
	sc_event question_manager_task_preemption_event;		/**< Event to question the manager after hitting a preemption point */
	sc_event configuration_done_event;						/**< Event to notify that module configuration is done */
	sc_event continue_execution_event;						/**< Event launched when the task can resume its execution after a preemption point is reached */
	sc_event shared_channel_ready_event;					/**< Event to notify that the physical shared channel is ready */
	sc_event release_physical_channels_event[Ni];			/**< Event to decrement number of shared channels requests */
	sc_event send_end_req_event[Ni];						/**< Events to send a END_REQ */
	sc_event start_algorithm_event;

	sc_event parameter_computation_event;					/**< Event launched by the scheduler to update its execution time (as preemption)  */
	Algorithm_Parameters* parameter_computation_data;							/**< Data related to the parameter_computation_event */

	// Events to update trace packets
	sc_event set_target_phase_idle_event[Ni];
	sc_event set_initiator_phase_idle_event[No];

	// Timing parameters
	sc_time begin_req_time[No];			/**< BEGIN_REQ phase delay */
	sc_time end_req_time[Ni];			/**< END_REQ phase delay */
	sc_time begin_resp_time[Ni];		/**< BEGIN_RESP phase delay */
	sc_time end_resp_time[No];			/**< END_RESP phase delay */
	sc_time deadline_time;				/**< Deadline time */
	sc_time absolute_deadline_time;		/**< Time at which the deadline will expire */
	sc_time last_absolute_deadline_time;		/**< Time at which the deadline will expire */

	// Data stream configuration
	int input_stream_packet_size[Ni];	/**< Size for each input packet */
	int output_stream_packet_size[No];	/**< Size for each output packet */

	// Lists of previous and next modules
	vector<int> preceding_modules_list;
	vector<int> following_modules_list;
	vector<int> channel_connection_vector;				/**< Following modules, data may be redundant */
	vector<string> following_modules_connection_name;	/**< Following connections name */
	vector<string> preceding_modules_connection_name;	/**< Preceding connections name */
	int application_offset;								/**< Application offset in the manager */

	string applicationName;

	// Misc
	bool static_module;
	int priority;

	// Log file stream
	ofstream* fout;

	// Trace file
	bool enable_configuration_trace;
	sc_trace_file *tf;
	trace_packet_t<Ni, No> trace_packet;

	// Interfaces stuff
	map<string, ModuleInterface> module_interfaces;	// Map with all interfaces needed by the module
	int current_implementation_id;					// Current implementation ID
	map<string, int> physical_channels_map;			// Number of physical channels available for each kind provided by the implementation

	// Periodic tasks parameters
	sc_time task_period;
	sc_time task_offset;

	// Cyclic diagrams parameters
	bool first_transaction;
	bool first_request;

	// User algorithm thread modification stuff
	sc_event update_user_algorithm_event;
	bool kill_user_algorithm_thread;

	map<int, vector<int> > channels_required;

	sc_buffer<bool> socket_data_received[Ni];
	sc_event update_socket_data_received_buffer_event[Ni];
	bool m_new_val_data_received[Ni];

	sc_buffer<bool> socket_data_sent[No];
	sc_event update_socket_data_sent_buffer_event[No];
	bool m_new_val_data_sent[No];
	bool first_data_send_on_socket[No];

	bool algorithm_ended;
	int nb_packets_remaining;

	int nb_algorithm_executions;

	string algorithm_execution_mode;

	bool algorithm_execution_requested;

public:
	SC_HAS_PROCESS(Module);
	Module(
		sc_module_name instname,
		void (*reconf_fct)(Reconf_thread_interface &, User_algorithm_interface<Ni, No> &),
		ofstream& logfile,
		sc_trace_file *mtf,
		string applicationName_) :
			sc_module(instname),
			target_socket("Module_T_Socket"),
			initiator_socket("Module_I_Socket"),
			manager_target_socket("Module_MT_socket"),
			manager_initiator_socket("Module_MI_socket"),
			m_peq(this, &Module::peq_callback),
			m_peq_targ(this, &Module::peq_targ_callback),
			fout(&logfile),
			tf(mtf),
			applicationName(applicationName_),
			parameter_computation_data(NULL) {

		init();

		// Dynamic thread (Reconf thread)
		sc_spawn(sc_bind(reconf_fct, sc_ref(*this), sc_ref(*this)));
	}

	~Module() {
		for(int i = 0; i < Ni; i++) delete[] data_in[i];
		for(int i = 0; i < No; i++) delete[] data_out[i];
	}

private:
	// Main process
	void init(void);
	int retrieve_channel_id(int module_id);
	void dynamic_transfer_process_method(int channel_id);
	void initiator_process(int channel_id);

	// Manager scheduling threads & methods
	void question_manager_task_end_computation_thread(void);
	void question_manager_task_preemption_thread(void);
	void question_manager_b_transport(int address, int data);

	// Payload event queue callback function
	virtual void peq_callback(tlm_generic_payload& trans, const tlm_phase& phase);
	void peq_targ_callback(tlm_generic_payload& trans, const tlm_phase& phase);

	// Helper functions
	void send_response(int id, tlm_generic_payload& trans);
	void send_error_response(int id, tlm_generic_payload& trans);
	tlm_sync_enum send_end_req(int id, tlm_generic_payload& trans);
	vector<ModuleImplementation>* get_implementation_vector_ptr(void);
	void send_end_req_method(int id);

	// Trace packet
	void init_packet(void);
	void update_trace_packet(void);
	void update_initiator_packet(int id, tlm_generic_payload* trans, tlm_phase phase);
	void update_initiator_packet(int id, tlm_generic_payload* trans);
	void update_target_packet(int id, tlm_phase phase);
	void update_target_packet(int id);
	void set_com_status_waiting(int id);
	void set_com_status_tx(int id);
	void set_com_status_idle(int id);
	void set_target_phase(int id, tlm_phase& phase);
	void set_initiator_phase(int id, tlm_phase& phase);
	void set_config_phase(tlm_phase& phase);
	void trace_update_algorithm_execution_mode(void);
	void set_target_phase_idle(int);
	void set_initiator_phase_idle(int);

	void remove_priority_channels_from_physical_channel_map(void);
	bool are_priority_channels_respected(string);
	void release_physical_channel(string connection_name);
	void acquire_physical_channel(string connection_name);

public:

	/** User_algorithm_interface methods */
	const char* get_name() const;
	string get_top_application_name(void) const;
	int* get_data_in_ptr(int);
	int* get_data_in_ptr(string);
	int get_data_in_length(int) const;
	int get_data_in_length(string) const;
	sc_dt::uint64 get_address_in(int) const;
	sc_dt::uint64 get_address_in(string) const;
	int* get_data_out_ptr(int);
	int* get_data_out_ptr(string);
	int get_data_out_length(int) const;
	int get_data_out_length(string) const;
	void set_address_out(int, sc_dt::uint64);
	void set_address_out(string, sc_dt::uint64);
	int get_input_socket_id(string name) const;
	int get_output_socket_id(string name) const;
	sc_time get_WCET(void) const;
	sc_time get_BCET(void) const;
	void end_of_algorithm(void);
	void preemption_point(void);
	void compute(sc_time duration);
	bool isTransactionInMemory(void);
	const sc_event& getEOREvent(void) const;
	const sc_event& update_user_algorithm(void) const;
	void wait_for_update_user_algorithm(void);
	bool& kill_user_algorithm(void);
	void b_data_received(string);
	void b_data_received(int);
	void b_all_data_received(void);
	bool nb_data_received(string);
	bool nb_data_received(int);
	bool nb_all_data_received(void);
	void nb_send_data(string);
	void nb_send_data(int);
	void nb_send_all_data(void);
	void b_send_data(string);
	void b_send_data(int);
	void b_send_all_data(void);
	void b_data_sent(int id);
	bool nb_data_sent(int id);
	void b_data_sent(string name);
	bool nb_data_sent(string name);
	void b_all_data_sent(void);
	bool nb_all_data_sent(void);
	void start_new_transaction_sequence(void);
	void start_new_transaction_sequence(int id);
	void start_new_transaction_sequence(string name);
	bool is_channel_transient(string);
	bool is_channel_transient(int);
	int get_nb_preemption_points(void);
	void set_algorithm_running(void);
	void set_algorithm_idle(void);
	void set_algorithm_waiting(void);
	void request_updated_parameters(void);
	void wait_until_next_period(void);
	ofstream& get_logfile(void);
	string get_algorithm_execution_mode(void);
	bool is_algorithm_execution_mode(string);
	void release_input_socket(int socketID);
	void release_input_socket(string connectionName);
	void release_all_input_sockets(void);
	void b_execution_requested(void);

	/** Reconf_thread_interface methods */
	const sc_event& configuration_updated() const;
	void finish_configuration(void);
	int get_current_implementation_id(void);
	vector<ModuleImplementation> get_implementation_vector(void);

	/** Manager interface */
	bool has_implementation(TaskImplementation&) const;
	void set_current_implementation(string);
	vector<ModuleImplementation> get_implementation_vector(void) const;
	sc_time get_deadline_time(void) const;
	sc_time get_task_deadline(void) const;
	void set_request_time(void);
	bool fits(RZ *rz);
	string getName(void) const;
	int get_nb_processed_transactions(void) const;
	vector<int> getFollowingList(void) const;
	vector<int> getPrecedingList(void) const;
	void updateLists(int offset);
	bool is_static(void) const;
	int get_task_priority(void) const;
	sc_time get_task_period(void) const;
	sc_time get_task_offset(void) const;
	string get_current_implementation_fullname(void);
	bool has_current_implementation(void);
	ModuleImplementation get_current_implementation(void);
	void update_algorithm_execution_mode(string command);
	bool use_context_switch_mode(string implementationName);
	void send_start_algorithm_event(void);
	sc_time get_new_period_start_time(void) const;
	int get_nb_transient_channels(void);
	void set_task_priority(int priority);

	/** User interface */

	// Transport functions
	tlm_sync_enum nb_transport_bw(int id, tlm_generic_payload& trans, tlm_phase& phase, sc_time& delay);
	tlm_sync_enum nb_transport_fw(int id, tlm_generic_payload& trans, tlm_phase& phase, sc_time& delay);
	void b_transport_conf(tlm_generic_payload& trans, sc_time& delay);

	// Timing
	void add_timing_info(tlm_phase_enum phase, string connection_name, sc_time time);
	void set_begin_req_time(string connection_name, sc_time t);
	void set_end_req_time(string connection_name, sc_time t);
	void set_begin_resp_time(string connection_name, sc_time t);
	void set_end_resp_time(string connection_name, sc_time t);
	void set_deadline_time(sc_time t);

	// Next/Previous modules
	void set_preceding_modules_list(vector<int> &v);
	void set_following_modules_list(vector<int> &v);
	void add_following_module_connection_name(string name);
	void add_preceding_module_connection_name(string name);

	// Data stream configuration
	void set_packet_size(string connection_name, int size);

	// Trace
	void enable_configuration_signals_trace(void);
	void activate_trace(void);

	// Misc
	void info(void);
	void verify_bindings(void);
	void set_implementation_vector(vector<ModuleImplementation>& impl);
	void set_static_module(void);
	void set_priority(int);

	// Priority channels and typing
	void add_interface(ModuleInterface mod_if);
	void release_physical_channels_method(int channel_id);
	bool is_physical_channel_available(string connection_name);
	void display_physical_channels_map(void);
	string get_input_connection_name(int id);
	string get_output_connection_name(int id);

	void set_task_period(sc_time p);
	void set_task_offset(sc_time o);

	void update_socket_data_received_buffer(int);
	void update_socket_data_sent_buffer(int);
};

template<int Ni, int No>
void Module<Ni, No>::update_socket_data_received_buffer(int id) {
	assert(id<Ni);
	socket_data_received[id].write(m_new_val_data_received[id]);
}

template<int Ni, int No>
void Module<Ni, No>::update_socket_data_sent_buffer(int id) {
	assert(id<No);
	socket_data_sent[id].write(m_new_val_data_sent[id]);
}


// Initialization function
template<int Ni, int No>
void Module<Ni, No>::init(void) {

	// Some initializations
	for(int i = 0; i < Ni; i++) {
		incoming_transaction[i] = 0;
		incoming_transaction_pending[i] = 0;
		socket_data_received[i].write(false);
		m_new_val_data_received[i] = false;
		end_req_time[i] = SC_ZERO_TIME;
		begin_resp_time[i] = SC_ZERO_TIME;

		// Data stream configuration (default to 16 words)
		input_stream_packet_size[i] = 16;
		data_in[i] = new int[input_stream_packet_size[i]];
		for(int j = 0; j < input_stream_packet_size[i]; j++) data_in[i][j] = 0;
	}

	for(int i = 0; i < No; i++) {
		outgoing_transaction[i] = 0;
		request_in_progress[i] = false;
		socket_data_sent[i].write(false);
		m_new_val_data_sent[i] = false;
		first_data_send_on_socket[i] = true;
		begin_req_time[i] = SC_ZERO_TIME;
		end_resp_time[i] = SC_ZERO_TIME;

		// Data stream configuration (default to 16 words)
		output_stream_packet_size[i] = 16;
		data_out[i] = new int[output_stream_packet_size[i]];
		for(int j = 0; j < output_stream_packet_size[i]; j++) data_out[i][j] = 0;
	}

	algorithm_execution_requested = false;

	transactions_processed = 0;
	priority = 1;

	task_period = SC_ZERO_TIME;
	task_offset = SC_ZERO_TIME;

	first_transaction = true;
	first_request = true;

	kill_user_algorithm_thread = false;

	current_implementation_id = -1;

	algorithm_ended = false;
	nb_packets_remaining = 0;
	static_module = false;

	nb_algorithm_executions = 0;

	algorithm_execution_mode = "Default";

	// Init timing parameters
	deadline_time = SC_ZERO_TIME;
	absolute_deadline_time = deadline_time;
	last_absolute_deadline_time = SC_ZERO_TIME;

	// Register transport methods
	initiator_socket.register_nb_transport_bw(this, &Module<Ni, No>::nb_transport_bw);
	target_socket.register_nb_transport_fw(this, &Module<Ni, No>::nb_transport_fw);
	manager_target_socket.register_b_transport(this, &Module<Ni, No>::b_transport_conf);

	application_offset = 0;

	// Init trace packet
	init_packet();

	// SC_THREADS initialization
	SC_THREAD(question_manager_task_end_computation_thread);
	SC_THREAD(question_manager_task_preemption_thread);

	// Dynamic threads initialization to handle dynamic channels
	for(int i = 0; i < Ni; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&release_physical_channels_event[i]);
		string process_name("decrement_nb_physical_channels_method_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::release_physical_channels_method, this, i), process_name.c_str(), &method_options);
	}

	// Dynamic threads initialization for channel communications
	for(int i = 0; i < No; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&start_initiator_channel_event[i]);
		string process_name("Dynamic_process_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::dynamic_transfer_process_method, this, i), process_name.c_str(), &method_options);
	}

	// Dynamic threads initialization for tracing purpose: set_target_phase_idle
	for(int i = 0; i < Ni; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&set_target_phase_idle_event[i]);
		string process_name("set_target_phase_idle_process_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::set_target_phase_idle, this, i), process_name.c_str(), &method_options);
	}

	// Dynamic threads initialization for tracing purpose: set_initiator_phase_idle
	for(int i = 0; i < No; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&set_initiator_phase_idle_event[i]);
		string process_name("set_initiator_phase_idle_process_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::set_initiator_phase_idle, this, i), process_name.c_str(), &method_options);
	}

	// Dynamic threads for sending END_REQ to transaction initiator
	for(int i = 0; i < Ni; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&send_end_req_event[i]);
		string process_name("send_end_req_process_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::send_end_req_method, this, i), process_name.c_str(), &method_options);
	}


	// Dynamic threads for channel buffer updates
	for(int i = 0; i < Ni; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&update_socket_data_received_buffer_event[i]);
		string process_name("update_data_received_buffer_process_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::update_socket_data_received_buffer, this, i), process_name.c_str(), &method_options);
	}

	// Dynamic threads for channel buffer updates
	for(int i = 0; i < No; i++) {
		sc_spawn_options method_options;
		method_options.spawn_method();
		method_options.dont_initialize();
		method_options.set_sensitivity(&update_socket_data_sent_buffer_event[i]);
		string process_name("update_data_sent_buffer_process_");
		ostringstream out;
		out << i;
		process_name.append(out.str());

		sc_spawn(sc_bind(&Module<Ni, No>::update_socket_data_sent_buffer, this, i), process_name.c_str(), &method_options);
	}
}

/***********************************
 ******     Initiator part    ******
 ***********************************/

// Backward path for initiator socket
template<int Ni, int No>
tlm_sync_enum Module<Ni, No>::nb_transport_bw(int id, tlm_generic_payload& trans, tlm_phase& phase, sc_time& delay) {

	// Put the transaction into the queue
	m_id_init_map[&trans] = id;
	m_peq.notify(trans, phase, delay);

	set_initiator_phase(id, phase);

    return TLM_ACCEPTED;
}

// Payload event queue callback function
template<int Ni, int No>
void Module<Ni, No>::peq_callback(tlm_generic_payload& trans, const tlm_phase& phase) {

	int socketID = m_id_init_map[&trans];

#ifdef DEBUG
	if (phase == END_REQ)
		FOUT << sc_time_stamp() << ": " << name() << " (I" << socketID << ") received END_REQ at address " << hex << trans.get_address() << endl;
	else if (phase == BEGIN_RESP)
		FOUT << sc_time_stamp() << ": " << name() << " (I" << socketID << ") received BEGIN_RESP at address " << hex << trans.get_address() << endl;
#endif

	//if (phase == END_REQ || (&trans == outgoing_transaction[socketID] && phase == BEGIN_RESP)) {
	if (phase == END_REQ) {

		update_initiator_packet(socketID, &trans, phase);

		nb_packets_remaining--;

		if(algorithm_ended && (nb_packets_remaining == 0)) {
			FOUT << sc_time_stamp() << ": " << name()<< ": every packet transfered" << endl;
			algorithm_ended = false;
		}

		m_new_val_data_sent[socketID] = true;
		update_socket_data_sent_buffer_event[socketID].notify();

		outgoing_transaction[socketID] = 0;

		release_physical_channel(get_output_connection_name(socketID));
		shared_channel_ready_event.notify();

		set_com_status_idle(socketID);

		//set_initiator_phase_end_resp_event[socketID].notify();
		set_initiator_phase_idle(socketID);

	} else if (phase == BEGIN_RESP) {
		// The target has finished processing

		update_initiator_packet(socketID, &trans, phase);
		Utils::check_transaction(trans);

		// Send final phase transition to target
		tlm_phase fw_phase = END_RESP;
		sc_time delay = end_resp_time[socketID];
		initiator_socket[socketID]->nb_transport_fw(trans, fw_phase, delay);
		// Ignore return value

		request_in_progress[socketID] = false;

		end_of_response_channel_event[socketID].notify();

		set_initiator_phase(socketID, fw_phase);
		set_initiator_phase_idle_event[socketID].notify(delay);
	} else if (phase == BEGIN_REQ || phase == END_RESP) SC_REPORT_FATAL("TLM-2", "Illegal transaction phase received by initiator");

}

template<int Ni, int No>
void Module<Ni, No>::display_physical_channels_map(void) {
	FOUT << "Physical channel map state" << endl;
	for(map<string, int>::const_iterator it = physical_channels_map.begin(); it != physical_channels_map.end(); it++) {
		FOUT << it->first << ": " << it->second << " channel(s) free" << endl;
	}
}

template<int Ni, int No>
bool Module<Ni, No>::is_physical_channel_available(string connection_name) {

	FOUT << "Checking interface availability for connection: " << connection_name << endl;

	display_physical_channels_map();

	if(are_priority_channels_respected(module_interfaces[connection_name].get_type())) {

		FOUT << "Priority channels respected" << endl;
		if(module_interfaces[connection_name].is_priority_channel()) {
			FOUT << "Priority channel!" << endl;
			return true;
		} else {
			// Not a priority channel, do nothing
		}

	}else {
		FOUT << "Not enough physical channels to respect priority => not considering priority channels!" << endl;
	}

	return (physical_channels_map[module_interfaces[connection_name].get_type()] != 0);
}

template<int Ni, int No>
void Module<Ni, No>::release_physical_channel(string connection_name) {
	if(!are_priority_channels_respected(module_interfaces[connection_name].get_type()) || !module_interfaces[connection_name].is_priority_channel()) {
		physical_channels_map[module_interfaces[connection_name].get_type()]++;
	}

	display_physical_channels_map();
}

template<int Ni, int No>
void Module<Ni, No>::acquire_physical_channel(string connection_name) {
	if(!are_priority_channels_respected(module_interfaces[connection_name].get_type()) || !module_interfaces[connection_name].is_priority_channel()) {
		physical_channels_map[module_interfaces[connection_name].get_type()]--;
	}

	display_physical_channels_map();
}


template<int Ni, int No>
void Module<Ni, No>::dynamic_transfer_process_method(int channel_id) {

	FOUT << sc_time_stamp() << ": Module " << name() << ": Channel " << channel_id << " ready! Launching initiator" << endl;

	if(request_in_progress[channel_id]) {
		FOUT << sc_time_stamp() << ": " << name() << ": request already in progress in channel " << channel_id << ", waiting for end_of_response_channel_event" << endl;
		next_trigger(end_of_response_channel_event[channel_id]);
		return;
	}

	if(!is_physical_channel_available(get_output_connection_name(channel_id)) && !static_module) {
		FOUT << sc_time_stamp() << ": " << name() << ": no channel available at this point, waiting for shared_channel_ready" << endl;
		next_trigger(shared_channel_ready_event);
		return;
	}

	// Decrease number of available channels
	acquire_physical_channel(get_output_connection_name(channel_id));

	initiator_process(channel_id);

	next_trigger();
}

template<int Ni, int No>
void Module<Ni, No>::initiator_process(int channel_id) {

	request_in_progress[channel_id] = true;

	tlm_generic_payload* trans;
	tlm_phase phase;
	sc_time delay;
	tlm_command cmd;

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	cmd = TLM_WRITE_COMMAND;
	phase = BEGIN_REQ;

	// Set all attributes except byte_enable_length and extensions (unused)
	trans->set_command(cmd);
	trans->set_address(address_out[channel_id]);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(data_out[channel_id]));
	trans->set_data_length(4 * output_stream_packet_size[channel_id]);
	trans->set_streaming_width(4 * output_stream_packet_size[channel_id]);	// = data_length to indicate no streaming
	trans->set_byte_enable_ptr(0);								// 0 indicates unused
	trans->set_dmi_allowed(false);								// Mandatory initial value
	trans->set_response_status(TLM_INCOMPLETE_RESPONSE);		// Mandatory initial value

	outgoing_transaction[channel_id] = trans;

	// Timing annotation models processing time of initiator prior to call
	delay = begin_req_time[channel_id];

	// Non-blocking transport call on the forward path
	tlm_sync_enum status;
	status = initiator_socket[channel_id]->nb_transport_fw(*trans, phase, delay);

	// Check value returned from nb_transport_fw
	if (status == TLM_UPDATED) {
		// The timing annotation must be honored
		m_id_init_map[trans] = channel_id;
		m_peq.notify(*trans, phase, delay);
	} else if (status == TLM_COMPLETED) {
		// The completion of the transaction necessarily ends the BEGIN_REQ phase
		outgoing_transaction[channel_id] = 0;

		//physical_channels_map[module_interfaces[get_output_connection_name(channel_id)].get_type()]++;
		release_physical_channel(get_output_connection_name(channel_id));
		shared_channel_ready_event.notify();

		request_in_progress[channel_id] = false;
		// The target has terminated the transaction
		Utils::check_transaction(*trans);
	}

	update_initiator_packet(channel_id, trans);
	//set_initiator_phase_begin_req_event[channel_id].notify(delay);
	set_initiator_phase(channel_id, phase);
	//update_target_packet(channel_id);

	set_com_status_tx(channel_id);
}

template<int Ni, int No>
int Module<Ni, No>::retrieve_channel_id(int module_id) {
	int id = 0;
	while(following_modules_list.at(id) != module_id) id++;
	return id;
}


template<int Ni, int No>
string Module<Ni, No>::get_input_connection_name(int id) {
	return preceding_modules_connection_name.at(id);
}

template<int Ni, int No>
string Module<Ni, No>::get_output_connection_name(int id) {
	return following_modules_connection_name.at(id);
}



/***********************************
 ******       Target part     ******
 ***********************************/

// Forward path for the target socket
template<int Ni, int No>
tlm_sync_enum Module<Ni, No>::nb_transport_fw(int id, tlm_generic_payload& trans, tlm_phase& phase, sc_time& delay) {

	unsigned int	len = trans.get_data_length();
	unsigned char*	byt = trans.get_byte_enable_ptr();
	unsigned int	wid = trans.get_streaming_width();

	// Obliged to check the transaction attributes for unsupported features
	// and to generate the appropriate error response
	if (byt != 0) {
		trans.set_response_status(TLM_BYTE_ENABLE_ERROR_RESPONSE);
		return TLM_COMPLETED;
	}
	if (len > (unsigned int)(4 * input_stream_packet_size[id]) || wid < len) {
		trans.set_response_status(TLM_BURST_ERROR_RESPONSE);
		return TLM_COMPLETED;
	}

	// Now queue the transaction until the annotated time has elapsed
	m_id_targ_map[&trans] = id;
	m_peq_targ.notify(trans, phase, delay);

	set_target_phase(id, phase);

	return TLM_ACCEPTED;
}

// Payload event queue callback (Target side)
template<int Ni, int No>
void Module<Ni, No>::peq_targ_callback(tlm_generic_payload& trans, const tlm_phase& phase) {

	//tlm_sync_enum status;
	sc_time delay;

	int socketID = m_id_targ_map[&trans];

#ifdef DEBUG
	if(phase == BEGIN_REQ)
		FOUT << sc_time_stamp() << ": " << name() << " (T" << socketID << ") received BEGIN_REQ at address " << hex << trans.get_address() << endl;
	else if(phase == END_RESP)
		FOUT << sc_time_stamp() << ": " << name() << " (T" << socketID << ") received END_RESP at address " << hex << trans.get_address() << endl;
#endif

	switch (phase) {
	case BEGIN_REQ:

		// Increment the transaction reference count
		trans.acquire();

		if(incoming_transaction_pending[socketID] != 0) {
			send_error_response(socketID, trans);
			return;
		} else incoming_transaction_pending[socketID] = &trans;

		// Always notify the user algorithm thread
		//update_user_algorithm_event.notify();

		// Consider both priority and non-priority channels equally
		// Physical channel verification is done inside the method sensitive to the event launched
		send_end_req_event[socketID].notify();

		update_target_packet(socketID, phase);
		//set_target_phase_begin_resp_event[socketID].notify();
		set_target_phase_idle(socketID);

		break;

	case END_RESP:

		update_target_packet(socketID, phase);
		//set_target_phase_begin_resp_event[socketID].notify();
		set_target_phase_idle(socketID);

		// On receiving END_RESP, the target can release the transaction
		trans.release();
		break;

	case END_REQ:
	case BEGIN_RESP:
		SC_REPORT_FATAL("TLM-2", "Illegal transaction phase received by target");
		break;

	}
}

template<int Ni, int No>
void Module<Ni, No>::release_physical_channels_method(int channel_id) {
	release_physical_channel(get_input_connection_name(channel_id));
	shared_channel_ready_event.notify();
}

template<int Ni, int No>
void Module<Ni, No>::send_end_req_method(int id) {

	if(!is_physical_channel_available(get_input_connection_name(id)) && !static_module) {
		FOUT << sc_time_stamp() << ": " << name() << ": (T) request already in progress in shared channel, waiting for shared_channel_in_ready_event" << endl;
		next_trigger(shared_channel_ready_event);
		return;
	}

	// Decrease number of available channels
	acquire_physical_channel(get_input_connection_name(id));

	// Store transaction address and data into local memory
	address_in[id] = incoming_transaction_pending[id]->get_address();
	memcpy(data_in[id], incoming_transaction_pending[id]->get_data_ptr(), incoming_transaction_pending[id]->get_data_length());
#ifdef DEBUG
	FOUT << sc_time_stamp() << ": " << name() << " (T" << id << ") Updating address and data for socket "
		<< id << ". New address: " << address_in[id] << endl;
#endif

	m_new_val_data_received[id] = true;
	update_socket_data_received_buffer_event[id].notify(end_req_time[id]);

	send_end_req(id, *incoming_transaction_pending[id]);

	next_trigger();
}

template<int Ni, int No>
tlm_sync_enum Module<Ni, No>::send_end_req(int id, tlm_generic_payload& trans) {

	tlm_sync_enum status;
	tlm_phase bw_phase;
    sc_time delay;

	// Queue the acceptance and the response with the appropriate latency
	bw_phase = END_REQ;
	delay = end_req_time[id];
	status = target_socket[id]->nb_transport_bw(trans, bw_phase, delay);
	if (status == TLM_COMPLETED) {
		// Transaction aborted by the initiator
		// (TLM_UPDATED cannot occur at this point in the base protocol, so need not be checked)
		trans.release();
		return status;
	}

	release_physical_channels_event[id].notify(delay);

	//set_target_phase_end_req_event[id].notify(delay);
	set_target_phase(id, bw_phase);
	set_target_phase_idle_event[id].notify(delay);

	return status;
}

template<int Ni, int No>
void Module<Ni, No>::send_response(int id, tlm_generic_payload& trans) {

	tlm_sync_enum status;
	tlm_phase bw_phase;
	sc_time delay;

	bw_phase = BEGIN_RESP;
	delay = begin_resp_time[id];

	status = target_socket[id]->nb_transport_bw(trans, bw_phase, delay);

	if (status == TLM_UPDATED) {
		// The timing annotation must be honored
		m_id_targ_map[&trans] = id;
		m_peq_targ.notify(trans, bw_phase, delay);
	} else if (status == TLM_COMPLETED) {
		// The initiator has terminated the transaction
		trans.release();
	}

	//set_target_phase_begin_resp_event[id].notify(delay);
	set_target_phase(id, bw_phase);
}

template<int Ni, int No>
void Module<Ni, No>::send_error_response(int id, tlm_generic_payload& trans) {

	tlm_sync_enum status;
	tlm_phase bw_phase;
	sc_time delay;

	bw_phase = BEGIN_RESP;
	delay = begin_resp_time[id];

	trans.set_response_status(TLM_COMMAND_ERROR_RESPONSE);
	status = target_socket[id]->nb_transport_bw(trans, bw_phase, delay);

	if (status == TLM_UPDATED) {
		// The timing annotation must be honored
		m_id_targ_map[&trans] = id;
		m_peq_targ.notify(trans, bw_phase, delay);
	} else if (status == TLM_COMPLETED) {
		// The initiator has terminated the transaction
		trans.release();
	}

	//set_target_phase_begin_resp_event[id].notify(delay);
	set_target_phase(id, bw_phase);
}


/***********************************
 ******  Configuration part   ******
 ***********************************/
template<int Ni, int No>
void Module<Ni, No>::b_transport_conf(tlm_generic_payload& trans, sc_time& delay) {

	if(trans.get_address() == MANAGER_COMMAND_SCHEDULING) {
		/* Scheduling */
		int data = *(reinterpret_cast<int *>(trans.get_data_ptr()));
		User_algorithm_action action = (User_algorithm_action) (data & 0xffff);
		int module_ready = data >> 16;

		FOUT << sc_time_stamp() << ": " << name() << ": Module " << module_ready << " ready" << endl;

		// If next module ready, launch initiator
		if(action == NEXT_MODULE_READY) {

			for(int i = 0; i < (int) channels_required[module_ready].size(); i++) {
				FOUT << sc_time_stamp() << ": " << name() << ": Channel " << channels_required[module_ready].at(i) << " ready" << endl;
				start_initiator_channel_event[channels_required[module_ready].at(i)].notify();
			}

			channels_required[module_ready].clear();

		} else if(action == CURRENT_MODULE_READY) {
			// Preemption point ready to be returned
			continue_execution_event.notify();
		}

		// Wait, but delay should be 0
		//wait(delay);
	}
	else if(trans.get_address() == MANAGER_UPDATE_PARAMETER_COMMAND_SCHEDULING) {
		/* Parameters updated : RZ */
		Algorithm_Parameters* ap = (reinterpret_cast<Algorithm_Parameters *>(trans.get_data_ptr()));

		FOUT << sc_time_stamp() << ": " << name() << ": updated parameters of RZ : " << ap->getCurrentRZ().getName()  << ", pt = " << ap->getCurrentRZ().getConfigFctPoint().toString() << endl;

		parameter_computation_data = ap;
		parameter_computation_event.notify();
	}

	trans.set_response_status(TLM_OK_RESPONSE);
}

template<int Ni, int No>
void Module<Ni, No>::question_manager_task_end_computation_thread(void) {
	while(true) {
		wait(question_manager_task_end_computation_event);
		question_manager_b_transport(MANAGER_UPLINK_TASK_END_COMPUTATION, 0);
	}
}

template<int Ni, int No>
void Module<Ni, No>::question_manager_task_preemption_thread(void) {
	while(true) {
		wait(question_manager_task_preemption_event);
		question_manager_b_transport(MANAGER_UPLINK_TASK_PREEMPTION, 0);
	}
}

template<int Ni, int No>
void Module<Ni, No>::question_manager_b_transport(int address, int data) {

	switch(address) {
		case MANAGER_UPLINK_TASK_END_COMPUTATION: FOUT << sc_time_stamp() << ": " << name() << " sending a request to the manager using the uplink (end computation)" << endl; break;
		case MANAGER_UPLINK_TASK_PREEMPTION: FOUT << sc_time_stamp() << ": " << name() << " sending a request to the manager using the uplink (preemption point)" << endl; break;
		case MANAGER_UPLINK_TASK_CHECK_SOCKET: FOUT << sc_time_stamp() << ": " << name() << " sending a request to the manager using the uplink (check socket)" << endl; break;
		case MANAGER_UPLINK_ALGO_STATE_UPDATE: FOUT << sc_time_stamp() << ": " << name() << " sending a request to the manager using the uplink (algo update)" << endl; break;
		case MANAGER_UPLINK_UPDATE_PARAMETER_COMMAND_SCHEDULING: FOUT << sc_time_stamp() << ": " << name() << " sending a request to the manager using the uplink (scheduling/algo update)" << endl; break;
		default: FOUT << sc_time_stamp() << ": " << name() << " cannot recognize address for manager b_transport (" << address << ")" << endl; break;
	}

	tlm_generic_payload* trans;
	sc_time delay(SC_ZERO_TIME);
	int local_data = data;

	// Grab a new transaction from the memory manager
	trans = m_mm.allocate();
	trans->acquire();

	// Set all attributes except byte_enable_length and extensions (unused)
	trans->set_command(TLM_READ_COMMAND);
	trans->set_address(address);
	trans->set_data_ptr(reinterpret_cast<unsigned char*>(&local_data));
	trans->set_data_length(4);
	trans->set_streaming_width(4);							// = data_length to indicate no streaming
	trans->set_byte_enable_ptr(0);							// 0 indicates unused
	trans->set_dmi_allowed(false);							// Mandatory initial value
	trans->set_response_status(TLM_INCOMPLETE_RESPONSE);	// Mandatory initial value

	manager_initiator_socket->b_transport(*trans, delay);

	if(trans->is_response_error()) SC_REPORT_ERROR("TLM-2", "Response error from manager b_transport");

	if(local_data == (int) NEXT_MODULE_READY) {
		if(address == MANAGER_UPLINK_TASK_CHECK_SOCKET) {

			for(int i = 0; i < (int) channels_required[data].size(); i++) {
				FOUT << sc_time_stamp() << ": " << name() << ": Channel " << channels_required[data].at(i) << " ready" << endl;
				start_initiator_channel_event[channels_required[data].at(i)].notify();
			}

			channels_required[data].clear();

		}
	} else if(local_data == (int) CURRENT_MODULE_READY) {
		// Task preempted, but ready to continue its execution
		continue_execution_event.notify();
	}

	trans->release();
}

/********
 * MISC *
 *******/
template<int Ni, int No>
void Module<Ni, No>::info(void) {

	FOUT << endl << "Module " << name();
	if(static_module) FOUT << " (static)";
	FOUT << ":"<< endl;

	for(int i = 0; i < No; i++) {
		FOUT << "Output socket " << i << " (Connection " << following_modules_connection_name.at(i) << "):" << endl;
		if(module_interfaces[following_modules_connection_name.at(i)].is_priority_channel()) FOUT << "  Priority channel" << endl;
		FOUT << "  Packet size: " << output_stream_packet_size[i] << endl;
		FOUT << "  BEGIN_REQ time: " << begin_req_time[i] << endl;
		FOUT << "  END_RESP time: " << end_resp_time[i] << endl;
	}

	for(int i = 0; i < Ni; i++) {
		FOUT << "Input socket " << i << " (Connection " << preceding_modules_connection_name.at(i) << "):" << endl;
		if(module_interfaces[preceding_modules_connection_name.at(i)].is_priority_channel()) FOUT << "  Priority channel" << endl;
		FOUT << "  Packet size: " << input_stream_packet_size[i] << endl;
		FOUT << "  END_REQ time: " << end_req_time[i] << endl;
		FOUT << "  BEGIN_RESP time: " << begin_resp_time[i] << endl;
	}

	FOUT << "Deadline time: " << deadline_time << endl;
	FOUT << "Following list: ";
	for(int i = 0; i < (int) following_modules_list.size(); i++) FOUT << dec << following_modules_list.at(i) << " ";
	FOUT << endl << "Preceding list: ";
	for(int i = 0; i < (int) preceding_modules_list.size(); i++) FOUT << dec << preceding_modules_list.at(i) << " ";
	FOUT << endl;
	FOUT << "Algorithm implementations:" << endl;
	for(int i = 0; i < (int) implementation_execution_times_vector.size(); i++) implementation_execution_times_vector.at(i).info((*fout));
}

template<int Ni, int No>
void Module<Ni, No>::updateLists(int offset) {
	application_offset = offset;
	for(int j = 0; j < (int) following_modules_list.size(); j++) {
		if(following_modules_list.at(j) != -1) following_modules_list[j] += offset;
	}
	for(int j = 0; j < (int) channel_connection_vector.size(); j++) {
		if(channel_connection_vector.at(j) != -1) channel_connection_vector[j] += offset;
	}
	for(int j = 0; j < (int) preceding_modules_list.size(); j++) {
		if(preceding_modules_list.at(j) != -1) preceding_modules_list[j] += offset;
	}
}

template<int Ni, int No>
void Module<Ni, No>::verify_bindings(void) {
	if(target_socket.size() != Ni || initiator_socket.size() != No || manager_target_socket.size() != 1 || manager_initiator_socket.size() != 1) {
		cerr << "ERROR: Incorrect bindings for module " << name() << "! Exiting..." << endl;
		exit(-1);
	}
}

template<int Ni, int No>
void Module<Ni, No>::enable_configuration_signals_trace(void) {
	trace_packet.trace_configuration_signals = true;
}

template<int Ni, int No>
void Module<Ni, No>::activate_trace(void) {
	sc_trace(tf, trace_packet, (string) name());
}

template<int Ni, int No>
void Module<Ni, No>::set_implementation_vector(vector<ModuleImplementation>& impl) {
	implementation_execution_times_vector = impl;
}

template<int Ni, int No>
vector<ModuleImplementation>* Module<Ni, No>::get_implementation_vector_ptr(void) {
	return &implementation_execution_times_vector;
}

template<int Ni, int No>
void Module<Ni, No>::set_static_module(void) {
	static_module = true;
	//cout << "Before addition" << endl;
	//set_current_implementation(implementation_execution_times_vector.at(0).get_name());
	//set_current_implementation("Hard1");
	//cout << "After addition" << endl;
}


/************************************
 * User_algorithm_interface methods *
 ************************************/
template<int Ni, int No>
const char* Module<Ni, No>::get_name() const {
	return name();
}

template<int Ni, int No>
string Module<Ni, No>::get_top_application_name(void) const {
    if (applicationName.find("/") != string::npos)
        return (applicationName.substr(applicationName.find("/") + 1)).c_str();
    else
        return applicationName;
}


template<int Ni, int No>
int* Module<Ni, No>::get_data_in_ptr(int socket) {
	return data_in[socket];
}

template<int Ni, int No>
int* Module<Ni, No>::get_data_in_ptr(string name) {
	// Retrieve socket ID
	int id = get_input_socket_id(name);
	if(id == -1) cerr << "ERROR: Unable to retrieve input socket ID from connection name" << endl;

	return data_in[id];
}

template<int Ni, int No>
int* Module<Ni, No>::get_data_out_ptr(int socket) {
	return data_out[socket];
}

template<int Ni, int No>
int* Module<Ni, No>::get_data_out_ptr(string name) {
	// Retrieve socket ID
	int id = get_output_socket_id(name);
	if(id == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;

	return data_out[id];
}

template<int Ni, int No>
int Module<Ni, No>::get_data_in_length(int id) const {
	return input_stream_packet_size[id] * 4; // in bytes
}

template<int Ni, int No>
int Module<Ni, No>::get_data_in_length(string name) const {

	// Retrieve socket ID
	int id = get_input_socket_id(name);
	if(id == -1) cerr << "ERROR: Unable to retrieve input socket ID from connection name" << endl;

	return input_stream_packet_size[id] * 4; // in bytes
}

template<int Ni, int No>
int Module<Ni, No>::get_data_out_length(int id) const {
	return output_stream_packet_size[id] * 4; // in bytes
}

template<int Ni, int No>
int Module<Ni, No>::get_data_out_length(string name) const {

	// Retrieve socket ID
	int id = get_output_socket_id(name);
	if(id == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;

	return output_stream_packet_size[id] * 4; // in bytes
}

template<int Ni, int No>
sc_dt::uint64 Module<Ni, No>::get_address_in(int socket) const {
	return address_in[socket];
}

template<int Ni, int No>
sc_dt::uint64 Module<Ni, No>::get_address_in(string name) const {
	// Retrieve socket ID
	int id = get_input_socket_id(name);
	if(id == -1) cerr << "ERROR: Unable to retrieve input socket ID from connection name" << endl;

	return address_in[id];
}

template<int Ni, int No>
void Module<Ni, No>::set_address_out(int socket, sc_dt::uint64 val) {
	address_out[socket] = val;
}

template<int Ni, int No>
void Module<Ni, No>::set_address_out(string name, sc_dt::uint64 val) {
	// Retrieve socket ID
	int id = get_output_socket_id(name);
	if(id == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;

	address_out[id] = val;
}

template<int Ni, int No>
int Module<Ni, No>::get_input_socket_id(string connection_name) const {
	int socketID = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}

	return socketID;
}

template<int Ni, int No>
int Module<Ni, No>::get_output_socket_id(string connection_name) const {
	int socketID = -1;
	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
		if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
			socketID = i;
			break;
		}
	}

	return socketID;
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_WCET(void) const {
	return implementation_execution_times_vector.at(current_implementation_id).get_worst_case_execution_time();
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_BCET(void) const {
	return implementation_execution_times_vector.at(current_implementation_id).get_best_case_execution_time();;
}

template<int Ni, int No>
void Module<Ni, No>::end_of_algorithm(void) {

	FOUT << sc_time_stamp() << ": Module " << name() << ": User process finished" << endl;

	// Question the manager about scheduling
	question_manager_task_end_computation_event.notify();

	transactions_processed++;

	// Sending response to all preceding modules
	//for(int i = 0; i < Ni; i++) {

	//	// Checking for transient connections
	//	bool transient = (module_interfaces[preceding_modules_connection_name.at(i)].is_transient());
	//	if(!(first_transaction && transient)) {
	//		incoming_transaction[i] = incoming_transaction_pending[i];
	//		incoming_transaction_pending[i] = 0;
	//		incoming_transaction[i]->set_response_status(TLM_OK_RESPONSE);
	//		send_response(i, *incoming_transaction[i]);
	//	} else FOUT << sc_time_stamp() << ": " << name() << ": Socket " << i << " (" << preceding_modules_connection_name.at(i) << ") is transient! Do not notify module" << endl;
	//}

	first_transaction = false;

	// After algorithm execution, reset boolean value to false
	kill_user_algorithm_thread = false;

	/*for(int i = 0; i < Ni; i++) {
		m_new_val_data_received[i] = false;
		update_socket_data_received_buffer_event[i].notify();
	}*/

	algorithm_ended = true;

	nb_algorithm_executions++;

	algorithm_execution_requested = false;

}

template<int Ni, int No>
void Module<Ni, No>::preemption_point(void) {

	if(!static_module) {

		question_manager_task_preemption_event.notify();

		// Wait for the manager to restart task
		wait(continue_execution_event);
	}
}

template<int Ni, int No>
void Module<Ni, No>::compute(sc_time duration) {

	if (duration == SC_ZERO_TIME) {
		wait(SC_ZERO_TIME);
		return;
	}

	if(!static_module) {

		float coeff, previous_coeff;
		Config_fct_point fctPt;
		Config_fct_point nominalFctPt;

		//cout << sc_time_stamp() << " Request parameters for init " << endl;

		//cout << sc_time_stamp() << " ****** START : " << duration << " ( " << get_name() << " )" << endl;

		bool first = true;
		parameter_computation_data = NULL;
		request_updated_parameters();

		sc_time ref_time = sc_time_stamp();
		sc_time current_duration = SC_ZERO_TIME;	// first : parameter_computation_event received

		do {
			//cout << sc_time_stamp() << " Wait Request parameters for init " << endl;
			wait (current_duration, parameter_computation_event);

			//cout << sc_time_stamp() << " OK Request parameters for init " << endl;
			if (parameter_computation_data != NULL) {

				fctPt = parameter_computation_data->getCurrentRZ().getConfigFctPoint();
				nominalFctPt = parameter_computation_data->getRZConfig().getConfigDomain()->getNominalFctPoint();

				if (nominalFctPt.getFrequency() != 0.0 && nominalFctPt.getFrequency() != 0.0) {

					coeff = nominalFctPt.getFrequency() / fctPt.getFrequency();
					FOUT << sc_time_stamp() << " Acc Coeff : " << coeff  << endl;
				}
				else {

					coeff = 1.0F;
					FOUT << sc_time_stamp() << "Warning: Frequency of " << parameter_computation_data->getCurrentRZ().getName() << " is 0 MHz." << endl;
				}

				// Stop Event sended by scheduler
				sc_time remaining_time;
				if (first == true) {
					remaining_time = duration;
				}
				else
					remaining_time = (current_duration - (sc_time_stamp() - ref_time)) / previous_coeff; // FAUX : !!! 3 ms - 9 ms !!!

				// If preemption (but here, just compute again exe time)
				// Send Premption Order to scheduler (Task sets to PREEMPTED State) and scheduler recalls
				//question_manager_task_preemption_event.notify();
				// Resume Event sended by scheduler
				//wait(continue_execution_event);
				/*
				cout << sc_time_stamp() << " ****** duration : " << duration << " ( " << get_name() << " )" << endl;
				cout << sc_time_stamp() << " sc_time_stamp : " << sc_time_stamp() << endl;
				cout << sc_time_stamp() << " ref_time : " << ref_time << endl;
				cout << sc_time_stamp() << " sc_time_stamp - ref_time : " << (sc_time_stamp() - ref_time) << endl;
				cout << sc_time_stamp() << " previous current_duration : " << current_duration <<endl;
				*/
				current_duration = remaining_time * coeff;

				//cout << sc_time_stamp() << " COEFF : " << coeff << ", remaining time : " << remaining_time << ", current_duration : " << current_duration <<endl;

				ref_time = sc_time_stamp();
				previous_coeff = coeff;

				first = false;
				// delete and Set NULL (commsuming ok)
				delete parameter_computation_data;
				parameter_computation_data = NULL;
			}
			else {
				// Exit
				current_duration = sc_time(0, SC_NS);

				if (first == true) {
					FOUT << sc_time_stamp() << " WARNING: No DVFS used !" << endl;
					wait (duration);
				}

			}

		} while (current_duration > sc_time(0, SC_NS));

	}
}

template<int Ni, int No>
const sc_event& Module<Ni, No>::update_user_algorithm(void) const {
	return update_user_algorithm_event;
}

template<int Ni, int No>
void Module<Ni, No>::wait_for_update_user_algorithm(void) {
	// Makes no sense when considering a static module (or several accelerators implemented...)
	if(!static_module) wait(update_user_algorithm());
}

template<int Ni, int No>
bool& Module<Ni, No>::kill_user_algorithm(void) {
	return kill_user_algorithm_thread;
}

// DATA RECEIVE
template<int Ni, int No>
void Module<Ni, No>::b_data_received(string connection_name) {

	// Retrieve socket ID
	int socketID = get_input_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve input socket ID from connection name" << endl;

	b_data_received(socketID);

}

template<int Ni, int No>
void Module<Ni, No>::b_data_received(int socketID) {

	string connection_name(preceding_modules_connection_name.at(socketID));

	if(module_interfaces[connection_name].is_transient() && first_transaction) {
		FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << preceding_modules_connection_name.at(socketID) << ") is transient!" << endl;
	} else if(socket_data_received[socketID].read()) {
		FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << preceding_modules_connection_name.at(socketID) << ") is ready!" << endl;
	} else {
		FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << preceding_modules_connection_name.at(socketID) << ") is not ready yet..." << endl;
		while(!socket_data_received[socketID].read()) wait(socket_data_received[socketID].value_changed_event());
		FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << preceding_modules_connection_name.at(socketID) << ") is now ready!" << endl;
	}

}

template<int Ni, int No>
void Module<Ni, No>::b_all_data_received(void) {
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		b_data_received(preceding_modules_connection_name.at(i));
	}
}

template<int Ni, int No>
bool Module<Ni, No>::nb_data_received(string connection_name) {

	// Retrieve socket ID
	int socketID = get_input_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve input socket ID from connection name" << endl;

	return (nb_data_received(socketID));
}

template<int Ni, int No>
bool Module<Ni, No>::nb_data_received(int socketID) {

	return (socket_data_received[socketID].read());
}

template<int Ni, int No>
bool Module<Ni, No>::nb_all_data_received(void) {
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(!nb_data_received(preceding_modules_connection_name.at(i))) return false;
	}

	return true;
}

// DATA SEND
template<int Ni, int No>
void Module<Ni, No>::nb_send_data(string connection_name) {

	// Retrieve socket ID
	int socketID = get_output_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;
	FOUT << "Socket ID: " << socketID << endl;

	nb_send_data(socketID);
}

template<int Ni, int No>
void Module<Ni, No>::nb_send_data(int socketID) {

	//// Reset buffer value => moved to start_new_transaction_sequence
	//m_new_val_data_sent[socketID] = false;
	//update_socket_data_sent_buffer_event[socketID].notify();

	first_data_send_on_socket[socketID] = false;

	// Retrieve module ID
	int module_id = channel_connection_vector.at(socketID);
	FOUT << "Module ID: " << module_id << endl;

	bool first_module_request = (channels_required[module_id].empty());
	channels_required[module_id].push_back(socketID);

	nb_packets_remaining++;

	if(first_module_request) question_manager_b_transport(MANAGER_UPLINK_TASK_CHECK_SOCKET, module_id);

	set_com_status_waiting(socketID);
}

template<int Ni, int No>
void Module<Ni, No>::nb_send_all_data(void) {
	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
		nb_send_data(following_modules_connection_name.at(i));
	}
}

template<int Ni, int No>
void Module<Ni, No>::b_send_data(string connection_name) {

	// Retrieve socket ID
	int socketID = get_output_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;
	FOUT << "Socket ID: " << socketID << endl;

	b_send_data(socketID);
}

template<int Ni, int No>
void Module<Ni, No>::b_send_data(int socketID) {

	nb_send_data(socketID);

	FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << following_modules_connection_name.at(socketID) << ") is not ready yet..." << endl;
	while(!socket_data_sent[socketID].read()) wait(socket_data_sent[socketID].value_changed_event());
	FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << following_modules_connection_name.at(socketID) << ") is now ready!" << endl;
}

template<int Ni, int No>
void Module<Ni, No>::b_send_all_data(void) {

	for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
		b_send_data(following_modules_connection_name.at(i));
	}
}

template<int Ni, int No>
void Module<Ni, No>::b_data_sent(int socketID) {
	if(!first_data_send_on_socket[socketID]) {
		while(!socket_data_sent[socketID].read()) wait(socket_data_sent[socketID].value_changed_event());
	}
}

template<int Ni, int No>
bool Module<Ni, No>::nb_data_sent(int socketID) {
	return (first_data_send_on_socket[socketID] || socket_data_sent[socketID].read());
}

template<int Ni, int No>
void Module<Ni, No>::b_data_sent(string connection_name) {

	// Retrieve socket ID
	int socketID = get_output_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;
	FOUT << "Socket ID: " << socketID << endl;

	b_data_sent(socketID);
}

template<int Ni, int No>
bool Module<Ni, No>::nb_data_sent(string connection_name) {

	// Retrieve socket ID
	int socketID = get_output_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;
	FOUT << "Socket ID: " << socketID << endl;

	return nb_data_sent(socketID);
}

template<int Ni, int No>
void Module<Ni, No>::b_all_data_sent(void) {
	for(int i = 0; i < No; i++) b_data_sent(i);
}

template<int Ni, int No>
bool Module<Ni, No>::nb_all_data_sent(void) {
	for(int i = 0; i < No; i++) {
		if(!nb_data_sent(i)) return false;
	}

	return true;
}

template<int Ni, int No>
void Module<Ni, No>::start_new_transaction_sequence(void) {
	for(int i = 0; i < No; i++) start_new_transaction_sequence(i);
}

template<int Ni, int No>
void Module<Ni, No>::start_new_transaction_sequence(int socketID) {
	// Reset buffer value
	m_new_val_data_sent[socketID] = false;
	update_socket_data_sent_buffer_event[socketID].notify();
}

template<int Ni, int No>
void Module<Ni, No>::start_new_transaction_sequence(string connection_name) {
	// Retrieve socket ID
	int socketID = get_output_socket_id(connection_name);
	if(socketID == -1) cerr << "ERROR: Unable to retrieve output socket ID from connection name" << endl;
	FOUT << "Socket ID: " << socketID << endl;

	start_new_transaction_sequence(socketID);
}



template<int Ni, int No>
bool Module<Ni, No>::is_channel_transient(string connection_name) {
	return (module_interfaces[connection_name].is_transient());
}

template<int Ni, int No>
bool Module<Ni, No>::is_channel_transient(int socketID) {
	return (module_interfaces[preceding_modules_connection_name.at(socketID)].is_transient());
}

template<int Ni, int No>
int Module<Ni, No>::get_nb_preemption_points(void) {
	return (implementation_execution_times_vector.at(current_implementation_id).get_nb_preemption_points());
}

template<int Ni, int No>
void Module<Ni, No>::set_algorithm_running(void) {
	question_manager_b_transport(MANAGER_UPLINK_ALGO_STATE_UPDATE, (int) RUNNING);
}

template<int Ni, int No>
void Module<Ni, No>::set_algorithm_idle(void) {
	question_manager_b_transport(MANAGER_UPLINK_ALGO_STATE_UPDATE, (int) IDLE);
}

template<int Ni, int No>
void Module<Ni, No>::set_algorithm_waiting(void) {
	question_manager_b_transport(MANAGER_UPLINK_ALGO_STATE_UPDATE, (int) WAITING);
}

template<int Ni, int No>
void Module<Ni, No>::request_updated_parameters(void) {
	// toto
	question_manager_b_transport(MANAGER_UPLINK_UPDATE_PARAMETER_COMMAND_SCHEDULING, 0);
}

template<int Ni, int No>
void Module<Ni, No>::wait_until_next_period(void) {
	if(nb_algorithm_executions == 0) {
		if(task_offset != SC_ZERO_TIME) {
			sc_time time_to_offset = task_offset - sc_time_stamp();
			FOUT << sc_time_stamp() << ": " << name() << ": Time to wait before offset is verified: " << time_to_offset << endl;
			if(time_to_offset > SC_ZERO_TIME) wait(time_to_offset);
			last_absolute_deadline_time = absolute_deadline_time;
		}
	} else {
		if(task_period != SC_ZERO_TIME) {
			sc_time time_to_period = nb_algorithm_executions * task_period - sc_time_stamp();
			FOUT << sc_time_stamp() << ": " << name() << ": Time to wait before beginning of next period: " << time_to_period << endl;
			if(time_to_period > SC_ZERO_TIME) wait(time_to_period);
			last_absolute_deadline_time = absolute_deadline_time;
		}
	}
}

template<int Ni, int No>
ofstream& Module<Ni, No>::get_logfile(void) {
	return (*fout);
}

template<int Ni, int No>
string Module<Ni, No>::get_algorithm_execution_mode(void) {
	return algorithm_execution_mode;
}

template<int Ni, int No>
bool Module<Ni, No>::is_algorithm_execution_mode(string mode) {
	return (algorithm_execution_mode.compare(mode) == 0);
}

template<int Ni, int No>
void Module<Ni, No>::release_input_socket(int socketID) {

	// Checking for transient connections
	bool transient = (module_interfaces[preceding_modules_connection_name.at(socketID)].is_transient());
	if(!(first_transaction && transient)) {
		incoming_transaction[socketID] = incoming_transaction_pending[socketID];
		incoming_transaction_pending[socketID] = 0;
		incoming_transaction[socketID]->set_response_status(TLM_OK_RESPONSE);
		send_response(socketID, *incoming_transaction[socketID]);
	} else FOUT << sc_time_stamp() << ": " << name() << ": Socket " << socketID << " (" << preceding_modules_connection_name.at(socketID) << ") is transient! Do not notify module" << endl;

	m_new_val_data_received[socketID] = false;
	update_socket_data_received_buffer_event[socketID].notify();
}

template<int Ni, int No>
void Module<Ni, No>::release_input_socket(string connectionName) {

	int socketID = get_input_socket_id(connectionName);
	if(socketID == -1) {
		cerr << name() << ": Unable to retrieve socket ID from connection name (" << connectionName << "). Exiting" << endl;
		exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
	}

	release_input_socket(socketID);
}

template<int Ni, int No>
void Module<Ni, No>::release_all_input_sockets(void) {
	for(int i = 0; i < Ni; i++) release_input_socket(i);
}

template<int Ni, int No>
void Module<Ni, No>::b_execution_requested(void) {
	if(!static_module) {
		if(!algorithm_execution_requested) wait(start_algorithm_event);
	}
}


/************************************
 *  Reconf_thread_interface methods *
 ************************************/
template<int Ni, int No>
const sc_event& Module<Ni, No>::configuration_updated() const {
	return configuration_updated_event;
}

template<int Ni, int No>
void Module<Ni, No>::finish_configuration(void) {
	configuration_done_event.notify();
}

template<int Ni, int No>
int Module<Ni, No>::get_current_implementation_id(void) {
	return current_implementation_id;
}

template<int Ni, int No>
vector<ModuleImplementation> Module<Ni, No>::get_implementation_vector(void) {
	return implementation_execution_times_vector;
}


/************************************
 *     Manager_interface methods    *
 ************************************/

template<int Ni, int No>
bool Module<Ni, No>::has_implementation(TaskImplementation& impl) const {
	return (getName().compare(impl.getTaskName()) == 0);
}

template<int Ni, int No>
void Module<Ni, No>::set_current_implementation(string implementationName) {

	int old_id = current_implementation_id;

	current_implementation_id = -1;
	for(int i = 0; i < (int) implementation_execution_times_vector.size(); i++) {
		if(implementation_execution_times_vector.at(i).get_name().compare(implementationName) == 0) {
			/*if(worst_case) computation_time = implementation_execution_times_vector.at(i).get_worst_case_execution_time();
			else computation_time = implementation_execution_times_vector.at(i).get_best_case_execution_time();*/
			current_implementation_id = i;
			FOUT << sc_time_stamp() << ": Module " << name() << ": Changing current implementation to " << implementationName << " (" << current_implementation_id << ")" << endl;
			break;
		}
	}

	if(current_implementation_id == -1) cerr << "ERROR: Implementation not found!" << endl;

	// Notify the reconfiguration control thread only if a new thread should be created
	if(current_implementation_id != old_id) {
		FOUT << sc_time_stamp() << ": " << name() << "Sending request to change conf" << endl;
		kill_user_algorithm_thread = true;
		configuration_updated_event.notify();
	} else {
		kill_user_algorithm_thread = false;
	}

	// Update physical channel map with algorithm data
	physical_channels_map.clear();
	map<string, int> channels(implementation_execution_times_vector.at(current_implementation_id).get_physical_channel_map());

	// Consider each channel type defined for the module. If not available in the implementation, use module values
	for(map<string, ModuleInterface>::iterator moduleInterface_it = module_interfaces.begin(); moduleInterface_it != module_interfaces.end(); moduleInterface_it++) {

		// First, check if the interface has been defined for the implementation
		bool interface_defined = false;
		for(map<string, int>::const_iterator it = channels.begin(); it != channels.end(); it++) {

			if(it->first.compare(moduleInterface_it->second.get_type()) == 0) {
				FOUT << "Updating channel resources for interface " << it->first << " with value " << it->second << endl;
				physical_channels_map.insert(pair<string, int>(it->first, it->second));
				interface_defined = true;
				break;
			}
		}

		if(!interface_defined) {

			FOUT << "Updating channel resources for interface " << moduleInterface_it->second.get_type() << " with module value" << endl;

			if(physical_channels_map.find(moduleInterface_it->first) != physical_channels_map.end()) {
				if(moduleInterface_it->second.is_priority_channel()) physical_channels_map[moduleInterface_it->second.get_type()]++;
			} else {
				if(moduleInterface_it->second.is_priority_channel()) physical_channels_map.insert(pair<string, int>(moduleInterface_it->second.get_type(), 2));
				else physical_channels_map.insert(pair<string, int>(moduleInterface_it->second.get_type(), 1));
			}
		}
	}

	display_physical_channels_map();

	remove_priority_channels_from_physical_channel_map();

	display_physical_channels_map();
}

template<int Ni, int No>
void Module<Ni, No>::remove_priority_channels_from_physical_channel_map(void) {

	FOUT << "Removing priority channels from map" << endl;
	for(map<string, int>::iterator it = physical_channels_map.begin(); it != physical_channels_map.end(); it++) {
		if(are_priority_channels_respected(it->first)) {
			// Count the number of priority channels and remove it from map
			int nbPriorityChannels = 0;
			for(map<string, ModuleInterface>::iterator modIT = module_interfaces.begin(); modIT != module_interfaces.end(); modIT++) {
				if(modIT->second.get_type().compare(it->first) == 0 && modIT->second.is_priority_channel()) nbPriorityChannels++;
			}

			physical_channels_map[it->first] -= nbPriorityChannels;
		}
	}
}

template<int Ni, int No>
bool Module<Ni, No>::are_priority_channels_respected(string interface_type) {

	// Check if priority channels apply here (i.e. nbIF > nb priority channels + 1)
	int nbIF = implementation_execution_times_vector.at(current_implementation_id).get_physical_channel_map()[interface_type];
	FOUT << "Number of " << interface_type << " interfaces provided by the implementation: " << nbIF << endl;

	int nbPriorityChannels = 0;
	for(map<string, ModuleInterface>::iterator it = module_interfaces.begin(); it != module_interfaces.end(); it++) {
		if(it->second.get_type().compare(interface_type) == 0 && it->second.is_priority_channel()) nbPriorityChannels++;
	}
	FOUT << "Number of " << interface_type << " priority channels: " << nbPriorityChannels << endl;

	if(nbPriorityChannels == module_interfaces.size()) return (nbIF >= nbPriorityChannels);
	else return (nbIF > nbPriorityChannels);
}

template<int Ni, int No>
vector<ModuleImplementation> Module<Ni, No>::get_implementation_vector(void) const {
	return implementation_execution_times_vector;
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_deadline_time(void) const {
	return absolute_deadline_time;
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_task_deadline(void) const {
	return deadline_time;
}

template<int Ni, int No>
void Module<Ni, No>::set_request_time(void) {
	if(task_period == SC_ZERO_TIME) absolute_deadline_time = task_offset + sc_time_stamp() + deadline_time;
	else absolute_deadline_time = task_offset + nb_algorithm_executions * task_period + deadline_time;
}

template<int Ni, int No>
bool Module<Ni, No>::fits(RZ *rz) {

	// Check if the task is in the RZ list of compatibility
	vector<TaskImplementation> taskVector = rz->getTaskImplementationVector();

	for(int i = 0; i < (int) taskVector.size(); i++) {
		if(taskVector.at(i).getTaskName().compare(name()) == 0) return true;
	}

	return false;
}

template<int Ni, int No>
string Module<Ni, No>::getName(void) const {
	return name();
}

template<int Ni, int No>
int Module<Ni, No>::get_nb_processed_transactions(void) const {
	return transactions_processed;
}

template<int Ni, int No>
vector<int> Module<Ni, No>::getFollowingList(void) const {
	return following_modules_list;
}

template<int Ni, int No>
vector<int> Module<Ni, No>::getPrecedingList(void) const {
	return preceding_modules_list;
}

template<int Ni, int No>
bool Module<Ni, No>::is_static(void) const {
	return static_module;
}

template<int Ni, int No>
int Module<Ni, No>::get_task_priority(void) const {
	return priority;
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_task_period(void) const {
	return task_period;
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_task_offset(void) const {
	return task_offset;
}

template<int Ni, int No>
string Module<Ni, No>::get_current_implementation_fullname(void) {
	return (name() + (string)"." + implementation_execution_times_vector.at(current_implementation_id).get_name());
}

template<int Ni, int No>
bool Module<Ni, No>::has_current_implementation(void) {
	return current_implementation_id != -1;
}

template<int Ni, int No>
ModuleImplementation Module<Ni, No>::get_current_implementation(void) {
	return implementation_execution_times_vector.at(current_implementation_id);
}

template<int Ni, int No>
void Module<Ni, No>::update_algorithm_execution_mode(string command) {
	algorithm_execution_mode = command;
	trace_update_algorithm_execution_mode();
}

template<int Ni, int No>
bool Module<Ni, No>::use_context_switch_mode(string implementationName) {
	// Look through module implementation vector to find the good one
	for(int i = 0; i < (int) implementation_execution_times_vector.size(); i++) {
		if(implementation_execution_times_vector.at(i).get_name().compare(implementationName) == 0) return implementation_execution_times_vector.at(i).use_context_switch_mode();
	}

	cerr << name() << ": Unable to retrieve implementation by its name... Implementation name: " << implementationName << ". Exiting" << endl;
	exit(RECOSIM_INTERNAL_ERROR_ERRCODE);
}

template<int Ni, int No>
void Module<Ni, No>::send_start_algorithm_event(void) {

	// Always notify the user algorithm thread
	update_user_algorithm_event.notify();

	FOUT << sc_time_stamp() << ": " << name() << " send start algorithm request" << endl;
	algorithm_execution_requested = true;
	start_algorithm_event.notify(SC_ZERO_TIME);
}

template<int Ni, int No>
sc_time Module<Ni, No>::get_new_period_start_time(void) const {
	sc_time new_period_start_time;
	if(nb_algorithm_executions == 0) new_period_start_time = (task_offset > sc_time_stamp()) ? task_offset : SC_ZERO_TIME;
	else new_period_start_time = ((nb_algorithm_executions * task_period) > sc_time_stamp()) ? nb_algorithm_executions * task_period : SC_ZERO_TIME;

	return new_period_start_time;
}

template<int Ni, int No>
int Module<Ni, No>::get_nb_transient_channels(void) {
	int nb_transient = 0;
	for(int i = 0; i < Ni; i++) {
		if(is_channel_transient(i)) nb_transient++;
	}

	return nb_transient;
}

template<int Ni, int No>
void Module<Ni, No>::set_task_priority(int p) {
	priority = p;
}






// Data stream configuration
template<int Ni, int No>
void Module<Ni, No>::set_packet_size(string connection_name, int size) {

	// First, find if target module is a following or a preceding module
	bool preceding_mod = false;
	int socketID = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			preceding_mod = true;
			socketID = i;
			break;
		}
	}

	if(!preceding_mod) {
		for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
			if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
				socketID = i;
				break;
			}
		}
	}

#ifdef CONSOLE_DEBUG
	if(preceding_mod) cout << "Preceding module" << endl;
	else cout << "Following module" << endl;
	cout << "Socket ID: " << socketID << endl;
#endif

	try {
		if(preceding_mod) {
			assert(socketID < Ni);
			input_stream_packet_size[socketID] = size;
			delete[] data_in[socketID];
			data_in[socketID] = 0;
			data_in[socketID] = new int[input_stream_packet_size[socketID]];
			for(int i = 0; i < input_stream_packet_size[socketID]; i++) data_in[socketID][i] = 0;
		} else {
			assert(socketID < No);
			output_stream_packet_size[socketID] = size;
			delete[] data_out[socketID];
			data_out[socketID] = 0;
			data_out[socketID] = new int[output_stream_packet_size[socketID]];
			for(int i = 0; i < output_stream_packet_size[socketID]; i++) data_out[socketID][i] = 0;
		}
	} catch (bad_alloc& ba) {
		cerr << "ERROR: " << name() << " bad_alloc caught: " << ba.what() << endl;
		//exit(-1);
		Simulation_controller::endSimulation();
	}

}

// Timing
template<int Ni, int No>
void Module<Ni, No>::set_deadline_time(sc_time t) {
	deadline_time = t;
}

template<int Ni, int No>
void Module<Ni, No>::set_begin_req_time(string connection_name, sc_time t) {
	add_timing_info(BEGIN_REQ, connection_name, t);
}

template<int Ni, int No>
void Module<Ni, No>::set_end_req_time(string connection_name, sc_time t) {
	add_timing_info(END_REQ, connection_name, t);
}

template<int Ni, int No>
void Module<Ni, No>::set_begin_resp_time(string connection_name, sc_time t) {
	add_timing_info(BEGIN_RESP, connection_name, t);
}

template<int Ni, int No>
void Module<Ni, No>::set_end_resp_time(string connection_name, sc_time t) {
	add_timing_info(END_RESP, connection_name, t);
}

template<int Ni, int No>
void Module<Ni, No>::add_timing_info(tlm_phase_enum phase, string connection_name, sc_time time) {

	// First, find if target module is a following or a preceding module
	bool preceding_mod = false;
	int socketID = -1;
	for(int i = 0; i < (int) preceding_modules_connection_name.size(); i++) {
		if(preceding_modules_connection_name.at(i).compare(connection_name) == 0) {
			preceding_mod = true;
			socketID = i;
			break;
		}
	}

	if(!preceding_mod) {
		for(int i = 0; i < (int) following_modules_connection_name.size(); i++) {
			if(following_modules_connection_name.at(i).compare(connection_name) == 0) {
				socketID = i;
				break;
			}
		}
	}

#ifdef CONSOLE_DEBUG
	if(preceding_mod) cout << "Preceding module" << endl;
	else cout << "Following module" << endl;
	cout << "Socket ID: " << socketID << endl;
#endif


	switch(phase) {
		case BEGIN_REQ: begin_req_time[socketID] = time; break;
		case END_REQ: end_req_time[socketID] = time; break;
		case BEGIN_RESP: begin_resp_time[socketID] = time; break;
		case END_RESP: end_resp_time[socketID] = time; break;
		default: cerr << "ERROR: Phase not recognized!" << endl;
	}
}

// Next/Previous modules
template<int Ni, int No>
void Module<Ni, No>::set_preceding_modules_list(vector<int> &v) {
	preceding_modules_list = v;
}

template<int Ni, int No>
void Module<Ni, No>::set_following_modules_list(vector<int> &v) {

	channel_connection_vector = v;
	following_modules_list.clear();

	// Check for redundancies for following modules list
	for(int i = 0; i < (int) channel_connection_vector.size(); i++) {
		bool item_found = false;
		for(int j = 0; j < (int) following_modules_list.size(); j++) {
			if(following_modules_list.at(j) == channel_connection_vector.at(i)) item_found = true;
		}
		if(!item_found) following_modules_list.push_back(channel_connection_vector.at(i));
	}
}

template<int Ni, int No>
void Module<Ni, No>::add_following_module_connection_name(string name) {
	following_modules_connection_name.push_back(name);
}

template<int Ni, int No>
void Module<Ni, No>::add_preceding_module_connection_name(string name) {
	preceding_modules_connection_name.push_back(name);
}

template<int Ni, int No>
void Module<Ni, No>::set_priority(int v) {
	priority = v;
}

template<int Ni, int No>
void Module<Ni, No>::add_interface(ModuleInterface mod_if) {
	module_interfaces.insert(pair<string, ModuleInterface>(mod_if.get_name(), mod_if));
}

template<int Ni, int No>
void Module<Ni, No>::set_task_period(sc_time p) {
	task_period = p;
}

template<int Ni, int No>
void Module<Ni, No>::set_task_offset(sc_time o) {
	task_offset = o;
}

/**********************************
 *      Packet trace methods      *
 **********************************/
template<int Ni, int No>
void Module<Ni, No>::init_packet(void) {
	trace_packet.algorithm_execution_mode = Utils::string_to_ascii(algorithm_execution_mode);
	for(int i = 0; i < Ni; i++) update_target_packet(i, 0);
	for(int i = 0; i < No; i++) {
		update_initiator_packet(i, 0, 0);
		trace_packet.communication_status[i] = IDLE_ASCII;
	}
}

template<int Ni, int No>
void Module<Ni, No>::update_trace_packet(void) {
	for(int i = 0; i < Ni; i++) {
		trace_packet.address_in[i] = (int) address_in[i];
		trace_packet.data_in[i] = data_in[i][0];
	}

	for(int i = 0; i < No; i++) {
		trace_packet.address_out[i] = (int) address_out[i];
		trace_packet.data_out[i] = data_out[i][0];
	}
}

template<int Ni, int No>
void Module<Ni, No>::update_initiator_packet(int id, tlm_generic_payload* trans, tlm_phase phase) {
	assert(id < No);
	trace_packet.initiator_packet[id].phase = tlm_phase_to_ascii(phase);
	update_initiator_packet(id, trans);
}

template<int Ni, int No>
void Module<Ni, No>::update_initiator_packet(int id, tlm_generic_payload* trans) {
	assert(id < No);
	update_trace_packet();
	trace_packet.initiator_packet[id].transaction.update_transaction(trans);
}

template<int Ni, int No>
void Module<Ni, No>::update_target_packet(int id, tlm_phase phase) {
	assert(id < Ni);
	update_target_packet(id);
	trace_packet.target_packet[id].phase = tlm_phase_to_ascii(phase);
}

template<int Ni, int No>
void Module<Ni, No>::update_target_packet(int id) {
	assert(id < Ni);
	update_trace_packet();
	trace_packet.target_packet[id].transaction.update_transaction(incoming_transaction[id]);
	trace_packet.target_packet[id].transaction_pending.update_transaction(incoming_transaction_pending[id]);
}

template<int Ni, int No>
void Module<Ni, No>::set_target_phase_idle(int id) {
	trace_packet.target_packet[id].phase = IDLE_ASCII;
}

template<int Ni, int No>
void Module<Ni, No>::set_target_phase(int id, tlm_phase& phase) {
	assert(id < Ni);
	trace_packet.target_packet[id].phase = tlm_phase_to_ascii(phase);
}

template<int Ni, int No>
void Module<Ni, No>::set_initiator_phase(int id, tlm_phase& phase) {
	assert(id < No);
	trace_packet.initiator_packet[id].phase = tlm_phase_to_ascii(phase);
}

template<int Ni, int No>
void Module<Ni, No>::set_initiator_phase_idle(int id) {
	trace_packet.initiator_packet[id].phase = IDLE_ASCII;
}

template<int Ni, int No>
void Module<Ni, No>::set_config_phase(tlm_phase& phase) {
	trace_packet.configuration_packet.phase = tlm_phase_to_ascii(phase);
}

template<int Ni, int No>
void Module<Ni, No>::set_com_status_waiting(int id) {
	assert(id < No);
	trace_packet.communication_status[id] = COM_WAITING_T_ASCII;
}

template<int Ni, int No>
void Module<Ni, No>::set_com_status_tx(int id) {
	assert(id < No);
	trace_packet.communication_status[id] = COM_TX_T_ASCII;
}

template<int Ni, int No>
void Module<Ni, No>::set_com_status_idle(int id) {
	assert(id < No);
	trace_packet.communication_status[id] = IDLE_ASCII;
}

template<int Ni, int No>
void Module<Ni, No>::trace_update_algorithm_execution_mode(void) {
	trace_packet.algorithm_execution_mode = Utils::string_to_ascii(algorithm_execution_mode);
}

#endif
