/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    application.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-08-30
 * @section DESCRIPTION
 *			Application representation
 */

#ifndef APPLICATION_H
#define APPLICATION_H

#define SC_INCLUDE_DYNAMIC_PROCESSES

#include <systemc.h>
#include <string>
#include <vector>
#include "manager_interface.h"
#include "application_interface.h"
//#include <list>

using namespace std;
using namespace sc_core;
using namespace sc_dt;

class Application : public Application_interface {

private:
	string name;
	string prefix;

	double qosFactor;
	int firstTaskID;
	int lastTaskID;
	int instanceID;

	int nbPassedDeadlines;
	int nbMissedDeadlines;
	int nbPassedDeadlinesOnHyperperiod;
	int nbMissedDeadlinesOnHyperperiod;

	sc_time achievedSimulationTime;

	sc_time offset;
	sc_time hyperperiod;

	bool status;

	vector<pair<string, sc_time> > missed_deadlines;
	vector<pair<string, sc_time> > missed_deadlines_last_hyperperiod;

	bool qosFailure;

	static sc_time minimumSimulationTime;
	static sc_time globalOffset;
	static sc_time globalHyperperiod;

	static sc_time userOffset;
	static sc_time userHyperperiod;
	static bool overrideOffset;
	static bool overrideHyperperiod;

	double minimalInstantQoS;

public:
	Application(string _name, string _prefix, int _instanceID, double qos, int start, int stop) :  name(_name), prefix(_prefix), instanceID(_instanceID), qosFactor(qos), firstTaskID(start), lastTaskID(stop) {
		nbPassedDeadlines = 0;
		nbMissedDeadlines = 0;
		nbPassedDeadlinesOnHyperperiod = 0;
		nbMissedDeadlinesOnHyperperiod = 0;
		offset = SC_ZERO_TIME;
		hyperperiod = SC_ZERO_TIME;
		status = false;
		qosFailure = false;

		minimalInstantQoS = 100.0;
	}

	Application(void) {}

	// Application_interface

	string getName(void);
	bool containsTask(int taskID);
	int getFirstTaskID();
	int getTaskNumber();
	



	//double getQoSFactor(void);
	double getAchievedQoS(void);
	double getMinimalQoS(void);

	string getFullModuleName(string moduleName);

	void incrementPassedDeadlines(void);
	void incrementMissedDeadlines(string taskname);

	void computeMinimumSimulationTime(vector<Manager_interface *>& table);

	void setAchievedSimulationTime(sc_time t);
	sc_time getAchievedSimulationTime(void);
	bool isSimulationTimeEnough(void);

	bool getStatus(void);
	void setStatus(bool);

	bool isQoSSatisfied(void);

	void checkQoSOnHyperperiod(void);

	

	vector<pair<string, sc_time> >& getMissedDeadlinesVector(void);
	bool hasMissedDeadlines(void);

	

	// Static members
	static sc_time getGlobalHypeperiod(void);
	static sc_time getGlobalOffset(void);
	static sc_time getMinimumSimulationTime(void);
	static void computeMinimumSimulationTime(vector<Application>&);

	static void setOffset(sc_time);
	static void setHyperperiod(sc_time);

private:
	sc_time getOffset(void);
	sc_time getHyperperiod(void);

	static void computeGlobalHypeperiod(vector<Application>&);
	static void computeGlobalOffset(vector<Application>&);

};

#endif
