/**
 * UNS - CNRS
 * Copyright 2013 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit� de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: licensing@sattse.com
 *
 * @file    device.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0-build:2014/06/04
 * @date    January 24, 2016 3:10:00 PM CET
 * @section DESCRIPTION
 *
 *          Some defines representing the hardware in xc6vlx240t device (Virtex6)
*/



#ifndef DEVICE_H
#define DEVICE_H

#include <string>
#include <iostream>
#include <set>
#include <map>
#include <vector>

#include "coordinates.h"
#include "cell.h"

using namespace std;

/** Enumeration of the supported devices */
enum DeviceClass{UNDEF, Virtex5, Virtex6, Virtex7, Kintex7, Zynq7, Artix7, CUSTOM};
enum PositionConstraint{BEGIN, END, BOTH};

enum RBType{EMAC, IOB, PCI, CLBl, CLBm, BRAM, Slice, BRAM_XXX, DSP_XXX, DSP, CLB_XXX, SliceM, MMCM, SliceL, EMPTY};

const Cell EMAC_cell("EMAC", false, 0, 0, 0, false);
const Cell IOB_cell("IOB", true, 40, 100, 11808, false);
const Cell PCI_cell("PCI", false, 0, 0, 0, false);
const Cell CLBl_cell("CLBl", true, 40, 1, 11808, true);
const Cell CLBm_cell("CLBm", true, 40, 4, 11808, true);
const Cell BRAM_cell("BRAM", true, 8, 24, 51824, false);
const Cell Slice_cell("Slice", true, 40, 1, 5904, true);
const Cell BRAM_XXX_cell("BRAM_XXX", false, 0, 0, 0, false);
const Cell DSP_XXX_cell("DSP_XXX", false, 0, 0, 0, false);
const Cell DSP_cell("DSP", true, 16, 24, 9840, false);
const Cell CLB_XXX_cell("CLB_XXX", false, 0, 0, 0, false);
const Cell SliceM_cell("SliceM", true, 40, 4, 5904, true);
const Cell MMCM_cell("MMCM", false, 0, 0, 0, false);
const Cell SliceL_cell("SliceL", true, 40, 1, 5904, true);
const Cell EMPTY_cell("EMPTY", false, 0, 0, 0, false);

const int NB_LINES_DEVICE   = 6;
const int NB_COLUMNS_DEVICE = 102;

const RBType fpga_device[NB_LINES_DEVICE][NB_COLUMNS_DEVICE] = {
{IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,MMCM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,CLBl,CLBm,CLBl,BRAM},
{IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,MMCM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,PCI,PCI,PCI,PCI},
{IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,CLB_XXX,CLB_XXX,CLB_XXX,CLB_XXX,CLB_XXX,CLB_XXX,MMCM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,CLBl,CLBm,CLBl,EMAC},
{IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,CLB_XXX,CLB_XXX,CLB_XXX,CLB_XXX,CLB_XXX,CLB_XXX,MMCM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,PCI,PCI,PCI,PCI},
{IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,MMCM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,CLBl,CLBm,CLBl,EMAC},
{IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,CLBl,MMCM,CLBm,CLBl,CLBm,CLBl,IOB,CLBm,CLBl,CLBm,CLBl,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,CLBm,BRAM,CLBm,CLBm,DSP,CLBm,CLBm,CLBm,CLBm,DSP,CLBm,CLBm,BRAM,CLBm,CLBl,CLBm,CLBl,CLBl,CLBm,CLBl,BRAM}
};

const int x_coordinate_fpga_device[NB_LINES_DEVICE][NB_COLUMNS_DEVICE] = {
{0,0,2,4,6,0,8,10,0,12,14,16,18,1,20,22,1,24,26,28,30,32,34,36,38,2,40,42,2,44,46,48,50,3,52,54,3,56,58,60,62,1,64,66,68,70,72,74,76,78,80,82,0,84,86,88,90,2,92,94,96,98,4,100,102,4,104,106,108,110,5,112,114,5,116,118,120,122,124,126,128,130,6,132,134,6,136,138,140,142,7,144,146,7,148,150,152,154,156,158,160,8},
{0,0,2,4,6,0,8,10,0,12,14,16,18,1,20,22,1,24,26,28,30,32,34,36,38,2,40,42,2,44,46,48,50,3,52,54,3,56,58,60,62,1,64,66,68,70,72,74,76,78,80,82,0,84,86,88,90,2,92,94,96,98,4,100,102,4,104,106,108,110,5,112,114,5,116,118,120,122,124,126,128,130,6,132,134,6,136,138,140,142,7,144,146,7,148,150,152,154,0,1,2,3},
{0,0,2,4,6,0,8,10,0,12,14,16,18,1,20,22,1,24,26,28,30,32,34,36,38,2,40,42,2,44,46,48,50,3,52,54,3,56,58,60,62,1,64,66,68,70,36,37,38,39,40,41,0,84,86,88,90,2,92,94,96,98,4,100,102,4,104,106,108,110,5,112,114,5,116,118,120,122,124,126,128,130,6,132,134,6,136,138,140,142,7,144,146,7,148,150,152,154,156,158,160,0},
{0,0,2,4,6,0,8,10,0,12,14,16,18,1,20,22,1,24,26,28,30,32,34,36,38,2,40,42,2,44,46,48,50,3,52,54,3,56,58,60,62,1,64,66,68,70,36,37,38,39,40,41,0,84,86,88,90,2,92,94,96,98,4,100,102,4,104,106,108,110,5,112,114,5,116,118,120,122,124,126,128,130,6,132,134,6,136,138,140,142,7,144,146,7,148,150,152,154,0,1,2,3},
{0,0,2,4,6,0,8,10,0,12,14,16,18,1,20,22,1,24,26,28,30,32,34,36,38,2,40,42,2,44,46,48,50,3,52,54,3,56,58,60,62,1,64,66,68,70,72,74,76,78,80,82,0,84,86,88,90,2,92,94,96,98,4,100,102,4,104,106,108,110,5,112,114,5,116,118,120,122,124,126,128,130,6,132,134,6,136,138,140,142,7,144,146,7,148,150,152,154,156,158,160,0},
{0,0,2,4,6,0,8,10,0,12,14,16,18,1,20,22,1,24,26,28,30,32,34,36,38,2,40,42,2,44,46,48,50,3,52,54,3,56,58,60,62,1,64,66,68,70,72,74,76,78,80,82,0,84,86,88,90,2,92,94,96,98,4,100,102,4,104,106,108,110,5,112,114,5,116,118,120,122,124,126,128,130,6,132,134,6,136,138,140,142,7,144,146,7,148,150,152,154,156,158,160,8}
};

const PositionConstraint columns_position_constraints[NB_COLUMNS_DEVICE] = {
BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH, BOTH};

const int NB_LUT_SLICE      = 4;
const int NB_FF_SLICE       = 8;


class Device {

private:
	/* Device identification */
	string deviceID;
	string package;
	string speedGrade;
	DeviceClass deviceClass;

	/* Device description */
	RBType **RBTable;
	int nb_lines;
	int nb_columns;

	/* Map to retrieve a Cell from a RBType */
	map<RBType, Cell> deviceCellMap;
	
	/* Location restrictions */
	vector<Coordinates> restrictedColumns;

	/* Ressources: TO BE REMOVED LATER */
	int nb_LUT_Slice;
	int nb_FF_Slice;
	
public:
	/**
	 * Device Constructor. RBTable is an array of pointers, each
	 * pointing to the corresponding row in the FPGA representation
	 * array 
	 */
	Device() {
		deviceClass = Virtex6;
		deviceID = "xc6vlx240t";
		package = "ff1156";
		speedGrade = "-1";

		
		nb_lines = NB_LINES_DEVICE;
		nb_columns = NB_COLUMNS_DEVICE;
		RBTable = new RBType*[nb_lines];
		for(int i = 0; i < nb_lines; i++) RBTable[i] = (RBType *) fpga_device[i];
		/* Cell map init */
		deviceCellMap[EMAC] = EMAC_cell;
		deviceCellMap[IOB] = IOB_cell;
		deviceCellMap[PCI] = PCI_cell;
		deviceCellMap[CLBl] = CLBl_cell;
		deviceCellMap[CLBm] = CLBm_cell;
		deviceCellMap[BRAM] = BRAM_cell;
		deviceCellMap[Slice] = Slice_cell;
		deviceCellMap[BRAM_XXX] = BRAM_XXX_cell;
		deviceCellMap[DSP_XXX] = DSP_XXX_cell;
		deviceCellMap[DSP] = DSP_cell;
		deviceCellMap[CLB_XXX] = CLB_XXX_cell;
		deviceCellMap[SliceM] = SliceM_cell;
		deviceCellMap[MMCM] = MMCM_cell;
		deviceCellMap[SliceL] = SliceL_cell;
		deviceCellMap[EMPTY] = EMPTY_cell;

		
		/* Ressources: TO BE REMOVED LATER */
		nb_LUT_Slice = NB_LUT_SLICE;
		nb_FF_Slice = NB_FF_SLICE;
		
		/* FPGA Static Constraints */
				// Empty

	};

	~Device() {
		delete[] RBTable;
	}

	/* Getters */
	int getNbLines(void);
	int getNbColumns(void);
	string getDeviceID(void);
	string getDevicePackage(void);
	string getDeviceSpeedGrade(void);
	DeviceClass getDeviceClass(void);

	/* Cost getter */
	int getResourceCost(RBType rbtype);
	int getResourceCount(RBType rbt);

	/* RB/Cell methods */
	RBType getRB(int line, int column) const;
	RBType getColumnRBType(int column);
	bool isRBReconfigurable(RBType rbt);
	Cell getCell(RBType);
	string getCellName(RBType);
	RBType getRBType(string) const;
	map<RBType, Cell> getDeviceCellMap(void);

	/* LOC constraints */
	bool isColumnUsable(int line, int column);
	bool canColumnBeginRZ(int column);
	bool canColumnEndRZ(int column);

	/* UCF generation */
	int getPhysicalXCoordinate(int line, int column);

	/* Misc */
	void printDevice(void);


	/* TO BE REMOVED LATER: Techno-dependent stuff */
	int getNbLUTInSlice(void);
	int getNbFFInSlice(void);
};

#endif
