/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    taskResourceFile.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2013-01-10
 * @section DESCRIPTION
 *			Task resource file (either .syr or XML)
 */

#ifndef TASK_RESOURCE_FILE_H
#define TASK_RESOURCE_FILE_H

#include <string>
#include "device.h"

class TaskResourceFile {

protected:

	static double STATIC_RESOURCES_MARGIN;		/*< Resources margin to take into account RZ routing */

	string name;								/*< Synthesis report name, without extension */
	string deviceID;							/*< Device ID */
	DeviceClass deviceClass;					/*< Device class */
	Device *device;								/*< Device pointer */
	map<RBType, int> physicalResources;			/*< Resources inferred from the report */
	map<RBType, int> staticResources;			/*< Static Resources inferred from the report */

	double resourceMargin;						/*< Might override static value */


public:

	TaskResourceFile(std::string n) : name(n), deviceID("undef"), deviceClass(UNDEF), resourceMargin(-1), device() {}
	TaskResourceFile() : device() {}

	TaskResourceFile(const TaskResourceFile& other) {
		name = other.name;
		deviceID = other.deviceID;
		deviceClass = other.deviceClass;
		device = new Device();		// FIXME
		physicalResources = other.physicalResources;
		staticResources = other.staticResources;
		resourceMargin = other.resourceMargin;
	}

	~TaskResourceFile() {
		if(device) delete device;
	}

	// Getters
	string getName(void);
	string getDeviceID(void);
	DeviceClass getDeviceClass(void);
	Device* getDevicePtr(void);
	map<RBType, int> getPhysicalResources(void);
	map<RBType, int> getStaticResources(void);

	// Resource margin
	static double getStaticResourceMargin(void);
	static void setStaticResourcesMargin(double);
	void setResourcesMargin(double);

	// Misc
	void info(void);
	
	// Pure virtual method to accomodate the kind of resource file to parse (.syr, .tsk)
	virtual int readResourceFile(void) = 0;

protected:
	void setDeviceID(string dev);
	void inferDeviceClass(void);

	void convertResourceMap(map<RBType, int>&);
	void convertAllResourceMaps(void);
};

#endif
