/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    synthesisReport.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-04-19
 * @section DESCRIPTION
 *			Class representing a synthesis report, used for task description
 */

#include "synthesisReport.h"

#include <fstream>
#include <math.h>
#include <time.h>
#include "utils.h"

const string SynthesisReport::STRING_SELECTED_DEVICE	= "Selected Device : ";
const string SynthesisReport::STRING_NB_SLICE_REGISTERS = " Number of Slice Registers: ";
const string SynthesisReport::STRING_NB_SLICE_LUTS		= " Number of Slice LUTs: ";
const string SynthesisReport::STRING_NB_SLICE_LUTS_MEM	= "    Number used as Memory: ";
const string SynthesisReport::STRING_NB_DSP48			= " Number of DSP48E1s: ";
const string SynthesisReport::STRING_NB_RAMB36			= " Number of Block RAM/FIFO: ";
const string SynthesisReport::STRING_NB_IOB				= " Number of bonded IOBs: ";

//#define SYNTHESIS_REPORT_DEBUG


/**
 * This function reads a synthesis report (.syr) from the path as given
 * in report name. The filename is the Task name, automatically appending
 * the .syr extension if needed.
 * The function retrieves information from the device utilization
 * summary, like the number of slice registers needed. It also retrieves
 * the targetted device.
 *
 * @return -1 if an error occured, 0 otherwise
 */
int SynthesisReport::readSynthesisReport(void) {

	string filename(name);
	if(name.find(".syr") == string::npos) filename.append(".syr");	// Add extension if needed
	ifstream file(filename.c_str());

	if(file) {
		cout << endl << "-----------------------------------------------------------" << endl;
		cout << "Parsing file " << filename << endl;
		if(resourceMargin == -1) cout << "Resource margin is set to " << STATIC_RESOURCES_MARGIN << "%" << endl;
		else cout << "Resource margin is set to " << resourceMargin << "%" << endl;

		// Search for the Device Utilization Summary
		string line_read;
		int line_number = 0;
		int nb_luts_i = 0;
		int nb_regs_i = 0;

		while(getline(file, line_read)) {
			line_number++;

			/* Selected device */
 			if(line_read.find(STRING_SELECTED_DEVICE) != string::npos) {
				string device_id = line_read.substr(STRING_SELECTED_DEVICE.size());
				device_id.erase(device_id.size() - 1, device_id.size());		// Erase last space in the string

#ifdef SYNTHESIS_REPORT_DEBUG
				cout << "\tFound selected device at line " << line_number << ": " << device_id << endl;
#endif

				if(deviceID.compare("undef") == 0) setDeviceID(device_id);
				else {
					// Device already initialised
					if(deviceID.compare(device_id) != 0) {
						cerr << endl << "Device already initialized to a different value (" << deviceID << "), exiting..." << endl;
						return -1;
					}
				}
			}

			/* Number of slice LUTS used as memory (SliceM only) */
			if(line_read.find(STRING_NB_SLICE_LUTS_MEM) != string::npos) {
				string str_nb_lut = line_read.substr(STRING_NB_SLICE_LUTS_MEM.size(), line_read.find("out") - STRING_NB_SLICE_LUTS_MEM.size());
				int nb_luts_mem = Utils::atoi(str_nb_lut);
				if(device->getCell(SliceM).isSentitiveToResourceMargin()) {
					if(resourceMargin == -1) nb_luts_mem += (int) ((double)nb_luts_mem * STATIC_RESOURCES_MARGIN / 100.0);
					else nb_luts_mem += (int) ((double)nb_luts_mem * resourceMargin / 100.0);
				}
				physicalResources.insert(pair<RBType,int>(SliceM, (int) ceil((double) nb_luts_mem/(device->getNbLUTInSlice() * device->getCell(CLBl).getNbCellsPerColumn()))));
				staticResources.insert(pair<RBType,int>(SliceM, (int) ceil((double) nb_luts_mem/device->getNbLUTInSlice())));

#ifdef SYNTHESIS_REPORT_DEBUG
				cout << "\tFound number of slice LUTs used as memory at line " << line_number << ": " << nb_luts_mem << endl;
#endif
			}

			/* Number of slice registers */
			if(line_read.find(STRING_NB_SLICE_REGISTERS) != string::npos) {
				string str_nb_reg = line_read.substr(STRING_NB_SLICE_REGISTERS.size(), line_read.find("out") - STRING_NB_SLICE_REGISTERS.size());
				nb_regs_i = Utils::atoi(str_nb_reg);
				if(device->getCell(Slice).isSentitiveToResourceMargin()) {
					if(resourceMargin == -1) nb_regs_i += (int) ((double)nb_regs_i * STATIC_RESOURCES_MARGIN / 100.0);
					else nb_regs_i += (int) ((double)nb_regs_i * resourceMargin / 100.0);
				}

#ifdef SYNTHESIS_REPORT_DEBUG
				cout << "\tFound number of slice registers at line " << line_number << ": " << nb_regs_i << endl;
#endif
			}

			/* Number of slice LUTS */
			if(line_read.find(STRING_NB_SLICE_LUTS) != string::npos) {
				string str_nb_lut = line_read.substr(STRING_NB_SLICE_LUTS.size(), line_read.find("out") - STRING_NB_SLICE_LUTS.size());
				nb_luts_i = Utils::atoi(str_nb_lut);
				if(device->getCell(Slice).isSentitiveToResourceMargin()) {
					if(resourceMargin == -1) nb_luts_i += (int) ((double)nb_luts_i * STATIC_RESOURCES_MARGIN / 100.0);
					else nb_luts_i += (int) ((double)nb_luts_i * resourceMargin / 100.0);
				}

				/* Since the number of registers is already known, infer number of slices */
				physicalResources.insert(pair<RBType,int>(Slice, max((int) ceil((double) nb_luts_i/(device->getNbLUTInSlice() * device->getCell(CLBl).getNbCellsPerColumn())),
																	(int) ceil((double) nb_regs_i/(device->getNbFFInSlice() * device->getCell(CLBl).getNbCellsPerColumn())))));

				staticResources.insert(pair<RBType,int>(Slice, max((int) ceil((double) nb_luts_i/device->getNbLUTInSlice()),
																	(int) ceil((double) nb_regs_i/device->getNbFFInSlice()))));

#ifdef SYNTHESIS_REPORT_DEBUG
				cout << "\tFound number of slice LUTs at line " << line_number << ": " << nb_luts_i << endl;
#endif
			}

			/* Number of RAMB36 */
			if(line_read.find(STRING_NB_RAMB36) != string::npos) {
				string str_nb_bram = line_read.substr(STRING_NB_RAMB36.size(), line_read.find("out") - STRING_NB_RAMB36.size());
				int nb_bram_i = Utils::atoi(str_nb_bram);
				if(device->getCell(BRAM).isSentitiveToResourceMargin()) {
					if(resourceMargin == -1) nb_bram_i += (int) ((double)nb_bram_i * STATIC_RESOURCES_MARGIN / 100.0);
					else nb_bram_i += (int) ((double)nb_bram_i * resourceMargin / 100.0);
				}
				physicalResources.insert(pair<RBType,int>(BRAM, (int) ceil((double) nb_bram_i/device->getCell(BRAM).getNbCellsPerColumn())));
				staticResources.insert(pair<RBType,int>(BRAM, nb_bram_i));

#ifdef SYNTHESIS_REPORT_DEBUG
				cout << "\tFound number of RAMB36 at line " << line_number << ": " << nb_bram_i << endl;
#endif
			}

			/* Number of DSP48 */
			if(line_read.find(STRING_NB_DSP48) != string::npos) {
				// Get number of DSP48
				string str_nb_dsp = line_read.substr(STRING_NB_DSP48.size(), line_read.find("out") - STRING_NB_DSP48.size());
				int nb_dsp_i = Utils::atoi(str_nb_dsp);
				if(device->getCell(DSP).isSentitiveToResourceMargin()) {
					if(resourceMargin == -1) nb_dsp_i += (int) ((double)nb_dsp_i * STATIC_RESOURCES_MARGIN / 100.0);
					else nb_dsp_i += (int) ((double)nb_dsp_i * resourceMargin / 100.0);
				}
				physicalResources.insert(pair<RBType,int>(DSP, (int) ceil((double) nb_dsp_i/device->getCell(DSP).getNbCellsPerColumn())));
				staticResources.insert(pair<RBType,int>(DSP, nb_dsp_i));

#ifdef SYNTHESIS_REPORT_DEBUG
				cout << "\tFound number of DSP48 at line " << line_number << ": " << nb_dsp_i << endl;
#endif
			}

			/* Number of IOBs */
			if(line_read.find(STRING_NB_IOB) != string::npos) {
				// Get number of IOB
				string str_nb_iob = line_read.substr(STRING_NB_IOB.size(), line_read.find("out") - STRING_NB_IOB.size());
				int nb_iob_i = Utils::atoi(str_nb_iob);
				if(device->getCell(IOB).isSentitiveToResourceMargin()) {
					if(resourceMargin == -1) nb_iob_i += (int) ((double)nb_iob_i * STATIC_RESOURCES_MARGIN / 100.0);
					else nb_iob_i += (int) ((double)nb_iob_i * resourceMargin / 100.0);
				}

				if(nb_iob_i != 0) {
					physicalResources.insert(pair<RBType,int>(IOB, (int) ceil((double) nb_iob_i/device->getCell(IOB).getNbCellsPerColumn())));
					staticResources.insert(pair<RBType,int>(IOB, nb_iob_i));

#ifdef SYNTHESIS_REPORT_DEBUG
					cout << "\tFound number of IOBs at line " << line_number << ": " << nb_iob_i << endl;
#endif
				}
			}
        }

		file.close();

	} else {
		cerr << "ERROR: Could not open file: " << filename << endl;
		return -1;
	}

	return 0;
}

int SynthesisReport::readResourceFile(void) {
	return readSynthesisReport();
}
