/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    device.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-10-04
 * @section DESCRIPTION
 *			Simple cartesian coordinates describing constraints
 */

#include "device.h"
#include <cstdlib>

/* Getters */

int Device::getNbLines(void) {
	return nb_lines;
}

int Device::getNbColumns(void) {
	return nb_columns;
}

string Device::getDeviceID(void) {
	return deviceID;
}

string Device::getDevicePackage(void) {
	return package;
}

string Device::getDeviceSpeedGrade(void) {
	return speedGrade;
}

DeviceClass Device::getDeviceClass(void) {
	return deviceClass;
}


/* Cost getter */
int Device::getResourceCost(RBType rbtype) {
	return deviceCellMap[rbtype].getCellCost();
}

/**
 * Return the total count of given resources
 * (reconfigurable or not) inside the device
 */
int Device::getResourceCount(RBType rbt) {

	int count = 0;

	for(int i = 0; i < nb_lines; i++) {
		for(int j = 0; j < nb_columns; j++) {
			if(RBTable[i][j] == rbt) count++;
		}
	}

	return count;
}


/* RB/Cell methods */

RBType Device::getRB(int line, int column) const {
	return RBTable[line][column];
}

RBType Device::getColumnRBType(int column) {
	return RBTable[0][column];
}

bool Device::isRBReconfigurable(RBType rbt) {
	return (deviceCellMap[rbt].isReconfigurable());
}

Cell Device::getCell(RBType rbt) {
	return deviceCellMap[rbt];
}

string Device::getCellName(RBType rbt) {
	return deviceCellMap[rbt].getName();
}

RBType Device::getRBType(string rbname) const {
	for(map<RBType, Cell>::const_iterator it = deviceCellMap.begin(); it != deviceCellMap.end(); it++) {
		if(it->second.getName().compare(rbname) == 0) return it->first;
	}

	cerr << "ERROR: Unable to retrieve RBType from its name (" << rbname << ") !" << endl;
	exit(-1);
}

bool Device::isColumnUsable(int line, int column) {

	for(int i = 0; i < (int) restrictedColumns.size(); i++) {
		if(restrictedColumns.at(i) == Coordinates(column, line)) return false;
	}

	return true;
}

bool Device::canColumnBeginRZ(int column) {
	return (columns_position_constraints[column] == BEGIN || columns_position_constraints[column] == BOTH);
}

bool Device::canColumnEndRZ(int column) {
	return (columns_position_constraints[column] == END || columns_position_constraints[column] == BOTH);
}

/**
 * Retrieves the physical X coordinate for a given column.
 */
int Device::getPhysicalXCoordinate(int line, int column) {
	return x_coordinate_fpga_device[line][column];
}

map<RBType, Cell> Device::getDeviceCellMap(void) {
	return deviceCellMap;
}

/* Misc */
void Device::printDevice(void) {
	cout << "ColumnsNumber : " << nb_columns << ", " << "LinesNumber : " << nb_lines << endl;
	for (int i = 0; i < nb_lines; i++) {
		for (int j = 0; j < nb_columns; j++) cout << RBTable[i][j] << " ";
		cout << endl;
	}
}


/* TO BE REMOVED LATER: Techno-dependent stuff */

int Device::getNbLUTInSlice(void) {
	return nb_LUT_Slice;
}

int Device::getNbFFInSlice(void) {
	return nb_FF_Slice;
}
