/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    RZ.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-08-18
 * @section DESCRIPTION
 *			Representation of a Reconfigurable Zone (RZ) for RZ generation.
 *			Coordinates represent a particulat resources, i.e. if a RZ is
 *			defined by coordinates {(0, 0), (0, 0)}, it is only composed of
 *			the resources located at (0, 0) in the FPGA model. If it is
 *			defined by {(1, 2), (3, 4)}:
 *
 *				x-----|-----|-----x
 *				|(1,2)|(2,2)|(3,2)|
 *				------|-----|------
 *				|(1,3)|(2,3)|(3,3)|
 *				------|-----|------
 *				|(1,4)|(2,4)|(3,4)|
 *				x-----|-----|-----x
 *
 *			Size of the array is : A = (x1-x0+1)*(y1-y0+1)
 */

#ifndef RZ_H
#define RZ_H

#include <iostream>
#include <string>
#include <vector>
#include <map>
using namespace std;

#include "device.h"
#include "coordinates.h"
#include "interface_location.h"
#include "utils.h"

/** 
 * Enumeration of the possible resources states inside a
 * reconfigurable zone.
 * UNUSED:	not constrained
 * USED:	constrained
 * STATIC:	not constrained but cannot be used by another RZ
 */
enum ResourceConstraint{UNUSED, USED, STATIC, NOT_RECONFIGURABLE};

/** 
 * Enumeration of the possible return values of RZ extension function
 * SUCCESS:				OK
 * END_OF_LINE:			End of line reached
 * RESTRICTED_COLUMN:	Column has been declared static and cannot be used
 */
enum RZExtensionReturnValue {SUCCESS, END_OF_LINE, RESTRICTED_COLUMN};

class RZ {

private:

	/** Static variables from FoRTReSS flow  */
	static double VERTEX_COST;
	static double TASK_CORRELATION_COST;
	static double INTERNAL_FRAGMENTATION_COST;
	static int MINIMUM_DISTANCE_BETWEEN_RZ;
	static bool IGNORE_RZ_POSITION_CONSTRAINTS;

	// RZ characterization
	static int static_rz_id;
	int x0, y0, height;
	string id;
	double cost;
	Device *device;
	bool reverse;

	// Resources
	vector<vector<pair<RBType, ResourceConstraint> > > resourcesArray;
	map<RBType, int> constrainedResourcesCount;

	// Compatible tasks
	vector<string> compatibleTasks;
	int nbIncompatibleTasks;

	double resourceWastage;


public:
	RZ(int _x0, int _y0, int _height, Device *dev, bool rev) : x0(_x0), y0(_y0), height(_height), device(dev), reverse(rev) {
		id = (string)"RZ_" + Utils::itoa(static_rz_id);
		static_rz_id++;
		nbIncompatibleTasks = 0;
		resourceWastage = 0;
	}

	RZ(string rz_id, int _x0, int _y0, int _height, Device *dev, bool rev) : x0(_x0), y0(_y0), height(_height), device(dev), reverse(rev) {
		id = rz_id;
		static_rz_id++;
		nbIncompatibleTasks = 0;
		resourceWastage = 0;
	}

	/** Copy constructor */
	RZ(const RZ &rz) {
		x0 = rz.x0;
		y0 = rz.y0;
		id = (string)"RZ_" + Utils::itoa(static_rz_id);
		//id = rz.id;
		static_rz_id++;
		height = rz.height;
		device = rz.device;
		resourcesArray = rz.resourcesArray;
		constrainedResourcesCount = rz.constrainedResourcesCount;
		cost = rz.cost;
		resourcesArray = rz.resourcesArray;
		reverse = rz.reverse;
		nbIncompatibleTasks = rz.nbIncompatibleTasks;
		resourceWastage = rz.resourceWastage;
	}

	// Getters
	int getX0(void);
	int getY0(void);
	int getHeight(void);
	map<RBType, int> getConstrainedResourcesCount(void);
	vector<vector<pair<RBType, ResourceConstraint> > > * getResourcesArrayPtr(void);
	int getRZCost(void) const;
	bool isReversed(void);
	string getID(void) const;
	Device * getDevicePtr(void);
	
	// RZ modification
	RZExtensionReturnValue extendRZ(void);
	void updateConstrainedResources();

	// Misc
	void printRZ(void);
	void info(void) const;
	int computeBitstreamSize(void);
	bool constrainsColumn(Coordinates c);

	// Cost
	void computeCost(void);

	// UCF generation
	void generateUCF(string taskname);
	void generateUCF(string taskname, string pblock_name, ofstream& file);

	// Operators overloadings
	bool operator==(const RZ &rz) const;
	bool operator!=(const RZ &rz) const;

	// Distance / overlapping
	int horizontalDistance(const RZ &other);
	int verticalDistance(const RZ &other);
	bool isOverlappingRZ(const RZ &other);
	bool overlapsRZInSet(vector<RZ> &rzs);

	// Compatible tasks
	void addCompatibleTask(string task);
	void setNbIncompatibleTasks(int nbTasks);
	int getNbIncompatibleTasks(void);
	vector<string> getCompatibleTaskVector(void);
	void clearCompatibleTasks(void);

	// Setters for static variables from FoRTReSS
	static void setTaskCorrelationCost(double c);
	static void setRZShapeCost(double c);
	static void setInternalFragmentationCost(double c);
	static void setMinimumDistanceBetweenRZs(int val);
	static void setIgnoreRZPositionCOnstraints(bool);

	void setResourceWastage(double val);

	bool isResourcePositionOK(InterfaceLocation loc);

	void setResourcesArray(vector<vector<pair<RBType, ResourceConstraint> > > &vec);

	// XML helper functions
	bool areColumnsIdentical(int col1, int col2);
	bool isColumnIdenticallyConstrained(int col);

	void setID(string);

	bool isRZWellShaped(void);
	bool isPurelyReconfigurable(void);

};

#endif
