/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    reconfigurable_zone.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2011-03-15
 * @section DESCRIPTION
 *			Represents a RZ
 */

#include "reconfigurable_zone.h"
#include "utils.h"

string RZ::getName(void) const {
	return name;
}

PEImplementation RZ::get_implementation_type(void) const {
	return implementation;
}

vector<TaskImplementation> RZ::getTaskImplementationVector(void) const {
	return taskImplementationVector;
}

vector<TaskImplementation>* RZ::getTaskImplementationVectorPtr(void) {
	return &taskImplementationVector;
}

Resources* RZ::get_resources_ptr(void) {
	return &resources;
}

int RZ::get_cpu_id(void) const {
	return cpuId;
}

rs_time RZ::getOccupationTime(RZState state) {
	return occupationTimes[state];
}

RZState RZ::getRZState(void) {
	return state;
}

void RZ::setRZState(RZState newState) {

	if(!taskImplementationVector.empty()) {

		// Update implementation info for previous RZ state
		taskImplementationVector.at(currentImplementationID).addOccupationTime(state, rs_time_stamp() - lastModificationTime);
		if(state == RZ_RUNNING) taskImplementationVector.at(currentImplementationID).addOccupationTime(RZ_ACTIVE, rs_time_stamp() - lastModificationTime);

		occupationTimes[state] += (rs_time_stamp() - Utils::maxTime(requestTime, last_window_reset));
		if(state == RZ_RUNNING) occupationTimes[RZ_ACTIVE] += (rs_time_stamp() - Utils::maxTime(requestTime, last_window_reset));

		lastModificationTime = rs_time_stamp();	// For implementations
		requestTime = rs_time_stamp();			// For RZs

		state = newState;
	}
}

void RZ::setCurrentImplementationID(int impl) {
	currentImplementationID = impl;
}

TaskImplementation RZ::getCurrentImplementation(void) {
	return taskImplementationVector.at(currentImplementationID);
}

void RZ::setResources(map<string, int> &res) {
	resources = res;
}

Config_fct_point& RZ::getConfigFctPoint() {
	return *current_cfg_fct_point;
}

void RZ::setConfigFctPoint(Config_fct_point* val) {
	current_cfg_fct_point->setFrequency(val->getFrequency());
	current_cfg_fct_point->setVoltage(val->getVoltage());
}

void RZ::setGroupDomainName(string domainName) {
	groupDomainName = domainName;
}

string RZ::getGroupDomainName() {
	return groupDomainName;
}


void RZ::resetWindow(void) {
	last_window_reset = rs_time_stamp();
}

//double RZ::getStateOccupationRateOverLastWindow(RZState stateToFetch) {
//
//	// Update values
//	setRZState(state);
//
//	rs_time newBusyTimeValue = occupationTimes[stateToFetch] - stateTimeOnLastCall[(int) stateToFetch];
//	stateTimeOnLastCall[(int) stateToFetch] = occupationTimes[stateToFetch];
//
//	window_function[(int) stateToFetch].setNewValue(newBusyTimeValue);
//
//	return window_function[(int) stateToFetch].getInstantOccupationRate();
//}
