/**
 * UNS - CNRS
 * Copyright 2013 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit� de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: licensing@sattse.com
 *
 * @file    recosim_main.cpp
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0-build:2014/06/04
 * @date    April 15, 2016 9:05:29 AM CEST
 * @section DESCRIPTION
 *
 *          Main class for xc7z020 device (Zynq7)
 *          Instantiates the transcoder and launches the simulation.
 *          The simulation produces a log file and a VCD file.
*/

#define FORTRESS_CONFIGURATION Config_Par


#include "rs_time.h"
#include "rs_event.h"
#include "my_priority_queue.h"
#include "user_scheduling.h"
#include "scheduler_request.h"
#include "user_parameters.h"
#include "reconfigurable_zone.h"
#include "testbench_in.h"
#include "testbench_out.h"
#include "module.h"
#include "reconf_threads.h"
#include "configuration_file_parser.h"
#include "utils.h"
#include "reconfiguration_manager.h"
#include "generation_helper.h"
#include "testbench_algorithms.h"
#include "module_algorithms.h"
#include "application.h"

#include "fpga.h"
#include "task.h"

#include <sstream>
#include <fstream>
#include <string>
#include <iostream>
#include <thread>
#include <chrono>
#include <map>

bool Utils::VERBOSE_MODE					= false;

template void user_algorithm_SW<1, 1>(User_algorithm_interface<1, 1>&);
template void dispatch<1, 1>(User_algorithm_interface<1, 1>&);
template void user_algorithm_HW<1, 1>(User_algorithm_interface<1, 1>&);


int main(int argc, char* argv[]) {
	
	filebuf fb0;
	fb0.open("/dev/pts/12",ios::out);
	ostream fout(&fb0);
	fout << string(80, '\n');

	string simulationName;

	if(argc == 2) {
		cout << "RZ file to parse: " << argv[1] << endl;
		simulationName = "";
	} else if(argc == 3) {
		cout << "RZ file to parse: " << argv[1] << endl;
		cout << "Simulation name: " << argv[2] << endl;
		simulationName = argv[2];
	} else {
		cout << "Need parameters. Usage: ./exec path_to_RZ_file simulation_name" << endl;
		return -1;
	}
	
	string path(argv[1]);

	/* Log Files */
	string logname("../../solutions/solution");
	if(argc == 3) logname.append("_");
	logname.append(simulationName);

	/* VCD Trace Files */
	string tracename("../../solutions/solution");
	if(argc == 3) tracename.append("_");
	tracename.append(simulationName);

	/* RZ Properties */
	map<string, RZ_config*> rz_cfg_table;
	RZ_config::initRZConfigTable(rz_cfg_table);
	
	/* Module Instances Declaration*/
	// Application : main.modulesystem
	Module<1, 1>* inst_main__Image_Load[1];
	Module<1, 1>* inst_main__Output_BMP[1];
	Module<1, 1>* inst_main__Cameras[1];
	TestbenchOut<1>* inst_main__TestBench[1];
	TestbenchIn<1>* inst_main__CheckBench[1];
	// Application : camera.modulesystem
	Module<1, 1>* inst_camera__Gradient[1];
	Module<1, 1>* inst_camera__FilterRow[1];
	Module<1, 1>* inst_camera__FilterColumn[1];
	TestbenchOut<1>* inst_camera__Camera_In[1];
	TestbenchIn<1>* inst_camera__Camera_Out[1];

	/* Read RZ Configuration File */
	vector<RZ *> rz_table;
	vector<Task> task_table;
	FPGA* fpga = new FPGA();
	if(readRZConfigurationFile(path, rz_table, task_table, fpga) != 0) {
		cerr << "ERROR: Unable to read RZ configuration file, exiting..." << endl;
		return -1;
	}
	Reconfiguration_manager manager("Reconfiguration_manager", rz_table, task_table, WAITING_QUEUE_HANDLER_REF(AMAP_EDF), fpga, fout, rz_cfg_table);
	
	/************************************************
	 * Application File : main.modulesystem
	 */

	/* Config Application Init */
	string main__config_app_inst_name[1];
	int main__config_app_QoS[1];
	main__config_app_inst_name[0] = "main/App0_0";
	main__config_app_QoS[0] = 0;


	for(int i = 0; i < 1; i++) {
		/* Module Instances */
		inst_main__Image_Load[i] = new Module<1, 1> ("main__Image_Load", reconf_thread_standard, fout, main__config_app_inst_name[i]);
		inst_main__Output_BMP[i] = new Module<1, 1> ("main__Output_BMP", reconf_thread_standard, fout, main__config_app_inst_name[i]);
		inst_main__Cameras[i] = new Module<1, 1> ("main__Cameras", reconf_thread_standard, fout, main__config_app_inst_name[i]);
		inst_main__TestBench[i] = new TestbenchOut<1> ("main__TestBench", MainTestBench<1>, fout);
		inst_main__TestBench[i]->set_period(rs_time(80000));
		inst_main__CheckBench[i] = new TestbenchIn<1> ("main__CheckBench", MainCheckBench<1>, fout);

		/* Add TestBench in Manager */

		/* Create a sub-modules table */
		vector<Manager_interface *> modules_table;
		modules_table.push_back(inst_main__Image_Load[i]);
		modules_table.push_back(inst_main__Output_BMP[i]);
		modules_table.push_back(inst_main__Cameras[i]);

		/* Following Node modules */
		vector<int> channel_connection_vector;

		channel_connection_vector.push_back(get_module_position_in_vector(inst_main__Cameras[i], modules_table));
		inst_main__Image_Load[i]->set_following_modules_list(channel_connection_vector);
		inst_main__Image_Load[i]->add_following_module_connection_name("track");

		channel_connection_vector.clear();
		channel_connection_vector.push_back(-1);
		inst_main__Output_BMP[i]->set_following_modules_list(channel_connection_vector);
		inst_main__Output_BMP[i]->add_following_module_connection_name("save");

		channel_connection_vector.clear();
		channel_connection_vector.push_back(get_module_position_in_vector(inst_main__Output_BMP[i], modules_table));
		inst_main__Cameras[i]->set_following_modules_list(channel_connection_vector);
		inst_main__Cameras[i]->add_following_module_connection_name("results");

		/* Preceding Node modules */
		vector<int> preceding_modules_list;
		preceding_modules_list.push_back(-1); // Entry point
		inst_main__Image_Load[i]->set_preceding_modules_list(preceding_modules_list);
		inst_main__Image_Load[i]->add_preceding_module_connection_name("load");

		preceding_modules_list.clear();
		preceding_modules_list.push_back(get_module_position_in_vector(inst_main__Cameras[i], modules_table));
		inst_main__Output_BMP[i]->set_preceding_modules_list(preceding_modules_list);
		inst_main__Output_BMP[i]->add_preceding_module_connection_name("results");

		preceding_modules_list.clear();
		preceding_modules_list.push_back(get_module_position_in_vector(inst_main__Image_Load[i], modules_table));
		inst_main__Cameras[i]->set_preceding_modules_list(preceding_modules_list);
		inst_main__Cameras[i]->add_preceding_module_connection_name("track");

		inst_main__CheckBench[i]->add_preceding_module_connection_name("save");

		/* Following Bench modules */
		channel_connection_vector.clear();
		channel_connection_vector.push_back(get_module_position_in_vector(inst_main__Image_Load[i], modules_table));
		inst_main__TestBench[i]->set_following_modules_list(channel_connection_vector);
		inst_main__TestBench[i]->add_following_module_connection_name("load");
		channel_connection_vector.clear();

		/* Module Hw/Sw Implementations */
		vector<ModuleImplementation> impl_Image_Load;
		impl_Image_Load.push_back(ModuleImplementation("SwImpl", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, rs_time(10000), rs_time(0.0)));
		impl_Image_Load.back().set_use_context_switch_mode(true);
		impl_Image_Load.back().set_nb_preemption_points(0);
		impl_Image_Load.back().set_algorithm_name("user_algorithm_SW");
		vector<ModuleImplementation> impl_Output_BMP;
		impl_Output_BMP.push_back(ModuleImplementation("SwImpl", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, rs_time(10000), rs_time(0.0)));
		impl_Output_BMP.back().set_use_context_switch_mode(true);
		impl_Output_BMP.back().set_nb_preemption_points(0);
		impl_Output_BMP.back().set_algorithm_name("user_algorithm_SW");
		vector<ModuleImplementation> impl_Cameras;
		impl_Cameras.push_back(ModuleImplementation("dispath", HARD, (algorithm_thread_type)dispatch<1, 1>, rs_time(0.0), rs_time(0.0)));
		impl_Cameras.back().set_use_context_switch_mode(true);
		impl_Cameras.back().set_nb_preemption_points(0);
		impl_Cameras.back().set_frequency(0);
		impl_Cameras.back().set_p_idle(0.0);
		impl_Cameras.back().set_p_run(0.0);
		impl_Cameras.back().set_netlist_name("");
		impl_Cameras.back().set_algorithm_name("dispatch");
		impl_Cameras.push_back(ModuleImplementation("foo", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, rs_time(0.0), rs_time(0.0)));
		impl_Cameras.back().set_use_context_switch_mode(true);
		impl_Cameras.back().set_nb_preemption_points(0);
		impl_Cameras.back().set_algorithm_name("user_algorithm_SW");

		/* Module Properties */
		inst_main__Image_Load[i]->set_deadline_time(rs_time(10000));
		inst_main__Image_Load[i]->set_task_period(rs_time(0));
		inst_main__Image_Load[i]->set_task_offset(rs_time(0));
		inst_main__Image_Load[i]->set_priority(0);
		inst_main__Image_Load[i]->set_implementation_vector(impl_Image_Load);
		inst_main__Output_BMP[i]->set_deadline_time(rs_time(10000));
		inst_main__Output_BMP[i]->set_task_period(rs_time(0));
		inst_main__Output_BMP[i]->set_task_offset(rs_time(0));
		inst_main__Output_BMP[i]->set_priority(0);
		inst_main__Output_BMP[i]->set_implementation_vector(impl_Output_BMP);
		inst_main__Cameras[i]->set_static_module();
		inst_main__Cameras[i]->set_deadline_time(rs_time(10000));
		inst_main__Cameras[i]->set_task_period(rs_time(0));
		inst_main__Cameras[i]->set_task_offset(rs_time(0));
		inst_main__Cameras[i]->set_priority(0);
		inst_main__Cameras[i]->set_implementation_vector(impl_Cameras);

		/* Connection Properties */
		inst_main__Image_Load[i]->set_packet_size("track", 16);
		inst_main__Image_Load[i]->add_interface(ModuleInterface("track", "DEFAULT_IF", false));
		inst_main__Image_Load[i]->set_packet_size("load", 16);
		inst_main__Image_Load[i]->add_interface(ModuleInterface("load", "DEFAULT_IF", false));

		inst_main__Output_BMP[i]->set_packet_size("save", 16);
		inst_main__Output_BMP[i]->add_interface(ModuleInterface("save", "DEFAULT_IF", false));
		inst_main__Output_BMP[i]->set_packet_size("results", 16);
		inst_main__Output_BMP[i]->add_interface(ModuleInterface("results", "DEFAULT_IF", false));

		inst_main__Cameras[i]->set_packet_size("results", 16);
		inst_main__Cameras[i]->add_interface(ModuleInterface("results", "DEFAULT_IF", false));
		inst_main__Cameras[i]->set_packet_size("track", 16);
		inst_main__Cameras[i]->add_interface(ModuleInterface("track", "DEFAULT_IF", false));

		inst_main__TestBench[i]->set_packet_size("load", 16);

		inst_main__CheckBench[i]->set_packet_size("save", 16);


		/* Module Connections */
		inst_main__Image_Load[i]->initiator_socket.bind(&inst_main__Cameras[i]->target_socket);
		inst_main__Output_BMP[i]->initiator_socket.bind(&inst_main__CheckBench[i]->input_target_socket);
		inst_main__Cameras[i]->initiator_socket.bind(&inst_main__Output_BMP[i]->target_socket);
		inst_main__TestBench[i]->output_initiator_socket.bind(&inst_main__Image_Load[i]->target_socket);
		inst_main__TestBench[i]->manager_initiator_socket.bind(&manager.testbench_target_socket);
		manager.testbench_initiator_socket.bind(&inst_main__TestBench[i]->manager_target_socket);

		/* Manager to modules downlink */
		manager.module_initiator_socket.bind(&inst_main__Image_Load[i]->manager_target_socket);
		manager.module_initiator_socket.bind(&inst_main__Output_BMP[i]->manager_target_socket);
		manager.module_initiator_socket.bind(&inst_main__Cameras[i]->manager_target_socket);

		/* Manager to modules uplink */
		inst_main__Image_Load[i]->manager_initiator_socket.bind(&manager.module_target_socket);
		inst_main__Output_BMP[i]->manager_initiator_socket.bind(&manager.module_target_socket);
		inst_main__Cameras[i]->manager_initiator_socket.bind(&manager.module_target_socket);

		/* Add Application Instance */
		int offset_0 = manager.addApplication(main__config_app_inst_name[i], "main__", modules_table, main__config_app_QoS[i]);

		/* Set TestBench (Source only) */
		inst_main__TestBench[i]->set_application_offset(offset_0);
		inst_main__TestBench[i]->set_application_name(main__config_app_inst_name[i]);
		inst_main__CheckBench[i]->set_application_name(main__config_app_inst_name[i]);

		/* Information Module */
		inst_main__Image_Load[i]->info();
		inst_main__Output_BMP[i]->info();
		inst_main__Cameras[i]->info();
		inst_main__TestBench[i]->info();
		inst_main__CheckBench[i]->info();

		/* Bindings verification before starting simulation */

		/* CONFIGURATION TRACE */


	}
	/************************************************
	 * Application File : camera.modulesystem
	 */

	/* Config Application Init */
	string camera__config_app_inst_name[1];
	int camera__config_app_QoS[1];
	camera__config_app_inst_name[0] = "camera/App0_10";
	camera__config_app_QoS[0] = 0;


	for(int i = 0; i < 1; i++) {
		/* Module Instances */
		inst_camera__Gradient[i] = new Module<1, 1> ("camera__Gradient", reconf_thread_standard, fout, camera__config_app_inst_name[i]);
		inst_camera__FilterRow[i] = new Module<1, 1> ("camera__FilterRow", reconf_thread_standard, fout, camera__config_app_inst_name[i]);
		inst_camera__FilterColumn[i] = new Module<1, 1> ("camera__FilterColumn", reconf_thread_standard, fout, camera__config_app_inst_name[i]);
		inst_camera__Camera_In[i] = new TestbenchOut<1> ("camera__Camera_In", Camera_In<1>, fout);
		inst_camera__Camera_In[i]->set_period(rs_time(0));
		inst_camera__Camera_Out[i] = new TestbenchIn<1> ("camera__Camera_Out", Camera_Out<1>, fout);

		/* Add TestBench in Manager */

		/* Create a sub-modules table */
		vector<Manager_interface *> modules_table;
		modules_table.push_back(inst_camera__Gradient[i]);
		modules_table.push_back(inst_camera__FilterRow[i]);
		modules_table.push_back(inst_camera__FilterColumn[i]);

		/* Following Node modules */
		vector<int> channel_connection_vector;

		channel_connection_vector.push_back(get_module_position_in_vector(inst_camera__FilterRow[i], modules_table));
		inst_camera__Gradient[i]->set_following_modules_list(channel_connection_vector);
		inst_camera__Gradient[i]->add_following_module_connection_name("c2");

		channel_connection_vector.clear();
		channel_connection_vector.push_back(get_module_position_in_vector(inst_camera__FilterColumn[i], modules_table));
		inst_camera__FilterRow[i]->set_following_modules_list(channel_connection_vector);
		inst_camera__FilterRow[i]->add_following_module_connection_name("c3");

		channel_connection_vector.clear();
		channel_connection_vector.push_back(-1);
		inst_camera__FilterColumn[i]->set_following_modules_list(channel_connection_vector);
		inst_camera__FilterColumn[i]->add_following_module_connection_name("c4");

		/* Preceding Node modules */
		vector<int> preceding_modules_list;
		preceding_modules_list.push_back(-1); // Entry point
		inst_camera__Gradient[i]->set_preceding_modules_list(preceding_modules_list);
		inst_camera__Gradient[i]->add_preceding_module_connection_name("c1");

		preceding_modules_list.clear();
		preceding_modules_list.push_back(get_module_position_in_vector(inst_camera__Gradient[i], modules_table));
		inst_camera__FilterRow[i]->set_preceding_modules_list(preceding_modules_list);
		inst_camera__FilterRow[i]->add_preceding_module_connection_name("c2");

		preceding_modules_list.clear();
		preceding_modules_list.push_back(get_module_position_in_vector(inst_camera__FilterRow[i], modules_table));
		inst_camera__FilterColumn[i]->set_preceding_modules_list(preceding_modules_list);
		inst_camera__FilterColumn[i]->add_preceding_module_connection_name("c3");

		inst_camera__Camera_Out[i]->add_preceding_module_connection_name("c4");

		/* Following Bench modules */
		channel_connection_vector.clear();
		channel_connection_vector.push_back(get_module_position_in_vector(inst_camera__Gradient[i], modules_table));
		inst_camera__Camera_In[i]->set_following_modules_list(channel_connection_vector);
		inst_camera__Camera_In[i]->add_following_module_connection_name("c1");
		channel_connection_vector.clear();

		/* Module Hw/Sw Implementations */
		vector<ModuleImplementation> impl_Gradient;
		impl_Gradient.push_back(ModuleImplementation("HwImpl", HARD, (algorithm_thread_type)user_algorithm_HW<1, 1>, rs_time(7000), rs_time(0.0)));
		impl_Gradient.back().set_use_context_switch_mode(true);
		impl_Gradient.back().set_nb_preemption_points(0);
		impl_Gradient.back().set_frequency(0);
		impl_Gradient.back().set_p_idle(0.0);
		impl_Gradient.back().set_p_run(0.0);
		impl_Gradient.back().set_netlist_name("");
		impl_Gradient.back().set_algorithm_name("");
		impl_Gradient.push_back(ModuleImplementation("SwImpl", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, rs_time(28000), rs_time(0.0)));
		impl_Gradient.back().set_use_context_switch_mode(true);
		impl_Gradient.back().set_nb_preemption_points(0);
		impl_Gradient.back().set_algorithm_name("user_algorithm_SW");
		vector<ModuleImplementation> impl_FilterRow;
		impl_FilterRow.push_back(ModuleImplementation("HwImpl", HARD, (algorithm_thread_type)user_algorithm_HW<1, 1>, rs_time(3500), rs_time(0.0)));
		impl_FilterRow.back().set_use_context_switch_mode(true);
		impl_FilterRow.back().set_nb_preemption_points(0);
		impl_FilterRow.back().set_frequency(0);
		impl_FilterRow.back().set_p_idle(0.0);
		impl_FilterRow.back().set_p_run(0.0);
		impl_FilterRow.back().set_netlist_name("");
		impl_FilterRow.back().set_algorithm_name("");
		impl_FilterRow.push_back(ModuleImplementation("SwImpl", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, rs_time(14000), rs_time(0.0)));
		impl_FilterRow.back().set_use_context_switch_mode(true);
		impl_FilterRow.back().set_nb_preemption_points(0);
		impl_FilterRow.back().set_algorithm_name("user_algorithm_SW");
		vector<ModuleImplementation> impl_FilterColumn;
		impl_FilterColumn.push_back(ModuleImplementation("HwImpl", HARD, (algorithm_thread_type)user_algorithm_HW<1, 1>, rs_time(3500), rs_time(0.0)));
		impl_FilterColumn.back().set_use_context_switch_mode(true);
		impl_FilterColumn.back().set_nb_preemption_points(0);
		impl_FilterColumn.back().set_frequency(0);
		impl_FilterColumn.back().set_p_idle(0.0);
		impl_FilterColumn.back().set_p_run(0.0);
		impl_FilterColumn.back().set_netlist_name("");
		impl_FilterColumn.back().set_algorithm_name("");
		impl_FilterColumn.push_back(ModuleImplementation("SwImpl", SOFT, (algorithm_thread_type)user_algorithm_SW<1, 1>, rs_time(14000), rs_time(0.0)));
		impl_FilterColumn.back().set_use_context_switch_mode(true);
		impl_FilterColumn.back().set_nb_preemption_points(0);
		impl_FilterColumn.back().set_algorithm_name("user_algorithm_SW");

		/* Module Properties */
		inst_camera__Gradient[i]->set_deadline_time(rs_time(10000));
		inst_camera__Gradient[i]->set_task_period(rs_time(0));
		inst_camera__Gradient[i]->set_task_offset(rs_time(0));
		inst_camera__Gradient[i]->set_priority(0);
		inst_camera__Gradient[i]->set_implementation_vector(impl_Gradient);
		inst_camera__FilterRow[i]->set_deadline_time(rs_time(10000));
		inst_camera__FilterRow[i]->set_task_period(rs_time(0));
		inst_camera__FilterRow[i]->set_task_offset(rs_time(0));
		inst_camera__FilterRow[i]->set_priority(0);
		inst_camera__FilterRow[i]->set_implementation_vector(impl_FilterRow);
		inst_camera__FilterColumn[i]->set_deadline_time(rs_time(10000));
		inst_camera__FilterColumn[i]->set_task_period(rs_time(0));
		inst_camera__FilterColumn[i]->set_task_offset(rs_time(0));
		inst_camera__FilterColumn[i]->set_priority(0);
		inst_camera__FilterColumn[i]->set_implementation_vector(impl_FilterColumn);

		/* Connection Properties */
		inst_camera__Gradient[i]->set_packet_size("c2", 16);
		inst_camera__Gradient[i]->add_interface(ModuleInterface("c2", "DEFAULT_IF", false));
		inst_camera__Gradient[i]->set_packet_size("c1", 16);
		inst_camera__Gradient[i]->add_interface(ModuleInterface("c1", "DEFAULT_IF", false));

		inst_camera__FilterRow[i]->set_packet_size("c3", 16);
		inst_camera__FilterRow[i]->add_interface(ModuleInterface("c3", "DEFAULT_IF", false));
		inst_camera__FilterRow[i]->set_packet_size("c2", 16);
		inst_camera__FilterRow[i]->add_interface(ModuleInterface("c2", "DEFAULT_IF", false));

		inst_camera__FilterColumn[i]->set_packet_size("c4", 16);
		inst_camera__FilterColumn[i]->add_interface(ModuleInterface("c4", "DEFAULT_IF", false));
		inst_camera__FilterColumn[i]->set_packet_size("c3", 16);
		inst_camera__FilterColumn[i]->add_interface(ModuleInterface("c3", "DEFAULT_IF", false));

		inst_camera__Camera_In[i]->set_packet_size("c1", 16);

		inst_camera__Camera_Out[i]->set_packet_size("c4", 16);


		/* Module Connections */
		inst_camera__Gradient[i]->initiator_socket.bind(&inst_camera__FilterRow[i]->target_socket);
		inst_camera__FilterRow[i]->initiator_socket.bind(&inst_camera__FilterColumn[i]->target_socket);
		inst_camera__FilterColumn[i]->initiator_socket.bind(&inst_camera__Camera_Out[i]->input_target_socket);
		inst_camera__Camera_In[i]->output_initiator_socket.bind(&inst_camera__Gradient[i]->target_socket);
		inst_camera__Camera_In[i]->manager_initiator_socket.bind(&manager.testbench_target_socket);
		manager.testbench_initiator_socket.bind(&inst_camera__Camera_In[i]->manager_target_socket);

		/* Manager to modules downlink */
		manager.module_initiator_socket.bind(&inst_camera__Gradient[i]->manager_target_socket);
		manager.module_initiator_socket.bind(&inst_camera__FilterRow[i]->manager_target_socket);
		manager.module_initiator_socket.bind(&inst_camera__FilterColumn[i]->manager_target_socket);

		/* Manager to modules uplink */
		inst_camera__Gradient[i]->manager_initiator_socket.bind(&manager.module_target_socket);
		inst_camera__FilterRow[i]->manager_initiator_socket.bind(&manager.module_target_socket);
		inst_camera__FilterColumn[i]->manager_initiator_socket.bind(&manager.module_target_socket);

		/* Add Application Instance */
		int offset_1 = manager.addApplication(camera__config_app_inst_name[i], "camera__", modules_table, camera__config_app_QoS[i]);

		/* Set TestBench (Source only) */
		inst_camera__Camera_In[i]->set_application_offset(offset_1);
		inst_camera__Camera_In[i]->set_application_name(camera__config_app_inst_name[i]);
		inst_camera__Camera_Out[i]->set_application_name(camera__config_app_inst_name[i]);

		/* Information Module */
		inst_camera__Gradient[i]->info();
		inst_camera__FilterRow[i]->info();
		inst_camera__FilterColumn[i]->info();
		inst_camera__Camera_In[i]->info();
		inst_camera__Camera_Out[i]->info();

		/* Bindings verification before starting simulation */

		/* CONFIGURATION TRACE */


	}


	/* Application Defined */
	manager.applicationDefined();
	
	/* DEBUG TRACE */
	// Disabled

	/* Activate Trace */

	/* Simulator Configuration */
	manager.set_scheduler_execution_time(rs_time(0));


	/* Offset & Hyperperiod */
	Application::setOffset(rs_time(0.0, RS_US));
	Application::setHyperperiod(rs_time(80000.0, RS_US));



	/* Start Application(s) */
	this_thread::sleep_for(chrono::seconds(4));
	
	for(int i = 0; i < 1; i++) {
		cout << "open TB out" << rs_time_stamp() << endl;
		inst_main__TestBench[i]->start_tb();
	}

	for(int i = 0; i < 1; i++) {
		cout << "open TB out" << rs_time_stamp() << endl;
		inst_camera__Camera_In[i]->start_tb();
	}



	rs_event* main_sleep;
	main_sleep = new rs_event();
	main_sleep->wait();

	/* End of Simulation */
	Reconfiguration_manager::verifySimulationTimes();
		
	/* Display Information */

	/* Flush Streams */
	fout.flush();
}


