/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    testbench_out_interface.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-01
 * @section DESCRIPTION
 *			Interface for testbenches with outputs
 */

#ifndef TESTBENCH_OUT_INTERFACE_H
#define TESTBENCH_OUT_INTERFACE_H

#ifdef RECOSIM_PART
	#define SC_INCLUDE_DYNAMIC_PROCESSES
	#include <systemc.h>
#endif

#ifdef MIDDLEWARE_GEN
	#include "rs_time.h"
	#include "rs_event.h"
	#include <iostream>
#endif

#include <string>
using namespace std;

template<int No>
class Testbench_out_interface /*: public sc_interface*/ {
	
public:

	virtual void wait_on_sinc() = 0;
	/**
	 * @brief Get testbench full name
	 * @return Constant char* representation of testbench full name
	 */
	virtual string TB_IF_name(void) const = 0;

	/**
	 * @brief Get testbench instance name
	 * @return string representation of testbench instance name
	 */
	virtual string TB_IF_get_instance_name(void) const = 0;
	virtual string TB_IF_get_instance_top_application_name(void) const = 0;
	/**
	 * @brief Get pointer to ougoing data structure
	 * @param socketID Socket ID
	 * @return Pointer to the outgoing transactions data
	 */
	virtual int* TB_IF_get_data_out_ptr(int socketID) const = 0;

	/**
	 * @brief Get pointer to ougoing data structure
	 * @param connectionName Name of the connection bound to the socket
	 * @return Pointer to the outgoing transactions data
	 */
	virtual int* TB_IF_get_data_out_ptr(string connectionName) const = 0;

	/**
	 * @brief Non-blocking data send through a given socket
	 * @param socketID Socket ID
	 */
	virtual void TB_IF_nb_send_data(int socketID) = 0;

	/**
	 * @brief Non-blocking data send through a given socket
	 * @param connectionName Name of the connection bound to the socket
	 */
	virtual void TB_IF_nb_send_data(string connectionName) = 0;

	/**
	 * @brief Non-blocking data send through every socket
	 */
	virtual void TB_IF_nb_send_all_data(void) = 0;

	/**
	 * @brief Watch the event issued once all TLM responses have been
	 * sent from following modules to testbench.
	 * @return constant reference to the SystemC event involved
	 */
	//virtual const rs_event& TB_IF_all_responses_received_event(void) const = 0;
	virtual void TB_IF_wait_all_responses_received(void) = 0;

	/**
	 * @brief Increments working transaction counter.
	 */
	virtual void TB_IF_increment_current_transaction_id(void) = 0;

	/**
	 * @brief Get application name (application in which this testbench
	 * has been instantiated).
	 * @return string representation of application name
	 */
	virtual string TB_IF_getApplicationName(void) = 0;

	/**
	 * @brief Get testbench period
	 * @return sc_time representation of testbench period
	 */
	virtual rs_time TB_IF_get_period(void) const = 0;

	/**
	 * @brief Add an algorithm control command to the waiting queue. After the delay passed as parameter,
	 * testbench will notify module 'module_name' that its algorithm should go to mode 'command'
	 * @param module_name Name of the targetted module, as displayed in FoRTReSS GUI
	 * @param command New control command
	 * @param delay Time before the command should be executed
	 */
	//virtual void TB_IF_add_execution_mode_command_to_queue(string module_name, string command, rs_time delay) = 0;

	/**
	 * @brief Get socket packet size
	 * @param id Socket ID
	 * @return Packet size (in 32-bit words)
	 */
	virtual int TB_IF_get_packet_size(int ID) const = 0;

	/**
	 * @brief Get socket packet size
	 * @param name Name of the connection bound to the socket
	 * @return Packet size (in 32-bit words)
	 */
	virtual int TB_IF_get_packet_size(string name) const = 0;

	/**
	 * @brief Get input socket id by the name of the connection
	 * @param name Connection name
	 * @return Socket ID
	 */
	virtual int TB_IF_get_output_socket_id(string name) const = 0;

	/**
	 * @brief Get logstream
	 * @return Log stream
	 */
	virtual ostream& get_logfile(void) = 0;
};

#endif
