/**
 * UNS - CNRS
 * Copyright 2012 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: contact@sattse.com
 *
 * @file    testbench_in_interface.h
 * @author  Francois Duhem (Francois.Duhem@unice.fr), Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0
 * @date    2012-10-01
 * @section DESCRIPTION
 *			Interface for testbenches with inputs
 */

#ifndef TESTBENCH_IN_INTERFACE_H
#define TESTBENCH_IN_INTERFACE_H

#ifdef RECOSIM_PART
	#define SC_INCLUDE_DYNAMIC_PROCESSES
	#include <systemc.h>
#endif

#ifdef MIDDLEWARE_GEN
	#include <iostream>
	#include "rs_event.h"
#endif

#include <string>
using namespace std;

template<int Ni>
class Testbench_in_interface /*: public sc_interface*/ {

public:

	/**
	 * @brief Get testbench full name
	 * @return Constant char* representation of testbench full name
	 */
	virtual string TB_IF_name(void) const = 0;

	/**
	 * @brief Get testbench instance name
	 * @return string representation of testbench instance name
	 */
	virtual string TB_IF_get_instance_name(void) const = 0;

	/**
	 * @brief Get top application instance name
	 * @return string representation of testbench instance name
	 */

	virtual string TB_IF_get_instance_top_application_name(void) const = 0;

	/**
	 * @brief Get the event launched when a transaction has been received
	 * @return constant reference to the event launched
	 */
	virtual const rs_event& TB_IF_transaction_received_event(void) const = 0;

	/**
	 * @brief Wait for the event launched when a transaction has been received, (Only available for middleware) 
	 */
	virtual void TB_IF_wait_transaction_received_event(void) = 0;
	/**
	 * @brief Wait for the event launched when a transaction has been received or timeout achieved, (Only available for middleware)
	 * @param rs_time timeout
	 */
	virtual void TB_IF_wait_transaction_received_event(rs_time) = 0;

	/**
	 * @brief Get the number of transactions processed per socket
	 * @param socketID Socket ID
	 * @return number of transactions
	 */
	virtual const int TB_IF_get_nb_transactions_received_per_socket(int socketID) const = 0;

	/**
	 * @brief Get the number of transactions processed per socket
	 * @param channelName Name of the connection linked to the socket
	 * @return number of transactions
	 */
	virtual const int TB_IF_get_nb_transactions_received_per_socket(string channelName) const = 0;

	/**
	 * @brief Get the number of transactions processed by every socket
	 * @return number of transactions
	 */
	virtual const int TB_IF_get_nb_transactions_received(void) const = 0;

	/**
	 * @brief Get incoming data pointer.
	 * @param socketID Socket ID
	 * @return 2D pointer to the incoming data structure
	 */
	virtual int* TB_IF_get_data_in_ptr(int socketID) const = 0;

	/**
	 * @brief Get incoming data pointer.
	 * @param connectionName Name of the connection bound to the socket
	 * @return 2D pointer to the incoming data structure
	 */
	virtual int* TB_IF_get_data_in_ptr(string connectionName) const = 0;

	/**
	 * @brief Notifies simulation controller about success/failure of
	 * the simulation (e.g. data corruption).
	 * @param success whether the simulation is successfull or not
	 */
	virtual void TB_IF_notify_simulation_controller(bool success) = 0;

	/**
	 * @brief Get socket packet size
	 * @param id Socket ID
	 * @return Packet size (in 32-bit words)
	 */
	virtual int TB_IF_get_packet_size(int ID) const = 0;

	/**
	 * @brief Get input socket id by the name of the connection
	 * @param name Connection name
	 * @return Socket ID
	 */
	virtual int TB_IF_get_input_socket_id(string name) const = 0;

	/**
	 * @brief Get the name of the connection bound to a particular socket
	 * @param id Socket ID
	 * @return Connection name
	 */
	virtual string TB_IF_get_connection_name(int id) const = 0;

	/**
	 * @brief Get a list of the sockets that received a new transaction
	 * since last function call. WARNING: Calling this function will reset
	 * the vector stored within the testbench and hence should be stored in a
	 * new vector within the algorithm thread.
	 * @return Vector of sockets IDs
	 */
	virtual vector<int> TB_IF_get_sockets_with_new_transactions(void) = 0;

    /**
	 * @brief Get logfstream
	 * @return Log stream
	 */
	virtual ostream& get_logfile(void)= 0;

};

#endif
