/**
 * UNS - CNRS
 * Copyright 2013 All Rights Reserved.
 *
 * These computer program listings and specifications, herein, are
 * the property of Universit de Nice Sophia-Antipolis (UNS) and Centre National
 * de la Recherche Scientifique (CNRS), and shall not be reproduced or
 * copied or used in whole or in part as the basis for manufacture
 * or sale of items without written permission.
 * For a license agreement, please contact: licensing@sattse.com
 *
 * @file    user_monitoring.h
 * @author  Fabrice Muller (Fabrice.Muller@unice.fr)
 *          University of Nice-Sophia Antipolis - LEAT/CNRS
 * @version 1.0-build:2013/06/10
 * @date    June 25, 2014 2:10:21 PM CEST
 * @section DESCRIPTION
 *
 *          Custom Monitoring
*/

#ifndef USER_MONITORING_H
#define USER_MONITORING_H

#include <systemc.h>
#include "monitoring.h"

/*
* NOTE:
*  Change the name of :
*  - user_monitoring_module_name : this is the monitoring module name declared in FoRTReSS preferences
*  - user_sig : example of a tracing variable 
*  - trace_update_thread : you can modify and add several threads
*
*  - Follow the steps below
*/


MONITORING_MODULE(Power_Monitoring_old) {  

	/***** STEP 1 *********************************************/
	/* Declaration                                            */
	/*                                                        */
	/* Types :                                                */
	/* double           : MONITORING_VAR_DOUBLE               */
	/* integer          : MONITORING_VAR_INT                  */
	/* unsigned integer : MONITORING_VAR_UINT                 */
	/* string           : MONITORING_VAR_STRING               */

	TRACE_VAR_DOUBLE(previous_hyper_energy);
	TRACE_VAR_DOUBLE(hyper_instant_energy);
	TRACE_VAR_DOUBLE(total_current_power);
	
	TRACE_VAR_DOUBLE(rz_reconf_current_power);
	TRACE_VAR_DOUBLE(rz_hard_current_static_power);
	TRACE_VAR_DOUBLE(rz_soft_current_idle_power);
	TRACE_VAR_DOUBLE(rz_soft_current_static_power);
	TRACE_VAR_DOUBLE(rz_hard_current_idle_power);
	TRACE_VAR_DOUBLE(rz_hard_current_run_power);
	TRACE_VAR_DOUBLE(rz_soft_current_run_power);

	double hyper_energy;

	
	/***** STEP 2 *********************************************/
	/* Init Variable(s) and Thread(s)                         */
	
	MONITORING_INIT(Power_Monitoring_old) {

		// Init Variable(s)
		hyper_energy = 0;
		total_current_power = 0;
		previous_hyper_energy =0;

		// Init Thread(s) for Trace
		MONITORING_ADD_THREAD(Power_Monitoring_old, trace_update_thread);
	}

	/***** STEP 3 *********************************************/
	/* Record Variable(s)                                     */

	MONITORING_ACTIVATE {

		/*  2 Methods :
		*    - MONITORING_ANALOG_TRACE(declaration_name, "name_of_signal", min value, max value, height in pixel)
		*        height : in pixel, less than value 50 is not right. About 80 is a right value.
		*
		*    - MONITORING_TRACE(declaration_name, "name_of_signal", format) 
		*        format value: hexadecimal, ascii, unsigned, decimal
		*
		*  Note: The declaration order will be the wave declaration order in the trace file
		*/

		MONITORING_ANALOG_TRACE(previous_hyper_energy, "Previous_Hyper_Energy", 0, 100, 75); 
		MONITORING_ANALOG_TRACE(hyper_instant_energy, "Hyper_Instant_Energy", 0, 100, 75); 
		MONITORING_ANALOG_TRACE(total_current_power, "Instant_current", 0, 2000, 75); 
		
		MONITORING_ANALOG_TRACE(rz_reconf_current_power, "Reconf_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_hard_current_static_power,  "RZ_static_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_hard_current_idle_power,  "RZ_idle_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_hard_current_run_power,  "RZ_run_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_soft_current_static_power,  "Cores_static_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_soft_current_idle_power,  "Cores_idle_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_soft_current_run_power,  "Cores_run_current", 0, 1000, 75);
		
		
	}

	/***** STEP 4 *********************************************/
	/* Define the Thread behavior(s)                          */

	MONITORING_THREAD(trace_update_thread) {

		//double total_current_power=0;
		sc_time current_time(SC_ZERO_TIME);
		sc_time previous_time(SC_ZERO_TIME);
		bool new_period=0;	
		
		total_current_power=0;
			double rz_current_power=0;
			rz_reconf_current_power=0;
			rz_hard_current_static_power=0;
			rz_soft_current_static_power=0;
			rz_hard_current_idle_power=0;
			rz_soft_current_idle_power=0;
			rz_hard_current_run_power=0;
			rz_soft_current_run_power=0;
			hyper_energy=0;
	
		// Example of behavior
		Scheduler_interface& scheduler = services.getShedulerInterface();
	
		while(true) {

			/* Sampling Period */
			//wait(monitoring_if.endOfSchedulingEvent());
			

			/* Modify Value of the Signal */
			vector<RZ *> all_RZs = scheduler.Scheduler_get_all_rz_vector();
			int i = 0;
			double Pstatic, Pidle, Prun;
			
			total_current_power=0;
			rz_current_power=0;
			rz_reconf_current_power=0;
			rz_hard_current_static_power=0;
			rz_soft_current_static_power=0;
			rz_hard_current_idle_power=0;
			rz_soft_current_idle_power=0;
			rz_hard_current_run_power=0;
			rz_soft_current_run_power=0;
			
			
			while(i < (int) all_RZs.size()) {
			
				//get powers for HW or SW RZ.
				if (all_RZs.at(i)->get_implementation_type()==HARD)
				{	
					Pstatic=P_STATIC_1CELL*all_RZs.at(i)->get_resources_ptr()->getResourceCount("Slice")*40; //*40  expliquer!!
					Pidle=all_RZs.at(i)->getCurrentImplementation().getPidle();
					Prun= all_RZs.at(i)->getCurrentImplementation().getPrun();
				}
				else //SOFT
				{
					/*if (all_RZs.at(i)->getName()=="Core_2") //handle foo CPU for power trace.
					{
						Pstatic=0;
						Pidle=0;
						Prun=0;
						//scheduler.Scheduler_get_output_stream() << "FooCPU found" << endl;
					}
					else*/
					{
						Pstatic=CPU_PSTATIC;
						Pidle=CPU_PIDLE;
						Prun=CPU_PRUN;
					}
					
					//scheduler.Scheduler_get_output_stream() << "RZname : " <<  << " get fooCPU : " << all_RZs.at(i)->get_resources_ptr()->getResourceCount("FooCPU")<< endl;	
					
				}
				
				//computer RZ total power following each state
				if(			scheduler.Scheduler_is_RZ_blank(all_RZs.at(i)) )
						{
							rz_current_power = Pstatic;
							if (all_RZs.at(i)->get_implementation_type()==HARD)
							{
								rz_hard_current_static_power += rz_current_power;
							}
							else
								rz_soft_current_static_power += rz_current_power;
						}
				else if (	scheduler.Scheduler_get_task_state(all_RZs.at(i)) == MAPPED ||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == PREEMPTED_MAPPED ||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == QUEUED || //if not, is it free?
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == INACTIVE||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == PREEMPTED_INACTIVE || //if not, is it free?
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == WAITING||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == IDLE || //if not, is it free?
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == STATIC)
						{
							 //scheduler.Scheduler_get_current_module_ID(all_RZs.at(i));;	
							if (all_RZs.at(i)->get_implementation_type()==HARD)
								{

								rz_hard_current_static_power += Pstatic;
								rz_hard_current_idle_power += Pidle;
								rz_current_power = Pstatic + Pidle;
								}
							else
								{
								//rz_soft_current_static_power += Pstatic;
								rz_soft_current_idle_power += Pidle;
								rz_current_power = Pidle;
								}
						}
				else if (	scheduler.Scheduler_get_task_state(all_RZs.at(i)) == RUNNING)
						{
							
							if (all_RZs.at(i)->get_implementation_type()==HARD)
							{

								rz_hard_current_static_power += Pstatic;
								rz_hard_current_idle_power += Pidle;
								rz_hard_current_run_power += Prun;
								rz_current_power = Pstatic + Pidle+ Prun;
							}
							else
							{
								//rz_soft_current_static_power += Pstatic;
								//rz_soft_current_idle_power += Pidle;
								rz_soft_current_run_power += Prun;
								rz_current_power = Prun;
							}
						}
				else if (	scheduler.Scheduler_get_task_state(all_RZs.at(i)) == CONFIG)
						{
							rz_current_power = Pstatic + P_RECONF; //maybe to refine with Pidle model during reconf?
							if (all_RZs.at(i)->get_implementation_type()==HARD)
							{
								rz_hard_current_static_power += Pstatic;

							}
							else
							{
								rz_soft_current_static_power += Pstatic;
							}
								
							rz_reconf_current_power += P_RECONF;
						}
							
							
				//scheduler.Scheduler_get_output_stream() << "ALL_RZ    RZname: " << all_RZs.at(i)->getName()<< " - RZSlices: " << all_RZs.at(i)->get_resources_ptr()->getResourceCount("Slice") << " - RZclbm: " << all_RZs.at(i)->get_resources_ptr()->getResourceCount("CLBm")<< " - RZclbl: " << all_RZs.at(i)->get_resources_ptr()->getResourceCount("CLBl")<< endl;	 
				//scheduler.Scheduler_get_output_stream() << "RZname: " << all_RZs.at(i)->getName()<< " power : " << rz_current_power << " mW " << endl;	 
				total_current_power+=rz_current_power;
				i++;
			}
			
			
			
			double hyper_period_step = (sc_time_stamp()-Application::getGlobalOffset())/Application::getGlobalHypeperiod();
			
			if (hyper_period_step == (long)hyper_period_step && new_period==1)
			{
				if (sc_time_stamp()>=Application::getGlobalOffset ())
					current_time=sc_time_stamp()-Application::getGlobalOffset()-((long)hyper_period_step)*Application::getGlobalHypeperiod();
				else
					current_time=sc_time_stamp();
				previous_time=current_time;
			}
			else
			{
				if (sc_time_stamp()>Application::getGlobalOffset ())
					current_time=sc_time_stamp()-Application::getGlobalOffset()-(long)hyper_period_step*Application::getGlobalHypeperiod();
				else
					current_time=sc_time_stamp();
			}
			
			if ((current_time-previous_time).to_seconds()>5) //patch ...
				previous_time=current_time;
			
			if (new_period==1 && previous_time!=current_time)
				new_period=0;
			
			hyper_energy+=total_current_power*(current_time-previous_time).to_seconds();
			
/*			scheduler.Scheduler_get_output_stream() << "Current power : total : " << total_current_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " soft_static: " << rz_soft_current_static_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " soft_idle: " << rz_soft_current_idle_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " soft_run: " << rz_soft_current_run_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " hard_static: " << rz_hard_current_static_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " hard_idle: " << rz_hard_current_idle_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " hard_run: " << rz_hard_current_run_power<< " mW" ;		
			scheduler.Scheduler_get_output_stream() << " reconf: " << rz_reconf_current_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " during: " << (current_time-previous_time).to_seconds()*1000 << " ms at "<< current_time.to_seconds()*1000 << " ms at global " << sc_time_stamp().to_seconds()*1000 << " ms" << endl;
*/					
			if (hyper_period_step == (long)hyper_period_step && new_period==0)
			{
/*				scheduler.Scheduler_get_output_stream() << "!!NEW HYPER!!" << endl;
				scheduler.Scheduler_get_output_stream() << "Energy : " << hyper_energy << " mJ" << endl;
*/				previous_hyper_energy=hyper_energy;
				hyper_energy=0;
				
				if (sc_time_stamp()>Application::getGlobalOffset ())
					current_time=sc_time_stamp()-Application::getGlobalOffset()-(long)hyper_period_step*Application::getGlobalHypeperiod();
				else
					current_time=sc_time_stamp();
				new_period=1;
				
				
			}

			
			hyper_instant_energy = hyper_energy;
			//scheduler.Scheduler_get_output_stream() << "scheduler_period : " << current_time-previous_time << endl;
			previous_time=current_time;

		wait(sc_time(100, SC_US));
		}
	}
	
};

MONITORING_MODULE(Power_Monitoring) {  

	/***** STEP 1 *********************************************/
	/* Declaration                                            */
	/*                                                        */
	/* Types :                                                */
	/* double           : MONITORING_VAR_DOUBLE               */
	/* integer          : MONITORING_VAR_INT                  */
	/* unsigned integer : MONITORING_VAR_UINT                 */
	/* string           : MONITORING_VAR_STRING               */

	TRACE_VAR_DOUBLE(previous_hyper_energy);
	TRACE_VAR_DOUBLE(hyper_instant_energy);
	TRACE_VAR_DOUBLE(total_current_power);
	
	TRACE_VAR_DOUBLE(rz_reconf_current_power);
	TRACE_VAR_DOUBLE(rz_hard_current_static_power);
	TRACE_VAR_DOUBLE(rz_soft_current_idle_power);
	TRACE_VAR_DOUBLE(rz_soft_current_static_power);
	TRACE_VAR_DOUBLE(rz_hard_current_idle_power);
	TRACE_VAR_DOUBLE(rz_hard_current_run_power);
	TRACE_VAR_DOUBLE(rz_soft_current_run_power);
	TRACE_VAR_DOUBLE(cpu_operating_point);

	double hyper_energy;

	
	/***** STEP 2 *********************************************/
	/* Init Variable(s) and Thread(s)                         */
	
	MONITORING_INIT(Power_Monitoring) {

		// Init Variable(s)
		hyper_energy = 0;
		total_current_power = 0;
		previous_hyper_energy =0;

		// Init Thread(s) for Trace
		MONITORING_ADD_THREAD(Power_Monitoring, trace_update_thread_cpu);
	}

	/***** STEP 3 *********************************************/
	/* Record Variable(s)                                     */

	MONITORING_ACTIVATE {

		/*  2 Methods :
		*    - MONITORING_ANALOG_TRACE(declaration_name, "name_of_signal", min value, max value, height in pixel)
		*        height : in pixel, less than value 50 is not right. About 80 is a right value.
		*
		*    - MONITORING_TRACE(declaration_name, "name_of_signal", format) 
		*        format value: hexadecimal, ascii, unsigned, decimal
		*
		*  Note: The declaration order will be the wave declaration order in the trace file
		*/

		MONITORING_ANALOG_TRACE(previous_hyper_energy, "Previous_Hyper_Energy", 0, 100, 75); 
		MONITORING_ANALOG_TRACE(hyper_instant_energy, "Hyper_Instant_Energy", 0, 100, 75); 
		MONITORING_ANALOG_TRACE(total_current_power, "Instant_current", 0, 2000, 75); 
		
		MONITORING_ANALOG_TRACE(rz_reconf_current_power, "Reconf_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_hard_current_static_power,  "RZ_static_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_hard_current_idle_power,  "RZ_idle_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_hard_current_run_power,  "RZ_run_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_soft_current_static_power,  "Cores_static_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_soft_current_idle_power,  "Cores_idle_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(rz_soft_current_run_power,  "Cores_run_current", 0, 1000, 75);
		MONITORING_ANALOG_TRACE(cpu_operating_point, "CPU_operating_point", 0, 2, 75);
		
	}

	/***** STEP 4 *********************************************/
	/* Define the Thread behavior(s)                          */

	MONITORING_THREAD(trace_update_thread_cpu) {

		//double total_current_power=0;
		sc_time current_time(SC_ZERO_TIME);
		sc_time previous_time(SC_ZERO_TIME);
		bool new_period=0;	
		
		total_current_power=0;
			double rz_current_power=0;
			rz_reconf_current_power=0;
			rz_hard_current_static_power=0;
			rz_soft_current_static_power=0;
			rz_hard_current_idle_power=0;
			rz_soft_current_idle_power=0;
			rz_hard_current_run_power=0;
			rz_soft_current_run_power=0;
			hyper_energy=0;
			
		int nb_CPU =0;
		int nb_Running_CPU=0;
		
		//for future use :
		double max_CPU_freq=PM_CPU_FREQUENCY;
		double max_CPU_V=PM_CPU_VOLTAGE;
	
		// Example of behavior
		Scheduler_interface& scheduler = services.getShedulerInterface();
	
		while(true) {

			/* Sampling Period */
			//wait(monitoring_if.endOfSchedulingEvent());
			

			/* Modify Value of the Signal */
			vector<RZ *> all_RZs = scheduler.Scheduler_get_all_rz_vector();
			int i = 0;
			double Pstatic, Pidle, Prun;
			
			total_current_power=0;
			rz_current_power=0;
			rz_reconf_current_power=0;
			rz_hard_current_static_power=0;
			rz_soft_current_static_power=0;
			rz_hard_current_idle_power=0;
			rz_soft_current_idle_power=0;
			rz_hard_current_run_power=0;
			rz_soft_current_run_power=0;
			
			nb_CPU=0;
			nb_Running_CPU=0;
			
			while(i < (int) all_RZs.size()) {
			
				//get powers for HW or SW RZ.
				if (all_RZs.at(i)->get_implementation_type()==HARD)
				{	
					Pstatic=P_STATIC_1CELL*all_RZs.at(i)->get_resources_ptr()->getResourceCount("Slice")*40; //*40  expliquer!!
					Pidle=all_RZs.at(i)->getCurrentImplementation().getPidle();
					Prun= all_RZs.at(i)->getCurrentImplementation().getPrun();
				}
				else //SOFT
				{
//						Pstatic=CPU_PSTATIC;
//						Pidle=CPU_PIDLE;
//						Prun=CPU_PRUN;
						nb_CPU++;
				}
				
				//computer RZ total power following each state
				if(scheduler.Scheduler_is_RZ_blank(all_RZs.at(i)) )
						{
							//rz_current_power = Pstatic;
							rz_current_power = 0;
							if (all_RZs.at(i)->get_implementation_type()==HARD)
							{
								rz_hard_current_static_power += Pstatic;
								rz_current_power+=Pstatic;
							}
							else
							{
								//rz_soft_current_static_power += rz_current_power;
							}
						}
				else if (	scheduler.Scheduler_get_task_state(all_RZs.at(i)) == MAPPED ||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == PREEMPTED_MAPPED ||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == QUEUED || 
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == INACTIVE||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == PREEMPTED_INACTIVE ||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == WAITING||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == IDLE ||
							scheduler.Scheduler_get_task_state(all_RZs.at(i)) == STATIC)
						{
							 //scheduler.Scheduler_get_current_module_ID(all_RZs.at(i));;	
							if (all_RZs.at(i)->get_implementation_type()==HARD)
								{

								rz_hard_current_static_power += Pstatic;
								rz_hard_current_idle_power += Pidle;
								rz_current_power = Pstatic + Pidle;
								}
							else
								{
								//rz_soft_current_static_power += Pstatic;
								//rz_soft_current_idle_power += Pidle;
								//rz_current_power = Pidle;
								}
						}
				else if (	scheduler.Scheduler_get_task_state(all_RZs.at(i)) == RUNNING)
						{
							
							if (all_RZs.at(i)->get_implementation_type()==HARD)
							{

								rz_hard_current_static_power += Pstatic;
								rz_hard_current_idle_power += Pidle;
								rz_hard_current_run_power += Prun;
								rz_current_power = Pstatic + Pidle+ Prun;
							}
							else
							{
								//rz_soft_current_static_power += Pstatic;
								//rz_soft_current_idle_power += Pidle;
								
								//rz_soft_current_run_power += Prun;
								//rz_current_power = Prun;
								nb_Running_CPU++;
							}
						}
				else if (	scheduler.Scheduler_get_task_state(all_RZs.at(i)) == CONFIG)
						{
							rz_current_power = Pstatic + P_RECONF; //maybe to refine with Pidle model during reconf?
							if (all_RZs.at(i)->get_implementation_type()==HARD)
							{
								rz_hard_current_static_power += Pstatic;

							}
							else
							{
								//rz_soft_current_static_power += Pstatic;
							}
								
							rz_reconf_current_power += P_RECONF;
						}
							
							
				//total_current_power+=rz_current_power;
				i++;
			}
			
			//now we have browsed all RZ, computer CPU power depending on CPUs USAGE
			//P=Pstatic*Vcore*Ncores+alpha*Vcore*Vcore*Ncores*Freq+beta*Vcore*Vcore*Ncoresrunning*Freq
			
			//max_CPU_V=CPU_VOLTAGES_STEPS[global_operating_point];
			//max_CPU_freqCPU_FREQUENCIES_STEPS[global_operating_point];
			cpu_operating_point=2; //global_operating_point;
			
			rz_soft_current_static_power=PM_CPU_STATIC*max_CPU_V*nb_CPU;
			rz_soft_current_idle_power=PM_CPU_ALPHA*max_CPU_V*max_CPU_V*nb_CPU*max_CPU_freq;
			rz_soft_current_run_power=PM_CPU_BETA*max_CPU_V*max_CPU_V*nb_Running_CPU*max_CPU_freq;
			
			total_current_power=rz_reconf_current_power+rz_hard_current_static_power+rz_hard_current_idle_power+rz_hard_current_run_power+rz_soft_current_static_power+rz_soft_current_idle_power+rz_soft_current_run_power;
			
			
			double hyper_period_step = (sc_time_stamp()-Application::getGlobalOffset())/Application::getGlobalHypeperiod();
			
			if (hyper_period_step == (long)hyper_period_step && new_period==1)
			{
				if (sc_time_stamp()>=Application::getGlobalOffset ())
					current_time=sc_time_stamp()-Application::getGlobalOffset()-((long)hyper_period_step)*Application::getGlobalHypeperiod();
				else
					current_time=sc_time_stamp();
				previous_time=current_time;
			}
			else
			{
				if (sc_time_stamp()>Application::getGlobalOffset ())
					current_time=sc_time_stamp()-Application::getGlobalOffset()-(long)hyper_period_step*Application::getGlobalHypeperiod();
				else
					current_time=sc_time_stamp();
			}
			
			if ((current_time-previous_time).to_seconds()>5) //patch ...
				previous_time=current_time;
			
			if (new_period==1 && previous_time!=current_time)
				new_period=0;
			
			hyper_energy+=total_current_power*(current_time-previous_time).to_seconds();
			
/*			scheduler.Scheduler_get_output_stream() << "Current power : total : " << total_current_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " soft_static: " << rz_soft_current_static_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " soft_idle: " << rz_soft_current_idle_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " soft_run: " << rz_soft_current_run_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " hard_static: " << rz_hard_current_static_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " hard_idle: " << rz_hard_current_idle_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " hard_run: " << rz_hard_current_run_power<< " mW" ;		
			scheduler.Scheduler_get_output_stream() << " reconf: " << rz_reconf_current_power<< " mW" ;
			scheduler.Scheduler_get_output_stream() << " during: " << (current_time-previous_time).to_seconds()*1000 << " ms at "<< current_time.to_seconds()*1000 << " ms at global " << sc_time_stamp().to_seconds()*1000 << " ms" << endl;
*/					
			if (hyper_period_step == (long)hyper_period_step && new_period==0)
			{
/*				scheduler.Scheduler_get_output_stream() << "!!NEW HYPER!!" << endl;
				scheduler.Scheduler_get_output_stream() << "Energy : " << hyper_energy << " mJ" << endl;
*/				previous_hyper_energy=hyper_energy;
				hyper_energy=0;
				
				if (sc_time_stamp()>Application::getGlobalOffset ())
					current_time=sc_time_stamp()-Application::getGlobalOffset()-(long)hyper_period_step*Application::getGlobalHypeperiod();
				else
					current_time=sc_time_stamp();
				new_period=1;
				
				
			}

			
			hyper_instant_energy = hyper_energy;
			//scheduler.Scheduler_get_output_stream() << "scheduler_period : " << current_time-previous_time << endl;
			previous_time=current_time;

		wait(sc_time(5, SC_US));
		}
	}
	
};


#endif
