/* ----------------------------------------------------------------------    
* Copyright (C) 2010-2013 ARM Limited. All rights reserved.    
*    
* $Date:        8. April 2013
* $Revision: 	V1.4.1
*    
* Project: 	    CMSIS DSP Library    
* Title:		arm_sin_cos_f32.c    
*    
* Description:	Sine and Cosine calculation for floating-point values.   
*    
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
*  
* Redistribution and use in source and binary forms, with or without 
* modification, are permitted provided that the following conditions
* are met:
*   - Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*   - Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the 
*     distribution.
*   - Neither the name of ARM LIMITED nor the names of its contributors
*     may be used to endorse or promote products derived from this
*     software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
* COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
* ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.   
* -------------------------------------------------------------------- */

#include "arm_math.h"

/**    
 * @ingroup groupController    
 */

/**    
 * @defgroup SinCos Sine Cosine   
 *    
 * Computes the trigonometric sine and cosine values using a combination of table lookup   
 * and linear interpolation.     
 * There are separate functions for Q31 and floating-point data types.   
 * The input to the floating-point version is in degrees while the   
 * fixed-point Q31 have a scaled input with the range   
 * [-1 0.9999] mapping to [-180 +180] degrees.   
 *
 * The floating point function also allows values that are out of the usual range. When this happens, the function will
 * take extra time to adjust the input value to the range of [-180 180].
 *   
 * The implementation is based on table lookup using 360 values together with linear interpolation.   
 * The steps used are:   
 *  -# Calculation of the nearest integer table index.   
 *  -# Compute the fractional portion (fract) of the input.   
 *  -# Fetch the value corresponding to \c index from sine table to \c y0 and also value from \c index+1 to \c y1.      
 *  -# Sine value is computed as <code> *psinVal = y0 + (fract * (y1 - y0))</code>.    
 *  -# Fetch the value corresponding to \c index from cosine table to \c y0 and also value from \c index+1 to \c y1.      
 *  -# Cosine value is computed as <code> *pcosVal = y0 + (fract * (y1 - y0))</code>.    
 */

 /**    
 * @addtogroup SinCos    
 * @{    
 */

/**    
* \par    
* Cosine Table is generated from:
*
* j=(-180-8:180+90+5)';
* npd = [ j(1:8:end), j(2:8:end), j(3:8:end), j(4:8:end), j(5:8:end), j(6:8:end), j(7:8:end), j(8:8:end)]
* sin(npd/180*pi)
* 
*/

static const float32_t sinTable[453] = {
   0.017452406437283f,
                -0.0f,  -0.017452406437283f,  -0.034899496702501f,  -0.052335956242944f,  -0.069756473744126f,  -0.087155742747658f,  -0.104528463267654f,  -0.121869343405148f,
  -0.139173100960065f,  -0.156434465040231f,  -0.173648177666931f,  -0.190808995376545f,  -0.207911690817759f,  -0.224951054343865f,  -0.241921895599668f,  -0.258819045102521f,
  -0.275637355816999f,  -0.292371704722737f,  -0.309016994374948f,  -0.325568154457157f,  -0.342020143325669f,  -0.358367949545301f,  -0.374606593415912f,  -0.390731128489274f,
  -0.406736643075800f,  -0.422618261740700f,  -0.438371146789078f,  -0.453990499739547f,  -0.469471562785891f,  -0.484809620246337f,  -0.500000000000000,  -0.515038074910054f,
  -0.529919264233205f,  -0.544639035015027f,  -0.559192903470747f,  -0.573576436351046f,  -0.587785252292473f,  -0.601815023152049f,  -0.615661475325658f,  -0.629320391049837f,
  -0.642787609686539f,  -0.656059028990507f,  -0.669130606358858f,  -0.681998360062499f,  -0.694658370458997f,  -0.707106781186548f,  -0.719339800338651f,  -0.731353701619171f,
  -0.743144825477394f,  -0.754709580222772f,  -0.766044443118978f,  -0.777145961456971f,  -0.788010753606722f,  -0.798635510047293f,  -0.809016994374947f,  -0.819152044288992f,
  -0.829037572555042f,  -0.838670567945424f,  -0.848048096156426f,  -0.857167300702112f,  -0.866025403784439f,  -0.874619707139396f,  -0.882947592858927f,  -0.891006524188368f,
  -0.898794046299167f,  -0.906307787036650f,  -0.913545457642601f,  -0.920504853452440f,  -0.927183854566787f,  -0.933580426497202f,  -0.939692620785908f,  -0.945518575599317f,
  -0.951056516295154f,  -0.956304755963036f,  -0.961261695938319f,  -0.965925826289068f,  -0.970295726275996f,  -0.974370064785235f,  -0.978147600733806f,  -0.981627183447664f,
  -0.984807753012208f,  -0.987688340595138f,  -0.990268068741570f,  -0.992546151641322f,  -0.994521895368273f,  -0.996194698091746f,  -0.997564050259824f,  -0.998629534754574f,
  -0.999390827019096f,  -0.999847695156391f,  -1.000000000000000f,  -0.999847695156391f,  -0.999390827019096f,  -0.998629534754574f,  -0.997564050259824f,  -0.996194698091746f,
  -0.994521895368273f,  -0.992546151641322f,  -0.990268068741570f,  -0.987688340595138f,  -0.984807753012208f,  -0.981627183447664f,  -0.978147600733806f,  -0.974370064785235f,
  -0.970295726275996f,  -0.965925826289068f,  -0.961261695938319f,  -0.956304755963035f,  -0.951056516295154f,  -0.945518575599317f,  -0.939692620785908f,  -0.933580426497202f,
  -0.927183854566787f,  -0.920504853452440f,  -0.913545457642601f,  -0.906307787036650f,  -0.898794046299167f,  -0.891006524188368f,  -0.882947592858927f,  -0.874619707139396f,
  -0.866025403784439f,  -0.857167300702112f,  -0.848048096156426f,  -0.838670567945424f,  -0.829037572555042f,  -0.819152044288992f,  -0.809016994374947f,  -0.798635510047293f,
  -0.788010753606722f,  -0.777145961456971f,  -0.766044443118978f,  -0.754709580222772f,  -0.743144825477394f,  -0.731353701619170f,  -0.719339800338651f,  -0.707106781186547f,
  -0.694658370458997f,  -0.681998360062498f,  -0.669130606358858f,  -0.656059028990507f,  -0.642787609686539f,  -0.629320391049838f,  -0.615661475325658f,  -0.601815023152048f,
  -0.587785252292473f,  -0.573576436351046f,  -0.559192903470747f,  -0.544639035015027f,  -0.529919264233205f,  -0.515038074910054f,  -0.500000000000000f,  -0.484809620246337f,
  -0.469471562785891f,  -0.453990499739547f,  -0.438371146789077f,  -0.422618261740699f,  -0.406736643075800f,  -0.390731128489274f,  -0.374606593415912f,  -0.358367949545300f,
  -0.342020143325669f,  -0.325568154457157f,  -0.309016994374947f,  -0.292371704722737f,  -0.275637355816999f,  -0.258819045102521f,  -0.241921895599668f,  -0.224951054343865f,
  -0.207911690817759f,  -0.190808995376545f,  -0.173648177666930f,  -0.156434465040231f,  -0.139173100960065f,  -0.121869343405147f,  -0.104528463267653f,  -0.087155742747658f,
  -0.069756473744125f,  -0.052335956242944f,  -0.034899496702501f,  -0.017452406437284f,                 0.0f,   0.017452406437284f,   0.034899496702501f,   0.052335956242944f,
   0.069756473744125f,   0.087155742747658f,   0.104528463267653f,   0.121869343405147f,   0.139173100960065f,   0.156434465040231f,   0.173648177666930f,   0.190808995376545f,
   0.207911690817759f,   0.224951054343865f,   0.241921895599668f,   0.258819045102521f,   0.275637355816999f,   0.292371704722737f,   0.309016994374947f,   0.325568154457157f,
   0.342020143325669f,   0.358367949545300f,   0.374606593415912f,   0.390731128489274f,   0.406736643075800f,   0.422618261740699f,   0.438371146789077f,   0.453990499739547f,
   0.469471562785891f,   0.484809620246337f,   0.500000000000000f,   0.515038074910054f,   0.529919264233205f,   0.544639035015027f,   0.559192903470747f,   0.573576436351046f,
   0.587785252292473f,   0.601815023152048f,   0.615661475325658f,   0.629320391049838f,   0.642787609686539f,   0.656059028990507f,   0.669130606358858f,   0.681998360062498f,
   0.694658370458997f,   0.707106781186547f,   0.719339800338651f,   0.731353701619170f,   0.743144825477394f,   0.754709580222772f,   0.766044443118978f,   0.777145961456971f,
   0.788010753606722f,   0.798635510047293f,   0.809016994374947f,   0.819152044288992f,   0.829037572555042f,   0.838670567945424f,   0.848048096156426f,   0.857167300702112f,
   0.866025403784439f,   0.874619707139396f,   0.882947592858927f,   0.891006524188368f,   0.898794046299167f,   0.906307787036650f,   0.913545457642601f,   0.920504853452440f,
   0.927183854566787f,   0.933580426497202f,   0.939692620785908f,   0.945518575599317f,   0.951056516295154f,   0.956304755963035f,   0.961261695938319f,   0.965925826289068f,
   0.970295726275996f,   0.974370064785235f,   0.978147600733806f,   0.981627183447664f,   0.984807753012208f,   0.987688340595138f,   0.990268068741570f,   0.992546151641322f,
   0.994521895368273f,   0.996194698091746f,   0.997564050259824f,   0.998629534754574f,   0.999390827019096f,   0.999847695156391f,   1.000000000000000f,   0.999847695156391f,
   0.999390827019096f,   0.998629534754574f,   0.997564050259824f,   0.996194698091746f,   0.994521895368273f,   0.992546151641322f,   0.990268068741570f,   0.987688340595138f,
   0.984807753012208f,   0.981627183447664f,   0.978147600733806f,   0.974370064785235f,   0.970295726275996f,   0.965925826289068f,   0.961261695938319f,   0.956304755963036f,
   0.951056516295154f,   0.945518575599317f,   0.939692620785908f,   0.933580426497202f,   0.927183854566787f,   0.920504853452440f,   0.913545457642601f,   0.906307787036650f,
   0.898794046299167f,   0.891006524188368f,   0.882947592858927f,   0.874619707139396f,   0.866025403784439f,   0.857167300702112f,   0.848048096156426f,   0.838670567945424f,
   0.829037572555042f,   0.819152044288992f,   0.809016994374947f,   0.798635510047293f,   0.788010753606722f,   0.777145961456971f,   0.766044443118978f,   0.754709580222772f,
   0.743144825477394f,   0.731353701619171f,   0.719339800338651f,   0.707106781186548f,   0.694658370458997f,   0.681998360062499f,   0.669130606358858f,   0.656059028990507f,
   0.642787609686539f,   0.629320391049837f,   0.615661475325658f,   0.601815023152049f,   0.587785252292473f,   0.573576436351046f,   0.559192903470747f,   0.544639035015027f,
   0.529919264233205f,   0.515038074910054f,   0.500000000000000f,   0.484809620246337f,   0.469471562785891f,   0.453990499739547f,   0.438371146789078f,   0.422618261740700f,
   0.406736643075800f,   0.390731128489274f,   0.374606593415912f,   0.358367949545301f,   0.342020143325669f,   0.325568154457157f,   0.309016994374948f,   0.292371704722737f,
   0.275637355816999f,   0.258819045102521f,   0.241921895599668f,   0.224951054343865f,   0.207911690817759f,   0.190808995376545f,   0.173648177666931f,   0.156434465040231f,
   0.139173100960065f,   0.121869343405148f,   0.104528463267654f,   0.087155742747658f,   0.069756473744126f,   0.052335956242944f,   0.034899496702501f,   0.017452406437283f,
   0.000000000000000f,  -0.017452406437283f,  -0.034899496702501f,  -0.052335956242944f,  -0.069756473744125f,  -0.087155742747658f,  -0.104528463267654f,  -0.121869343405148f,
  -0.139173100960066f,  -0.156434465040231f,  -0.173648177666930f,  -0.190808995376545f,  -0.207911690817759f,  -0.224951054343865f,  -0.241921895599668f,  -0.258819045102520f,
  -0.275637355816999f,  -0.292371704722737f,  -0.309016994374948f,  -0.325568154457157f,  -0.342020143325669f,  -0.358367949545300f,  -0.374606593415912f,  -0.390731128489274f,
  -0.406736643075800f,  -0.422618261740699f,  -0.438371146789077f,  -0.453990499739546f,  -0.469471562785890f,  -0.484809620246337f,  -0.500000000000000f,  -0.515038074910054f,
  -0.529919264233205f,  -0.544639035015027f,  -0.559192903470747f,  -0.573576436351046f,  -0.587785252292473f,  -0.601815023152048f,  -0.615661475325658f,  -0.629320391049837f,
  -0.642787609686539f,  -0.656059028990507f,  -0.669130606358858f,  -0.681998360062498f,  -0.694658370458997f,  -0.707106781186547f,  -0.719339800338651f,  -0.731353701619170f,
  -0.743144825477394f,  -0.754709580222772f,  -0.766044443118978f,  -0.777145961456971f,  -0.788010753606722f,  -0.798635510047293f,  -0.809016994374947f,  -0.819152044288992f,
  -0.829037572555041f,  -0.838670567945424f,  -0.848048096156426f,  -0.857167300702112f,  -0.866025403784438f,  -0.874619707139396f,  -0.882947592858927f,  -0.891006524188368f,
  -0.898794046299167f,  -0.906307787036650f,  -0.913545457642601f,  -0.920504853452440f,  -0.927183854566787f,  -0.933580426497202f,  -0.939692620785908f,  -0.945518575599317f,
  -0.951056516295154f,  -0.956304755963035f,  -0.961261695938319f,  -0.965925826289068f,  -0.970295726275996f,  -0.974370064785235f,  -0.978147600733806f,  -0.981627183447664f,
  -0.984807753012208f,  -0.987688340595138f,  -0.990268068741570f,  -0.992546151641322f,  -0.994521895368273f,  -0.996194698091746f,  -0.997564050259824f,  -0.998629534754574f,
  -0.999390827019096f,  -0.999847695156391f,  -1.000000000000000f,  -0.999847695156391f
  };

/**    
 * @brief  Floating-point sin_cos function.   
 * @param[in]  theta    input value in degrees    
 * @param[out] *pSinVal points to the processed sine output.    
 * @param[out] *pCosVal points to the processed cos output.    
 * @return none.   
 */

void arm_sin_cos_f32(
  float32_t theta,
  float32_t * pSinVal,
  float32_t * pCosVal)
{
  int32_t ic;                                    /* Index for cos */
  float32_t y0, y1;                              /* nearest output values */
  float32_t y2, y3;
  float32_t fract;                               /* fractional part of input */

  theta += 181.0f;

    /* Checking min and max index of table */
  if(theta < 0.0f)
       theta = fmod(theta,-360.0f) + 360.0f;
  else if(theta > 361.0f)
       theta = fmod(theta,360.0f);

   /* Calculation of fractional part */
  ic = (uint32_t) (theta);

  /* index calculation for reading nearest output values */
  fract = theta - (float32_t) ic;

  /* reading nearest sine output values */
  y0 = sinTable[ic];
  y1 = sinTable[ic + 1u];

  /* reading nearest cosine output values */
  y2 = sinTable[ic + 90u];
  y3 = sinTable[ic + 91u];

  y1 -= y0;
  y3 -= y2;

  y1 *= fract;
  y3 *= fract;

  /* Calculation of sine value */
  *pSinVal = y0 + y1;

  /* Calculation of cosine value */
  *pCosVal = y2 + y3;

}
/**    
 * @} end of SinCos group    
 */
